# -*- coding: utf-8 -*-
import re
import mpfs.engine.process

from mpfs.common.util.filetypes import getGroupByName, MediaType
from mpfs.common.util.mobile_client import MobileClientVersion
from mpfs.common.util.user_agent_parser import UserAgentParser
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.core.photoslice.albums.static import IOS_DIMENSIONS
from mpfs.metastorage.postgres.schema import PhotosliceAlbumType

FEATURE_TOGGLES_PHOTOSLICE_ALBUMS_IOS_VERSION = MobileClientVersion.build_from_version(
    settings.feature_toggles['photoslice_albums_ios_version'])
FEATURE_TOGGLES_PHOTOSLICE_ALBUMS_ANDR_VERSION = MobileClientVersion.build_from_version(
    settings.feature_toggles['photoslice_albums_andr_version'])


def resolve_photoslice_album_type(uid, mimetype, photoslice_album_type_data, link=None, autouploaded=False):
    # не делаем монговую реализацию. Можно убрать после того, как перевезем всех на ПГ
    if not mpfs.engine.process.usrctl().is_user_in_postgres(uid):
        return

    if photoslice_album_type_data is None:
        return None

    upload_address = Address(photoslice_album_type_data.upload_raw_path)
    return _resolve_photoslice_album_type_by_media_type(
        upload_address, getGroupByName(upload_address.path, mtype=mimetype),
        photoslice_album_type_data, link=link, autouploaded=autouploaded
    )


def update_file_data_with_photoslice_album_type(uid, file_data, photoslice_album_type_data):
    """
    :param file_data: ресурс в виде словаря
    """
    # не делаем монговую реализацию. Можно убрать после того, как перевезем всех на ПГ
    if not mpfs.engine.process.usrctl().is_user_in_postgres(uid):
        return

    if photoslice_album_type_data is None:
        return

    user_agent = UserAgentParser.parse(photoslice_album_type_data.user_agent)
    if not _is_old_ios(user_agent):
        return

    width, height = file_data['meta'].get('width'), file_data['meta'].get('height')
    if not width or not height:
        return

    upload_address = Address(photoslice_album_type_data.upload_raw_path)
    if (upload_address.ext.lower() == 'png'
            and ((width, height) in IOS_DIMENSIONS or (height, width) in IOS_DIMENSIONS)):
        file_data['meta']['photoslice_album_type'] = PhotosliceAlbumType.SCREENSHOTS


def _resolve_photoslice_album_type_by_media_type(
        upload_address, media_type, photoslice_album_type_data, link=None, autouploaded=False):
    user_agent = UserAgentParser.parse(photoslice_album_type_data.user_agent)
    if not user_agent.is_mobile_client():
        return None

    if _is_old_android(user_agent):
        return _resolve_photoslice_album_type_for_android_old(
            media_type,
            photoslice_album_type_data.is_screenshot,
            photoslice_album_type_data.client_etime,
            upload_address,
        )
    elif _is_new_android(user_agent):
        return _resolve_photoslice_album_type_for_android(
            media_type,
            photoslice_album_type_data.device_original_path,
            photoslice_album_type_data.client_etime,
            autouploaded=autouploaded,
        )
    elif _is_old_ios(user_agent):
        return _resolve_photoslice_album_type_for_ios_old(
            media_type,
            photoslice_album_type_data.client_etime,
            upload_address,
            link=link,
        )
    elif _is_new_ios(user_agent):
        return _resolve_photoslice_album_type_for_ios(
            media_type,
            photoslice_album_type_data.device_collections,
            photoslice_album_type_data.is_screenshot,
            photoslice_album_type_data.client_etime
        )
    else:
        raise NotImplementedError("Unsupported os %s" % user_agent.get_os_family())


def _is_android_camera_path(folder_path):
    """Проверяет является ли путь camera-папкой у Android.

    camera-пути:
      /DCIM/Selfie/
      /DCIM/Camera/
      /DCIM/\d{3}\w+/
      /DCIM/OpenCamera/
    """
    if folder_path is None or not folder_path.startswith('/'):
        return False

    if re.match(r'.*/DCIM/(camera|opencamera|selfie|\d{3}\w+)/', folder_path, re.IGNORECASE):
        return True

    return False


def _resolve_photoslice_album_type_for_android(media_type, device_original_path, device_etime, autouploaded=False):
    if media_type == MediaType.IMAGE:
        if (device_original_path is not None and
                    'screenshot' in device_original_path.lower()):
            return PhotosliceAlbumType.SCREENSHOTS
        if (device_etime or
                (autouploaded and
                 _is_android_camera_path(device_original_path))):
            return PhotosliceAlbumType.CAMERA
    elif media_type == MediaType.VIDEO:
        if autouploaded and _is_android_camera_path(device_original_path):
            return PhotosliceAlbumType.CAMERA
    return None


def _resolve_photoslice_album_type_for_android_old(media_type, is_screenshot, device_etime, upload_address):
    if media_type == MediaType.IMAGE:
        if device_etime:
            return PhotosliceAlbumType.CAMERA
        if is_screenshot:
            return PhotosliceAlbumType.SCREENSHOTS
    elif media_type == MediaType.VIDEO:
        if upload_address.storage_name == 'photostream':
            return PhotosliceAlbumType.CAMERA
    return None


def _resolve_photoslice_album_type_for_ios(media_type, device_collections, is_screenshot, device_etime):
    if media_type == MediaType.IMAGE:
        if is_screenshot:
            return PhotosliceAlbumType.SCREENSHOTS
        if (device_etime and
                (device_collections is None or
                 not len(device_collections))):
            return PhotosliceAlbumType.CAMERA
    elif media_type == MediaType.VIDEO:
        if device_collections is None or not len(device_collections):
            return PhotosliceAlbumType.CAMERA
    return None


def _resolve_photoslice_album_type_for_ios_old(media_type, device_etime, upload_address, link=None):
    if media_type == MediaType.IMAGE:
        if device_etime:
            return PhotosliceAlbumType.CAMERA
        if link and link.photoslice_album_type and link.photoslice_album_type == PhotosliceAlbumType.SCREENSHOTS:
            return PhotosliceAlbumType.SCREENSHOTS
    elif media_type == MediaType.VIDEO:
        if upload_address.storage_name == 'photostream':
            return PhotosliceAlbumType.CAMERA
    return None


def _is_new_android(user_agent):
    return user_agent.is_android() and user_agent.get_version() >= FEATURE_TOGGLES_PHOTOSLICE_ALBUMS_ANDR_VERSION


def _is_new_ios(user_agent):
    return user_agent.is_ios() and user_agent.get_version() >= FEATURE_TOGGLES_PHOTOSLICE_ALBUMS_IOS_VERSION


def _is_old_android(user_agent):
    return user_agent.is_android() and user_agent.get_version() < FEATURE_TOGGLES_PHOTOSLICE_ALBUMS_ANDR_VERSION


def _is_old_ios(user_agent):
    return user_agent.is_ios() and user_agent.get_version() < FEATURE_TOGGLES_PHOTOSLICE_ALBUMS_IOS_VERSION
