# -*- coding: utf-8 -*-
import datetime

import click

from mpfs.common.static.tags.billing import (
    PRIMARY_2018_DISCOUNT_30,
    PRIMARY_2018_DISCOUNT_20,
    PRIMARY_2018_DISCOUNT_10,
)


help_doc = u"""
Создать шаблон скидки
Примеры:

* Создать шаблон скидки на 20%, заканчивающийся 2018-10-09 13:40
mpfs-cli promo_codes create_discount_template 20 https://st.yandex-team.ru/CHEMODAN-0000 --end_datetime 2018-10-09 13:40
"""


@click.command('create_discount_template', help=help_doc)
@click.argument('discount_line', type=click.Choice(['10', '20', '30']))
@click.argument('comment', type=click.STRING)
@click.option('--end_datetime', type=click.STRING, help=u'Время окончания скидки')
@click.option('--period_days', type=click.INT, help=u'Сколько дней дается пользователю на использование скидки со времени создания')
@click.option('--disposable', is_flag=True)
def create_discount_template(discount_line, comment, end_datetime, period_days, disposable):
    import mpfs.engine.process
    mpfs.engine.process.setup_admin_script()

    from mpfs.core.promo_codes.logic.discount import DiscountTemplate

    discount_line_map = {
        '10': PRIMARY_2018_DISCOUNT_10,
        '20': PRIMARY_2018_DISCOUNT_20,
        '30': PRIMARY_2018_DISCOUNT_30,
    }

    if (end_datetime and period_days) or (end_datetime is None and period_days is None):
        raise ValueError('Either end_datetime or period_days should be specified')

    end_datetime_obj = None
    period_timedelta_obj = None
    if end_datetime:
        end_datetime_obj = datetime.datetime.strptime(end_datetime, '%Y-%m-%d %H:%M:%S')
    else:
        period_timedelta_obj = datetime.timedelta(days=period_days)

    discount_template = DiscountTemplate.create(discount_line_map[discount_line], comment, disposable,
                                                period_timedelta=period_timedelta_obj, end_datetime=end_datetime_obj)
    discount_template.save()
    click.echo(discount_template.id)
