# -*- coding: utf-8 -*-
from mpfs.dao.base import BaseDAOItem, PostgresBaseDAOImplementation, MongoBaseDAOImplementation, BaseDAO
from mpfs.dao.fields import (
    BoolField,
    DateTimeDeltaField,
    DateTimeField,
    DiscountProvidableLinesTypeField,
    StringField,
    UidField,
    UuidField,
)
from mpfs.metastorage.postgres.queries import SQL_FIND_DISCOUNT_TEMPLATE, SQL_UPSERT_DISCOUNT_TEMPLATE, \
    SQL_FIND_DISCOUNTS_ARCHIVE, SQL_UPSERT_DISCOUNTS_ARCHIVE, SQL_DELETE_DISCOUNTS_ARCHIVE
from mpfs.metastorage.postgres.schema import discount_templates, discounts_archive


class DiscountTemplateDAOItem(BaseDAOItem):
    mongo_collection_name = 'discount_templates'
    postgres_table_obj = discount_templates
    is_sharded = False

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'

    id = UuidField(mongo_path='_id', pg_path=discount_templates.c.id)
    description = StringField(mongo_path='description', pg_path=discount_templates.c.description)
    provided_line = DiscountProvidableLinesTypeField(mongo_path='provided_line', pg_path=discount_templates.c.provided_line)
    disposable = BoolField(mongo_path='disposable', pg_path=discount_templates.c.disposable)
    creation_datetime = DateTimeField(mongo_path='creation_datetime', pg_path=discount_templates.c.creation_datetime)
    period_timedelta = DateTimeDeltaField(mongo_path='period_timedelta', pg_path=discount_templates.c.period_timedelta, default_value=None)
    end_datetime = DateTimeField(mongo_path='end_datetime', pg_path=discount_templates.c.end_datetime, default_value=None)


class DiscountTemplateDAO(BaseDAO):
    dao_item_cls = DiscountTemplateDAOItem

    def __init__(self):
        super(DiscountTemplateDAO, self).__init__()
        self._mongo_impl = MongoDiscountTemplateDAOImplementation(self.dao_item_cls)
        self._pg_impl = PostgresDiscountTemplateDAOImplementation(self.dao_item_cls)

    def save(self, item):
        return self._get_impl(None).save(item)

    def find_discount_template(self, id_):
        return self._get_impl(None).find_discount_template(id_)


class MongoDiscountTemplateDAOImplementation(MongoBaseDAOImplementation):
    def find_discount_template(self, id_):
        spec = {'_id': id_}
        result = super(MongoDiscountTemplateDAOImplementation, self).find_one(spec)
        if result is None:
            return None
        return DiscountTemplateDAOItem.create_from_mongo_dict(result)

    def save(self, item):
        self.update({'_id': item.id}, item.get_mongo_representation(), upsert=True)


class PostgresDiscountTemplateDAOImplementation(PostgresBaseDAOImplementation):
    def find_discount_template(self, id_):
        session = self._get_session()
        result = session.execute(SQL_FIND_DISCOUNT_TEMPLATE, {'id': id_}).fetchone()
        if result:
            return DiscountTemplateDAOItem.create_from_pg_data(result)
        return None

    def save(self, item):
        session = self._get_session()
        session.execute(SQL_UPSERT_DISCOUNT_TEMPLATE, item.as_raw_pg_dict())


class DiscountArchiveDAOItem(BaseDAOItem):
    mongo_collection_name = 'discounts_archive'
    postgres_table_obj = discounts_archive
    is_sharded = False

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'

    id = UuidField(mongo_path='_id', pg_path=discounts_archive.c.id)
    uid = UidField(mongo_path='uid', pg_path=discounts_archive.c.uid)
    discount_template_id = UuidField(mongo_path='discount_template_id', pg_path=discounts_archive.c.discount_template_id)


class DiscountArchiveDAO(BaseDAO):
    dao_item_cls = DiscountArchiveDAOItem

    def __init__(self):
        super(DiscountArchiveDAO, self).__init__()
        self._mongo_impl = MongoDiscountArchiveDAOImplementation(self.dao_item_cls)
        self._pg_impl = PostgresDiscountArchiveDAOImplementation(self.dao_item_cls)

    def save(self, item):
        return self._get_impl(None).save(item)

    def delete(self, item):
        return self._get_impl(None).delete(item)

    def find_used_discount(self, uid, discount_template_id):
        return self._get_impl(None).find_used_discount(uid, discount_template_id)


class MongoDiscountArchiveDAOImplementation(MongoBaseDAOImplementation):
    def save(self, item):
        self.update({'_id': item.id}, item.get_mongo_representation(), upsert=True)

    def delete(self, item):
        self.remove({'_id': item.id})

    def find_used_discount(self, uid, discount_template_id):
        spec = {
            'uid': uid,
            'discount_template_id': discount_template_id,
        }
        result = super(MongoDiscountArchiveDAOImplementation, self).find_one(spec)
        if result is None:
            return None
        return DiscountTemplateDAOItem.create_from_mongo_dict(result)


class PostgresDiscountArchiveDAOImplementation(PostgresBaseDAOImplementation):
    def save(self, item):
        session = self._get_session()
        session.execute(SQL_UPSERT_DISCOUNTS_ARCHIVE, item.as_raw_pg_dict())

    def delete(self, item):
        session = self._get_session()
        session.execute(SQL_DELETE_DISCOUNTS_ARCHIVE, {'id': item.id})

    def find_used_discount(self, uid, discount_template_id):
        session = self._get_session()
        result = session.execute(SQL_FIND_DISCOUNTS_ARCHIVE,
                                 {'uid': uid, 'discount_template_id': discount_template_id}).fetchone()
        if result:
            return DiscountArchiveDAOItem.create_from_pg_data(result)
        return None
