# -*- coding: utf-8 -*-
import random
import time

from pymongo.errors import DuplicateKeyError

from mpfs.common.errors.billing import BillingProductNotFound, BillingInvalidMarketLineProduct
from mpfs.common.static.tags.billing import PROMO_CODE
from mpfs.config import settings
from mpfs.core.billing import Catalog
from mpfs.core.billing import Market
from mpfs.core.billing import Product
from mpfs.core.promo_codes.dao.promo_codes import PromoCodeDAO
from mpfs.core.promo_codes.logic.discount import DiscountTemplate
from mpfs.core.promo_codes.logic.errors import DiscountTemplateNotFound

AVAILABLE_CHARACTERS = '23456789ABCDEFGHJKMNPQRSTUVWXYZ'
CODE_LENGTH = 10


class PromoCodeGeneratorFactory(object):
    @classmethod
    def get_generator(cls, **kwargs):
        if kwargs.get('pid'):
            return ServicePromoCodeGenerator(**kwargs)
        elif kwargs.get('discount_template_id'):
            return DiscountPromoCodeGenerator(**kwargs)
        raise NotImplemented('Unexpected promo code generator type')


class PromoCodeGeneratorBase(object):
    _generator = None
    begin_timestamp = None
    end_timestamp = None
    codes_num = None
    activation_num = None

    def __init__(self, **kwargs):
        """
        :param kwargs: begin_timestamp, end_timestamp
        """
        if kwargs['begin_timestamp'] > kwargs['end_timestamp']:
            raise ValueError('start_timestamp > end_timestamp')
        if kwargs['end_timestamp'] < int(time.time()):
            raise ValueError('end_timestamp < current time')

        self._generator = random.SystemRandom()
        self.begin_timestamp = kwargs['begin_timestamp']
        self.end_timestamp = kwargs['end_timestamp']
        self.activation_num = kwargs.get('activation_num', 1)

    def generate_new_code(self):
        new_code = ''
        for i in range(CODE_LENGTH):
            current_char = AVAILABLE_CHARACTERS[self._generator.randint(0, len(AVAILABLE_CHARACTERS) - 1)]
            new_code += current_char
        return new_code

    def generate_and_insert_promo_code(self, code_id=None):
        dao = PromoCodeDAO()
        promo_code_doc = self._build_data_for_insert()

        if code_id:
            code_id = code_id.upper()
            promo_code_doc['_id'] = code_id
            dao.insert(promo_code_doc)
            return code_id

        while True:
            new_code = self.generate_new_code()
            promo_code_doc['_id'] = new_code
            try:
                dao.insert(promo_code_doc)
                return new_code
            except DuplicateKeyError:
                pass

    def _build_data_for_insert(self):
        doc = {
            'begin_datetime': self.begin_timestamp,
            'end_datetime': self.end_timestamp,
            'count': self.activation_num,
        }
        return doc


class ServicePromoCodeGenerator(PromoCodeGeneratorBase):
    pid = None

    def __init__(self, **kwargs):
        """
        :param kwargs: begin_timestamp, end_timestamp, pid
        """
        super(ServicePromoCodeGenerator, self).__init__(**kwargs)
        try:
            Product(kwargs['pid'])
        except BillingProductNotFound:
            raise ValueError('pid %s doesn\'t exist' % kwargs['pid'])

        if kwargs['pid'] not in Catalog().get_pids(Market(settings.billing['default_market']), PROMO_CODE):
            raise BillingInvalidMarketLineProduct(kwargs['pid'])
        self.pid = kwargs['pid']

    def _build_data_for_insert(self):
        doc = super(ServicePromoCodeGenerator, self)._build_data_for_insert()
        doc['pid'] = self.pid
        return doc


class DiscountPromoCodeGenerator(PromoCodeGeneratorBase):
    discount_template = None

    def __init__(self, **kwargs):
        """
        :param kwargs: begin_timestamp, end_timestamp, discount_template_id
        """
        super(DiscountPromoCodeGenerator, self).__init__(**kwargs)
        try:
            discount_template = DiscountTemplate.get_discount_template(kwargs['discount_template_id'])
        except DiscountTemplateNotFound:
            raise ValueError('discount_template with id %s not found' % kwargs['discount_template_id'])

        self.discount_template = discount_template

    def _build_data_for_insert(self):
        doc = super(DiscountPromoCodeGenerator, self)._build_data_for_insert()
        doc['discount_template_id'] = self.discount_template.id
        return doc
