# -*- coding: utf-8 -*-
import datetime
import time
import uuid

from mpfs.common.static.tags import experiment_names
from mpfs.common.static.tags.billing import (
    PRIMARY_2018_DISCOUNT_30,
    PRIMARY_2018_DISCOUNT_20,
    PRIMARY_2018_DISCOUNT_10,
    PRIMARY_2019_DISCOUNT_30,
    PRIMARY_2019_DISCOUNT_20,
    PRIMARY_2019_DISCOUNT_10,
    PRIMARY_2019_V2_DISCOUNT_10,
    PRIMARY_2019_V2_DISCOUNT_20,
    PRIMARY_2019_V2_DISCOUNT_30,
    PRIMARY_2019_V3_DISCOUNT_10,
    PRIMARY_2019_V3_DISCOUNT_20,
    PRIMARY_2019_V3_DISCOUNT_30,
    PRIMARY_2019_V4_DISCOUNT_10,
    PRIMARY_2019_V4_DISCOUNT_20,
    PRIMARY_2019_V4_DISCOUNT_30,
    THIRD_TARIFF_2019_DISCOUNT_10,
    THIRD_TARIFF_2019_DISCOUNT_20,
    THIRD_TARIFF_2019_DISCOUNT_30
)
from mpfs.common.util.experiments.logic import experiment_gateway_wrapper, experiment_manager
from mpfs.core.promo_codes.dao.discounts import (
    DiscountTemplateDAO,
    DiscountTemplateDAOItem,
    DiscountArchiveDAO,
    DiscountArchiveDAOItem,
)
from mpfs.core.promo_codes.logic.errors import DiscountTemplateNotFound
from mpfs.metastorage.postgres.schema import DiscountProvidableLinesType


class DiscountTemplate(object):
    dao = DiscountTemplateDAO()
    dao_item_class = DiscountTemplateDAOItem

    def __init__(self, dao_item):
        if not isinstance(dao_item, self.dao_item_class):
            raise TypeError('`%s` expected. Got: %r' % (self.dao_item_class.__name__, dao_item))
        self.dao_item = dao_item

    @classmethod
    def create(cls, provided_line, description, disposable=True, period_timedelta=None, end_datetime=None):
        dao_item = cls.dao_item_class()
        dao_item.id = uuid.uuid4().hex
        dao_item.description = description
        dao_item.provided_line = DiscountProvidableLinesType(provided_line)
        dao_item.disposable = disposable
        dao_item.creation_datetime = datetime.datetime.now()
        dao_item.period_timedelta = period_timedelta
        dao_item.end_datetime = end_datetime
        return DiscountTemplate(dao_item)

    def save(self):
        self.dao.save(self.dao_item)

    @classmethod
    def get_discount_template(cls, id_):
        dao_item = cls.dao.find_discount_template(id_)
        if dao_item is None:
            raise DiscountTemplateNotFound()
        return cls(dao_item)

    @property
    def id(self):
        return self.dao_item.id

    @property
    def provided_line(self):
        LINE_OLD_TO_NEW = {PRIMARY_2018_DISCOUNT_10: PRIMARY_2019_V4_DISCOUNT_10,
                           PRIMARY_2018_DISCOUNT_20: PRIMARY_2019_V4_DISCOUNT_20,
                           PRIMARY_2018_DISCOUNT_30: PRIMARY_2019_V4_DISCOUNT_30}
        return LINE_OLD_TO_NEW.get(self.dao_item.provided_line.value, self.dao_item.provided_line.value)

    @property
    def provided_lines(self):
        LINE_OLD_TO_NEW = {PRIMARY_2018_DISCOUNT_10: [PRIMARY_2019_V4_DISCOUNT_10, THIRD_TARIFF_2019_DISCOUNT_10],
                           PRIMARY_2018_DISCOUNT_20: [PRIMARY_2019_V4_DISCOUNT_20, THIRD_TARIFF_2019_DISCOUNT_20],
                           PRIMARY_2018_DISCOUNT_30: [PRIMARY_2019_V4_DISCOUNT_30, THIRD_TARIFF_2019_DISCOUNT_30]}
        return LINE_OLD_TO_NEW.get(self.dao_item.provided_line.value, self.dao_item.provided_line.value)

    @property
    def disposable(self):
        """Является ли скидки одноразовой"""
        return self.dao_item.disposable

    @property
    def period_timedelta(self):
        """
        Сколько времени дается пользователю на использование скидки со времени создания.

        Приоритетнее, чем  end_datetime
        """
        return self.dao_item.period_timedelta

    @property
    def end_datetime(self):
        """До какого времени пользователь может использовать скидку"""
        return self.dao_item.end_datetime

    def to_dict(self):
        return self.dao_item.get_mongo_representation()


class Discount(object):
    """
    Скидка пользователя.

    Создается по DiscountTemplate. Информация, связанная с пользователем по скидке, хранится в disk_info, все остальное
    восстанавливается по DiscountTemplate. Если сравнивать с услугами биллинга, то DiscountTemplate == Product, а
    Discount == Service
    """
    uid = None
    creation_datetime = None
    end_datetime = None
    """Время окончания скидки, после этого времени скидка перенесется в архив"""
    disposable = False
    provided_line = None
    discount_template_id = None

    @classmethod
    def create(cls, uid, ctime, discount_template):
        discount = cls()
        discount.uid = uid
        discount.creation_datetime = datetime.datetime.fromtimestamp(ctime)
        if discount_template.period_timedelta is not None:
            discount.end_datetime = discount.creation_datetime + discount_template.period_timedelta
        elif discount_template.end_datetime is not None:
            discount.end_datetime = discount_template.end_datetime
        else:
            discount.end_datetime = None
        discount.disposable = discount_template.disposable
        discount.provided_line = discount_template.provided_line
        discount.discount_template_id = discount_template.id
        return discount

    @property
    def line_order(self):
        return [PRIMARY_2019_V4_DISCOUNT_30, PRIMARY_2019_V4_DISCOUNT_20, PRIMARY_2019_V4_DISCOUNT_10]

    def get_database_dict(self):
        """Возвращает словарь в том виде, в котором скидка пользователя представлена в disk_info"""
        db_dict = {
            '_id': self.discount_template_id,
            'ctime': int(time.mktime(self.creation_datetime.timetuple()))
        }
        return db_dict

    def __lt__(self, other):
        if self.provided_line != other.provided_line:
            return self.line_order.index(self.provided_line) < other.line_order.index(other.provided_line)
        if self.disposable != other.disposable:
            return not self.disposable and other.disposable
        if self.end_datetime is None and other.end_datetime is None:
            return False
        if self.end_datetime is None:
            return True
        if other.end_datetime is None:
            return False
        return self.end_datetime < other.end_datetime


class DiscountArchive(object):
    dao = DiscountArchiveDAO()
    dao_item_class = DiscountArchiveDAOItem

    def __init__(self, dao_item):
        if not isinstance(dao_item, self.dao_item_class):
            raise TypeError('`%s` expected. Got: %r' % (self.dao_item_class.__name__, dao_item))
        self.dao_item = dao_item

    @classmethod
    def create(cls, uid, discount):
        dao_item = cls.dao_item_class()
        dao_item.id = uuid.uuid4().hex
        dao_item.uid = uid
        dao_item.discount_template_id = discount.discount_template_id
        return DiscountArchive(dao_item)

    def save(self):
        self.dao.save(self.dao_item)

    def delete(self):
        self.dao.delete(self.dao_item)

    @classmethod
    def fetch(cls, uid, discount_template_id):
        item = cls.dao.find_used_discount(uid, discount_template_id)
        if item is None:
            return

        dao_item = cls.dao_item_class()
        dao_item.id = item.id
        dao_item.uid = uid
        dao_item.discount_template_id = discount_template_id
        return DiscountArchive(dao_item)

    @classmethod
    def exists(cls, uid, discount_template_id):
        return cls.dao.find_used_discount(uid, discount_template_id) is not None
