# -*- coding: utf-8 -*-

import datetime
import time

import mpfs.engine.process
from mpfs.core.billing.product import get_default_products_line

from mpfs.core.metastorage.control import disk_info
from mpfs.core.promo_codes.logic.discount import DiscountTemplate, DiscountArchive, Discount
from mpfs.core.promo_codes.logic.errors import AttemptToActivateDiscountRepeatedly, AttemptToActivateArchivedDiscount


default_log = mpfs.engine.process.get_default_log()


class DiscountManager(object):
    key = '/active_discounts'

    @classmethod
    def add_discount_to_user(cls, uid, discount_template_id, check_archive=True):
        discount_template = DiscountTemplate.get_discount_template(discount_template_id)
        result_discounts = cls._get_all_user_discounts(uid)
        try:
            for discount in result_discounts:
                if discount.discount_template_id == discount_template_id:
                    raise AttemptToActivateDiscountRepeatedly(
                        "User used such discount_template uid=%s discount_template_id=%s" % (uid, discount_template_id))
            if check_archive and DiscountArchive.exists(uid, discount_template_id):
                raise AttemptToActivateArchivedDiscount(
                    "User used such discount_template uid=%s discount_template_id=%s" % (uid, discount_template_id))
            discount = Discount.create(uid, int(time.time()), discount_template)
            result_discounts.append(discount)
        finally:
            cls._archive_outdated_and_save_discounts(uid, result_discounts, force_update=True)
        return discount

    @classmethod
    def get_cheapest_available_discount(cls, uid):
        """
        :return: Discount
        """
        all_user_discounts = cls._get_all_user_discounts(uid)
        all_user_discounts = cls._archive_outdated_and_save_discounts(uid, all_user_discounts)
        if not all_user_discounts:
            return None
        return sorted(all_user_discounts)[0]

    @classmethod
    def get_cheapest_available_line(cls, uid):
        """
        Вернуть наиболее "лучшую" скидку для пользователя

        При наличии нескольких активных скидок у пользователя, возвращает ту, которая дает лучшую скидку. Это
        необходимо из-за продуктового требования, что пользователь, имея несколько скидок, может купить продукты только
        по самой лучшей скидке, т.е. если у пользователя есть скидки 10% и 20%, то он может купить услугу только по
        скидке 20%
        """
        discount = cls.get_cheapest_available_discount(uid)
        if discount is None:
            return get_default_products_line()
        return discount.provided_line

    @classmethod
    def use_discount(cls, uid, discount_template_id):
        """
        Попытаться использовать скидку пользователя

        Если скидки нет или она многоразовая, то ничего не делаем
        """
        discount = cls.find_discount(uid, discount_template_id)
        if not discount:
            default_log.info('Couldn\'t find discount among active for user. Possible fraud? uid=%s discount_template_id=%s' %
                             (uid, discount_template_id))
            return
        if not discount.disposable:
            return
        # Скидка есть и она одноразовая, значит надо перенести ее в архив
        cls.archive_discount(uid, discount_template_id)

    @classmethod
    def find_discount(cls, uid, discount_template_id):
        for discount in cls._get_all_user_discounts(uid):
            if discount_template_id == discount.discount_template_id:
                return discount

    @classmethod
    def archive_discount(cls, uid, discount_template_id):
        """Перенести скидку в архив"""
        discounts = cls._get_all_user_discounts(uid)
        left_discounts = []
        for discount in discounts:
            if discount.discount_template_id == discount_template_id:
                DiscountArchive.create(uid, discount).save()
                continue
            left_discounts.append(discount)

        cls._save_discounts_as_active(uid, left_discounts)

    @classmethod
    def restore(cls, uid, discount_template_id):
        """Восстонавливает скидку из архива"""
        # Выдаем скидку не проверяя архив
        DiscountManager.add_discount_to_user(uid, discount_template_id, check_archive=False)
        # Удаляем из архива
        DiscountArchive.fetch(uid, discount_template_id).delete()

    @classmethod
    def _archive_outdated_and_save_discounts(cls, uid, discounts, force_update=False):
        cur_datetime = datetime.datetime.now()
        left_discounts = []
        for discount in discounts:
            if discount.end_datetime is not None and discount.end_datetime.replace(tzinfo=None) < cur_datetime:
                DiscountArchive.create(uid, discount).save()
                continue
            left_discounts.append(discount)

        if len(discounts) != len(left_discounts) or force_update:
            cls._save_discounts_as_active(uid, left_discounts)

        return left_discounts

    @classmethod
    def _get_all_user_discounts(cls, uid):
        res = disk_info.find_one_by_field(uid, {'key': cls.key})
        if res is None or not res['data']:
            return []
        all_user_discounts = [
            Discount.create(uid, i['ctime'], DiscountTemplate.get_discount_template(i['_id'])) for i in res['data']
        ]
        return all_user_discounts

    @classmethod
    def _save_discounts_as_active(cls, uid, discounts):
        data_to_write = []
        for discount in discounts:
            data_to_write.append(discount.get_database_dict())
        disk_info.put(uid, cls.key, data_to_write)
