# -*- coding: utf-8 -*-
import copy
import time
import uuid

import mpfs.engine.process
from mpfs.common.static.tags.promo_codes import ACTIVATED

from mpfs.core.billing import (
    BTIME,
    Client,
    CLIENT,
    DELETETIME,
    PID,
    Product,
    PRODUCT,
    PROMO_CODE,
    ServiceList,
    ServiceListHistory,
    SID,
    ProductCard
)
from mpfs.core.billing.api import service_create
from mpfs.core.promo_codes.dao.promo_codes import (
    PromoCodeArchiveDAO,
    PromoCodeArchiveDAOItem,
    PromoCodeDAO,
    PromoCodeDAOItem,
)
from mpfs.core.promo_codes.logic.discount_manager import DiscountManager

from mpfs.core.promo_codes.logic.errors import PromoCodeNotFound, UserAlreadyHasSuchPromoService
from mpfs.frontend.formatter.disk.billing import format_service_data

default_log = mpfs.engine.process.get_default_log()


class PromoCodeBase(object):
    dao = None
    dao_item_class = None

    def __init__(self, dao_item):
        if not isinstance(dao_item, self.dao_item_class):
            raise TypeError('`%s` expected. Got: %r' % (self.dao_item_class.__name__, dao_item))
        self.dao_item = dao_item

    @property
    def code(self):
        raise NotImplementedError()

    @classmethod
    def get_promo_code(cls, raw_promo_code):
        dao_item = cls.dao.find_promo_code(raw_promo_code)
        if dao_item is None:
            raise PromoCodeNotFound()
        return cls(dao_item)


class PromoCode(PromoCodeBase):
    dao = PromoCodeDAO()
    promo_code_archive_dao = PromoCodeArchiveDAO()
    dao_item_class = PromoCodeDAOItem

    @classmethod
    def get_promo_code(cls, raw_promo_code):
        dao_item = cls.dao.find_promo_code(raw_promo_code)
        return cls._build_promo_code_by_dao_item(dao_item)

    @classmethod
    def get_available_promo_code(cls, raw_promo_code):
        dao_item = cls.dao.find_available_promo_code(raw_promo_code)
        return cls._build_promo_code_by_dao_item(dao_item)

    @property
    def code(self):
        return self.dao_item.id

    @property
    def count(self):
        return self.dao_item.count

    @property
    def begin_datetime(self):
        return self.dao_item.begin_datetime

    @property
    def end_datetime(self):
        return self.dao_item.end_datetime

    def get_begin_timestamp(self):
        return self.dao_item.get_field_mongo_representation('begin_datetime', self.begin_datetime)

    def get_end_timestamp(self):
        return self.dao_item.get_field_mongo_representation('end_datetime', self.end_datetime)

    def is_last(self):
        return self.count == 1

    def remove(self):
        self.dao.remove(self.code)

    def format_dict(self):
        repr_data = {
            'promo_code_begin_timestamp': self.get_begin_timestamp(),
            'promo_code_end_timestamp': self.get_end_timestamp(),
        }
        return repr_data

    def apply(self, uid):
        raise NotImplemented('Not implemented for base promo code type')

    def decrement_count(self):
        self.dao_item.count -= 1
        self.dao.decrement_count(self.code)

    def add_to_archive(self, **kwargs):
        raise NotImplemented('Not implemented for base promo code type')

    def _get_data_for_archive(self, **kwargs):
        data_dict = copy.copy(self.dao_item.get_mongo_representation())
        data_dict['uid'] = kwargs['uid']
        data_dict['status'] = kwargs['status_to_set']

        data_dict.pop('begin_datetime')
        data_dict.pop('end_datetime')
        data_dict.pop('count')

        data_dict['promo_code'] = data_dict['_id']
        data_dict['_id'] = uuid.uuid4().hex
        data_dict['activation_timestamp'] = int(time.time())

        return data_dict

    def _archive_and_remove_if_necessary(self, **kwargs):
        """
        Добавляет промокод в архив и и удаляет промокод из активных, если число активаций исчерпалось.

        :param kwargs: данные, которые должны улететь в архив (зависит от типа промокода)
        """
        self.add_to_archive(**kwargs)
        if not self.is_last():
            self.decrement_count()
            return
        # Если при выборке объекта его count был 1, то значит теперь он поюзался и его надо удалить
        default_log.info(
            'The number of promo code usage was depleted, removing promo code. '
            'promo_code=%s, begin_datetime=%s, end_datetime=%s' %
            (self.code, self.begin_datetime, self.end_datetime)
        )
        self.remove()

    @classmethod
    def _build_promo_code_by_dao_item(cls, dao_item):
        if dao_item is None:
            raise PromoCodeNotFound()
        if dao_item.pid:
            return ServiceProviderPromoCode(dao_item)
        if dao_item.discount_template_id:
            return DiscountProviderPromoCode(dao_item)
        raise NotImplemented('Not implemented promo code type')


class ServiceProviderPromoCode(PromoCode):
    default_product_line = PROMO_CODE

    @property
    def pid(self):
        return self.dao_item.pid

    def add_to_archive(self, **kwargs):
        data_dict = super(ServiceProviderPromoCode, self)._get_data_for_archive(**kwargs)
        data_dict['sid'] = kwargs['linked_sid']
        data_dict['pid'] = self.pid

        archive_dao_item = PromoCodeArchiveDAOItem.create_from_mongo_dict(data_dict)
        archive_promo_code = PromoCodeArchive(archive_dao_item)
        archive_promo_code.save()

    def format_dict(self):
        repr_data = super(ServiceProviderPromoCode, self).format_dict()
        product = Product(self.pid)
        repr_data['names'] = product.name
        return repr_data

    def apply(self, uid):
        params = {CLIENT: Client(uid), PRODUCT: self.pid}
        services = ServiceList(**params) or ServiceListHistory(**params)
        if services:
            default_log.info('User already has promo code product. uid=%s, sid=%s, pid=%s, promo_code=%s' %
                             (uid, services[0]['sid'], self.pid, self.code))
            raise UserAlreadyHasSuchPromoService()
        new_service = service_create(uid, self.pid, self.default_product_line)
        default_log.info('Provided new service by using promo_code. uid=%s, sid=%s, pid=%s, promo_code=%s' %
                         (uid, new_service.sid, self.pid, self.code))
        self._archive_and_remove_if_necessary(linked_sid=new_service.sid, uid=uid, status_to_set=ACTIVATED)
        return {'billing_service': self._convert_service_to_output_format(new_service)}

    @staticmethod
    def _convert_service_to_output_format(service):
        formatted_serivce = copy.copy(service.dict())
        formatted_serivce[SID] = formatted_serivce['_id']
        formatted_serivce.pop('_id', None)
        formatted_serivce[DELETETIME] = formatted_serivce[BTIME]
        formatted_serivce[PRODUCT] = Product(formatted_serivce[PID])
        return format_service_data(formatted_serivce)


class DiscountProviderPromoCode(PromoCode):

    @property
    def discount_template_id(self):
        return self.dao_item.discount_template_id

    def add_to_archive(self, **kwargs):
        data_dict = super(DiscountProviderPromoCode, self)._get_data_for_archive(**kwargs)
        data_dict['discount_template_id'] = self.discount_template_id

        archive_dao_item = PromoCodeArchiveDAOItem.create_from_mongo_dict(data_dict)
        archive_promo_code = PromoCodeArchive(archive_dao_item)
        archive_promo_code.save()

    def apply(self, uid):
        discount = DiscountManager.add_discount_to_user(uid, self.discount_template_id)

        default_log.info('Provided new discount by using promo_code. uid=%s, discount_template_id=%s, promo_code=%s' %
                         (uid, discount.discount_template_id, self.code))
        self._archive_and_remove_if_necessary(discount_template_id=discount.discount_template_id, uid=uid,
                                              status_to_set=ACTIVATED)
        return {'discount': self._convert_discount_to_output_format(discount)}

    @staticmethod
    def _convert_discount_to_output_format(discount):
        result = {
            'disposable': discount.disposable,
            'percentage': ProductCard.get_discount_percentage_by_line(discount.provided_line)
        }
        if discount.end_datetime:
            result['end_timestamp'] = int(time.mktime(discount.end_datetime.timetuple()))
        return result


class PromoCodeArchive(PromoCodeBase):
    dao = PromoCodeArchiveDAO()
    dao_item_class = PromoCodeArchiveDAOItem

    @property
    def code(self):
        return self.dao_item.promo_code

    def save(self):
        self.dao.save(self.dao_item)
