# -*- coding: utf-8 -*-
import datetime

import mpfs.engine.process

from mpfs.core.promo_codes.logic.errors import (
    AttemptToActivateUsedPromoCode,
    PromoCodeNotFound,
    TooEarlyPromoCodeActivation,
    TooLatePromoCodeActivation,
)
from mpfs.core.promo_codes.logic.promo_code import (
    PromoCode,
    PromoCodeArchive
)

default_log = mpfs.engine.process.get_default_log()


class PromoCodeManager(object):

    @classmethod
    def find_and_activate_promo_code(cls, uid, raw_promo_code):
        """
        Активировать промокод. В случае успеха вернет только что выданную биллинговую услугу

        :rtype: CommonService
        """
        try:
            promo_code = PromoCode.get_available_promo_code(raw_promo_code)
        except PromoCodeNotFound:
            cls._handle_missing_active_promo_code(raw_promo_code)
            raise

        result = promo_code.apply(uid)
        return result

    @staticmethod
    def _handle_missing_active_promo_code(raw_promo_code):
        try:
            promo_code = PromoCode.get_promo_code(raw_promo_code)
        except PromoCodeNotFound:
            pass
        else:
            # Нашли промокод среди неархивированных, но либо акция еще неактивна, либо ошибка с count
            if promo_code.count < 1:
                # В этом случае считаем, что промокода не нашли
                default_log.info('Promo code count < 1. promo_code=%s count=%s' % (promo_code.code, promo_code.count))
                raise PromoCodeNotFound()

            repr_data = promo_code.format_dict()
            if datetime.datetime.now() > promo_code.end_datetime.replace(tzinfo=None):
                raise TooLatePromoCodeActivation(data=repr_data)
            else:
                raise TooEarlyPromoCodeActivation(data=repr_data)

        PromoCodeArchive.get_promo_code(raw_promo_code)
        # если дошли до сюда, то промокод уже активирован
        raise AttemptToActivateUsedPromoCode()
