# -*- coding: utf-8 -*-
"""

MPFS
CORE

Действия push-нотификаций

"""
import time
import inspect
import sys
from copy import deepcopy, copy
from itertools import ifilter
from collections import defaultdict

import mpfs.engine.process

from mpfs.core.services.passport_service import Passport
from mpfs.common.static.tags.push import SPACE, SPACE_GROUP
from mpfs.core.filesystem.quota import Quota

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

passport_instance = Passport()


class Default(object):
    name = 'default'

    def process(self, data):
        return [data,]

    def complete_uid(self, uid):
        pass

    def resources_filter(self, xdata, resources):
        return xdata

    def extract_fids(self, xdata):
        return []


class CommonDiff(Default):
    def resources_filter(self, xdata, resources):
        values = []
        for item in xdata.get('values', []):
            if item.get('parameters', {}).get('fid') in resources:
                values.append(item)
        if values:
            xdata['values'] = values
            return xdata
        else:
            return []

    def extract_fids(self, xdata):
        values = []
        for item in xdata.get('values', []):
            fid = item.get('parameters', {}).get('fid')
            if fid:
                values.append(fid)
        return values


class Diff(CommonDiff):
    
    name = 'diff'
    
    def process(self, data):
        values = []
        with_shared = False
        is_group_root = False
        
        for each in data['xiva_data']:
            each['type'] = each.pop('op')
            each["folder"] = each['key'][:each['key'].rfind('/')+1]
            if not with_shared:
                with_shared = each.pop('with_shared', None)
            if not is_group_root:
                is_group_root = each.pop('is_group_root', None)
            values.append({'tag' : 'op', 'value' : '', 'parameters' : each})
            
        xdata = deepcopy(data)
        xdata['uid'] = data['uid']
        xdata['xiva_data'] = {
            'root' : {
                'tag' : 'diff', 
                'parameters' : {
                    'new' : data['new_version'],
                    'old' : data['old_version'],
                    }
                },
            'values' : values,
        }
        
        if with_shared:
            xdata['xiva_data']['root']['parameters']['with_shared'] = 1
        elif is_group_root:
            xdata['xiva_data']['root']['parameters']['is_group_root'] = 1
        
        return [xdata,]

class DiffGroup(CommonDiff):
    
    name = 'diff_group'
    
    def process(self, data):
        result = []
        from mpfs.core.social.share import Group
        
        job_data = deepcopy(data)
        gid = job_data.pop('gid')
        group = Group.load(gid)

        committer_uid = job_data.pop('committer')
        committer_info = passport_instance.userinfo_summary(committer_uid)
        committer_info = dict(('committer_%s' % k, v) for k,v in committer_info.iteritems())

        xiva_data = job_data.pop('xiva_data')

        new_version = job_data.pop('new_version')
        old_version = job_data.pop('old_version')
        values = []
        with_shared = False
        is_group_root = False
        broken = False
        
        for each in xiva_data:
            if committer_uid != group.owner:
                each['key'] = group.get_group_path(each['key'], committer_uid)
            each['type'] = each.pop('op')
            each["folder"] = each['key'][:each['key'].rfind('/')+1]
            if not broken:
                broken = each.get('broken')
            if not with_shared:
                with_shared = each.pop('with_shared', None)
            if not is_group_root:
                is_group_root = each.pop('is_group_root', None)
            values.append({'tag' : 'op', 'value' : '', 'parameters' : each})
             
        #=======================================================================
        # Отправка изменений владельцу группы
        try:
            owner_old_version = old_version[group.owner]
        except KeyError:
            pass
        else:
            xdata = deepcopy(job_data)
            xdata['uid'] = group.owner
            xdata['xiva_data'] = {
                'root' : {
                    'tag' : 'diff', 
                    'parameters' : {
                        'new' : new_version,
                        'old' : owner_old_version,
                    }
                },
                'values' : deepcopy(values),
            }
            
            if committer_uid != group.owner:
                xdata.pop('connection_id', '')
                if not broken:
                    xdata['xiva_data']['root']['parameters'].update(committer_info)
            else:
                if with_shared:
                    xdata['xiva_data']['root']['parameters']['with_shared'] = 1
                elif is_group_root:
                    xdata['xiva_data']['root']['parameters']['is_group_root'] = 1
                    
            xdata['new_version'] = new_version
            xdata['old_version'] = owner_old_version
             
            result.append(xdata)

        
        #=======================================================================
        # Отправка изменений пользователям, подключенным к группе
        for link in group.iterlinks():
            try:
                user_old_version = old_version[link.uid]
            except KeyError:
                continue
            else:
                local_values = deepcopy(values)
                xdata = deepcopy(job_data)
                xdata['uid'] = link.uid
                for each in local_values:
                    each['parameters']['key'] = link.get_link_path(each['parameters']['key'])
                    each['parameters']['folder'] = each['parameters']['key'][:each['parameters']['key'].rfind('/')+1]
                xdata['xiva_data'] = {
                    'root' : {
                        'tag' : 'diff', 
                        'parameters' : {
                            'new' : new_version,
                            'old' : user_old_version,
                        }
                    },
                    'values' : local_values,
                }
                
                if committer_uid != link.uid:
                    xdata.pop('connection_id', '')
                    if not broken:
                        xdata['xiva_data']['root']['parameters'].update(committer_info)
                else:
                    if with_shared:
                        xdata['xiva_data']['root']['parameters']['with_shared'] = 1
                    elif is_group_root:
                        xdata['xiva_data']['root']['parameters']['is_group_root'] = 1
                
                xdata['new_version'] = new_version
                xdata['old_version'] = user_old_version
                
                result.append(xdata)
                
        return result

class DiffGroupMove(DiffGroup):
    
    name = 'diff_group_move'
    
    def get_new_version(self, uid):
        new_version = self.data.get('new_version')
        if isinstance(new_version, dict):
            return new_version.get(uid) or new_version['group']
        else:
            return new_version

    def group_diff_for_user(self, link, job_data, new_version, user_old_version, committer_info):
        xdata = {}
        xdata['uid'] = link.uid
        job_data = deepcopy(job_data)
        for each in job_data:
            each['parameters']['key'] = link.get_link_path(each['parameters']['key'])
            each['parameters']['folder'] = each['parameters']['key'][:each['parameters']['key'].rfind('/')+1]
        xdata['xiva_data'] = {
            'root' : {
                'tag' : 'diff', 
                'parameters' : {
                    'new' : new_version,
                    'old' : user_old_version,
                }
            },
            'values' : job_data,
        }
        if committer_info['committer_uid'] != link.uid:
            xdata['xiva_data']['root']['parameters'].update(committer_info)
        return xdata

    
    def process(self, data):
        result = []
        from mpfs.core.social.share import Group
    
        self.data = data
        
        job_data = deepcopy(data)
        #=======================================================================
        #
        committer_uid = job_data.pop('committer')
        committer_info = passport_instance.userinfo_summary(committer_uid)
        committer_info = dict(('committer_%s' % k, v) for k,v in committer_info.iteritems())
        #=======================================================================
        #
        old_version_src = job_data.get('old_version_src')
        new_version_src = job_data.get('new_version_src')
        new_version_tgt = job_data.get('new_version_tgt')
        old_version_tgt = job_data.get('old_version_tgt')
        connection_id = job_data.get('connection_id')
        #=======================================================================
        #
        source_change = job_data['xiva_data']['source']['data']
        target_change = job_data['xiva_data']['target']['data']
        #=======================================================================
        # 
        source_gid = job_data['xiva_data']['source']['gid']
        target_gid = job_data['xiva_data']['target']['gid']
        #=======================================================================
        if source_gid:
            source_group = Group.load(source_gid)
            source_group_links = dict((link.uid, link) for link in ifilter(lambda link: link.uid in old_version_src, source_group.iterlinks()))
        else:
            source_group = None
            source_group_links = {}
        if target_gid:
            target_group = Group.load(target_gid)
            target_group_links = dict((link.uid, link) for link in ifilter(lambda link: link.uid in old_version_tgt, target_group.iterlinks()))
        else:
            target_group = None
            target_group_links = {}
        #=======================================================================
        src_with_shared = False
        src_is_group_root = False
        src_values = []
        for each in source_change:
            each['type'] = each.pop('op')
            if source_gid and committer_uid != source_group.owner:
                each['key'] = source_group.get_group_path(each['key'], committer_uid)
            each["folder"] = each['key'][:each['key'].rfind('/')+1]
            if not src_with_shared:
                src_with_shared = each.pop('with_shared', None)
            if not src_is_group_root:
                src_is_group_root = each.pop('is_group_root', None)
            src_values.append({'tag' : 'op', 'value' : '', 'parameters' : each})
        dst_values = []
        dst_with_shared = False
        dst_is_group_root = False
        for each in target_change:
            each['type'] = each.pop('op')
            if target_gid and committer_uid != target_group.owner:
                each['key'] = target_group.get_group_path(each['key'], committer_uid)
            each["folder"] = each['key'][:each['key'].rfind('/')+1]
            if not dst_with_shared:
                dst_with_shared = each.pop('with_shared', None)
            if not dst_is_group_root:
                dst_is_group_root = each.pop('is_group_root', None)
            dst_values.append({'tag' : 'op', 'value' : '', 'parameters' : each})

        users_in_both_groups = {}
        unique_source_links = {}
        for uid, slink in source_group_links.iteritems():
            try:
                tlink = target_group_links.pop(uid)
            except KeyError:
                unique_source_links[uid] = slink
            else:
                users_in_both_groups[uid] = (slink, tlink)

        for uid, (slink, tlink) in users_in_both_groups.iteritems():
            old_version = old_version_src[uid]
            new_version = new_version_tgt
            src_data = self.group_diff_for_user(slink, src_values, new_version, old_version, committer_info)
            dst_data = self.group_diff_for_user(tlink, dst_values, new_version, old_version_tgt[uid], committer_info)
            src_data['xiva_data']['values'].extend(dst_data['xiva_data']['values'])
            if uid == committer_uid:
                src_data['connection_id'] = connection_id
            src_data['new_version'] = new_version
            src_data['old_version'] = old_version
            result.append(src_data)
            
        for uid, link in unique_source_links.iteritems():
            old_version = old_version_src[uid]
            new_version = new_version_src
            src_data = self.group_diff_for_user(link, src_values, new_version, old_version, committer_info)
            if uid == committer_uid:
                src_data['connection_id'] = connection_id
                src_data['xiva_data']['values'].extend(dst_values)
                if src_with_shared:
                    src_data['xiva_data']['root']['parameters']['with_shared'] = 1
                elif src_is_group_root:
                    src_data['xiva_data']['root']['parameters']['is_group_root'] = 1
            src_data['new_version'] = new_version
            src_data['old_version'] = old_version
            result.append(src_data)
            
        for uid, link in target_group_links.iteritems():
            old_version = old_version_tgt[uid]
            if uid == committer_uid:
                new_version = new_version_src
            else:
                new_version = new_version_tgt
            dst_data = self.group_diff_for_user(link, dst_values, new_version, old_version, committer_info)
            if uid == committer_uid:
                dst_data['connection_id'] = connection_id
                dst_data['xiva_data']['values'].extend(src_values)
                if dst_with_shared:
                    src_data['xiva_data']['root']['parameters']['with_shared'] = 1
                elif dst_is_group_root:
                    src_data['xiva_data']['root']['parameters']['is_group_root'] = 1
            dst_data['new_version'] = new_version
            dst_data['old_version'] = old_version
            result.append(dst_data)

        if source_gid and target_gid and source_group.owner == target_group.owner:
            new_version = new_version_tgt
            #=======================================================================
            # отправка владельцу исходного каталога
            try:
                old_version = old_version_src[source_group.owner]
            except Exception:
                pass
            else:
                xdata = {}
                xdata['uid'] = source_group.owner
                values = []
                values.extend(src_values)
                values.extend(dst_values)
                xdata['xiva_data'] = {
                                      'root' : {
                                                'tag' : 'diff', 
                                                'parameters' : {
                                                                'new' : new_version,
                                                                'old' : old_version,
                                                                }
                                                },
                                      'values' : values,
                                      }
                if committer_uid == source_group.owner:
                    xdata['connection_id'] = connection_id
                else:
                    xdata['xiva_data']['root']['parameters'].update(committer_info)
                xdata['new_version'] = new_version
                xdata['old_version'] = old_version
                result.append(xdata)
        else:
            if source_gid:
                #=======================================================================
                # отправка владельцу исходного каталога
                try:
                    old_version = old_version_src[source_group.owner]
                except Exception:
                    pass
                else:
                    xdata = {}
                    xdata['uid'] = source_group.owner
                    new_version = new_version_src
                    xdata['xiva_data'] = {
                                          'root' : {
                                                    'tag' : 'diff', 
                                                    'parameters' : {
                                                                    'new' : new_version,
                                                                    'old' : old_version,
                                                                    }
                                                    },
                                          'values' : src_values,
                                          }
                    if committer_uid == source_group.owner:
                        xdata['connection_id'] = connection_id
                        xdata['xiva_data']['values'].extend(dst_values)
                        if src_with_shared:
                            xdata['xiva_data']['root']['parameters']['with_shared'] = 1
                        elif src_is_group_root:
                            xdata['xiva_data']['root']['parameters']['is_group_root'] = 1
                    else:
                        xdata['xiva_data']['root']['parameters'].update(committer_info)
                    xdata['new_version'] = new_version
                    xdata['old_version'] = old_version
                    result.append(xdata)
            if target_gid:
                #=======================================================================
                # отправка владельцу каталога назначения
                try:
                    old_version = old_version_tgt[target_group.owner]
                except Exception:
                    pass
                else:
                    xdata = {}
                    xdata['uid'] = target_group.owner
                    if committer_uid == target_group.owner:
                        new_version = new_version_src
                    else:
                        new_version = new_version_tgt
                    xdata['xiva_data'] = {
                                          'root' : {
                                                    'tag' : 'diff', 
                                                    'parameters' : {
                                                                    'new' : new_version,
                                                                    'old' : old_version,
                                                                    }
                                                    },
                                          'values' : dst_values,
                                          }
                    if committer_uid == target_group.owner:
                        xdata['connection_id'] = connection_id
                        xdata['xiva_data']['values'].extend(src_values)
                        if dst_with_shared:
                            xdata['xiva_data']['root']['parameters']['with_shared'] = 1
                        elif dst_is_group_root:
                            xdata['xiva_data']['root']['parameters']['is_group_root'] = 1
                    else:
                        xdata['xiva_data']['root']['parameters'].update(committer_info)
                    xdata['new_version'] = new_version
                    xdata['old_version'] = old_version
                    result.append(xdata)

        return result


class DiffGroupTrashAppend(DiffGroupMove):

    name = 'diff_group_trash_append'

    def process(self, data):
        result = []
        from mpfs.core.social.share import Group
        self.data = data
        job_data = deepcopy(data)
        #=======================================================================
        #
        committer_uid = job_data.pop('committer')
        committer_info = passport_instance.userinfo_summary(committer_uid)
        committer_info = dict(('committer_%s' % k, v) for k,v in committer_info.iteritems())
        #=======================================================================
        old_version_src = job_data.get('old_version_src')
        new_version_src = job_data.get('new_version_src')
        connection_id = job_data.get('connection_id')
        #=======================================================================
        source_change = job_data['xiva_data']['source']['data']
        target_change = job_data['xiva_data']['target']['data']
        #=======================================================================
        source_gid = job_data['xiva_data']['source']['gid']
        #=======================================================================
        if source_gid:
            source_group = Group.load(source_gid)
            source_group_links = dict((link.uid, link) for link in ifilter(lambda link: link.uid in old_version_src, source_group.iterlinks()))
        else:
            source_group = None
            source_group_links = {}
        #=======================================================================
        src_with_shared = False
        src_is_group_root = False
        src_values = []
        for each in source_change:
            each['type'] = each.pop('op')
            if source_gid and committer_uid != source_group.owner:
                each['key'] = source_group.get_group_path(each['key'], committer_uid)
            each["folder"] = each['key'][:each['key'].rfind('/')+1]
            if not src_with_shared:
                src_with_shared = each.pop('with_shared', None)
            if not src_is_group_root:
                src_is_group_root = each.pop('is_group_root', None)
            src_values.append({'tag' : 'op', 'value' : '', 'parameters' : each})
        dst_values = defaultdict(list)
        for uid, targe_data in target_change.iteritems():
            for each in targe_data:
                each['type'] = 'new'
                each["folder"] = '/trash/'
                dst_values[uid].append({'tag' : 'op', 'value' : '', 'parameters' : each})

        for uid, link in source_group_links.iteritems():
            old_version = old_version_src[uid]
            new_version = new_version_src
            src_data = self.group_diff_for_user(link, src_values, new_version, old_version, committer_info)
            dst_data = dst_values[uid]
            if dst_data:
                if uid == committer_uid:
                    src_data['connection_id'] = connection_id
                src_data['xiva_data']['values'].extend(dst_data)
            src_data['new_version'] = new_version
            src_data['old_version'] = old_version
            result.append(src_data)

        if source_gid:
            #=======================================================================
            # отправка владельцу исходного каталога
            try:
                old_version = old_version_src[source_group.owner]
            except Exception:
                pass
            else:
                xdata = {}
                xdata['uid'] = uid = source_group.owner
                new_version = new_version_src
                xdata['xiva_data'] = {
                                      'root' : {
                                                'tag' : 'diff',
                                                'parameters' : {
                                                                'new' : new_version,
                                                                'old' : old_version,
                                                                }
                                                },
                                      'values' : src_values,
                                      }
                if committer_uid == uid:
                    xdata['connection_id'] = connection_id
                else:
                    xdata['xiva_data']['root']['parameters'].update(committer_info)
                xdata['xiva_data']['values'].extend(dst_values[uid])
                xdata['new_version'] = new_version
                xdata['old_version'] = old_version
                result.append(xdata)

        return result


class DiffBrowser(Default):

    name = 'diff_browser'


class Space(Default):
    name = SPACE

    def process(self, data):
        self.push_classes = {}
        uid = data['uid']
        push_class = data['class']
        if self.controller.is_notification_sent(uid, push_class):
            return []
        else:
            self.push_classes = {
                                 uid : push_class,
                                 }
            return [data,]

    def complete_uid(self, uid):
        self.controller.set_notification_sent(uid, self.push_classes[uid])
        

class SpaceGroup(Space):
    name = SPACE_GROUP

    def process(self, data):
        self.push_classes = {}
        result = []
        from mpfs.core.social.share import Group
        self.data = data
        quota = Quota()
        group = Group.load(self.data['gid'])
        committer = self.data['committer']
        method = self.data['method']
        connection_id = self.data.get('connection_id', '')

        def _process_one_uid(uid):
            space_info = quota.report(group.owner)
            limit = space_info['limit']
            used = space_info['used']
            free = space_info['free']
            push_tag = quota.get_push_type(uid)
            if push_tag:
                if not self.controller.is_notification_sent(uid, push_tag.klass):
                    xiva_data = {
                                 'root' : {
                                    'tag' : SPACE,
                                    'parameters' : {
                                        'type' : push_tag.type,
                                        'limit' : limit,
                                        'free' : free,
                                        'used' : used,
                                        },
                                    },
                                'values' : [],
                                }
                    if committer == uid:
                        if method == 'move_resource':
                            return
                        else:
                            cid = connection_id
                    else:
                        cid = ''
                    data = {
                            'class': push_tag.klass,
                            'uid' : uid,
                            'new_version' : 1,
                            'xiva_data' : xiva_data,
                            'connection_id' : cid,
                            'operation' : 'action',
                            'action_name' : SPACE,
                            }
                    result.append(data)
                    self.push_classes[uid] = push_tag.klass

        #=======================================================================
        # owner
        _process_one_uid(group.owner)
        #=======================================================================
        # users
        for uid in group.iteruids():
            _process_one_uid(uid)
        #=======================================================================
        return result


actions = dict((cls.name, cls()) for (clsname, cls) in
            filter(lambda (c_name, c): hasattr(c, 'name') and c.name and (isinstance(c.name, str)),
                inspect.getmembers(sys.modules[__name__], inspect.isclass)))

def get(name):
    try:
        action = actions[name]
    except KeyError:
        error_log.error('Wrong action name: %s' % name)
        raise
    else:
        return action
    
