# -*- coding: utf-8 -*-
"""

MPFS
CORE

Контроллер push-нотификаций

"""
import time

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common.util import to_json
from mpfs.core.pushnotifier import actions
from mpfs.common.static.tags.push import *
from mpfs.core.user.constants import PUSH_TRANSLATIONS, DEFAULT_LOCALE
from mpfs.core.services.passport_service import passport
from mpfs.core.services.push_service import XivaSendService, XivaSubscribeService

error_log = mpfs.engine.process.get_error_log()


class PushController(object):
    sender = XivaSendService()
    subscriber = XivaSubscribeService()

    @staticmethod
    def _create_xiva_filter_for_apps(events=None, file_ids=None):
        """
        Создать xiva-фильтр для мобильных приложений

        :param events: list. Cобытия, на которые хотим подписаться
        :param file_ids: list. Список file_ids, для события `diff`
            Если переданы `file_ids` и событие `diff`, то отсылаем diff-ы только для переданных `file_ids`
            если `file_ids` нет, но есть `diff`, то шлем все diff-ы
        :return: xiva_filter
        """
        if not events:
            return None
        events = set(events)

        if file_ids and 'diff' in events:
            events.remove('diff')
            return {
                'vars': {
                    "DiffEvent": {"$event": ["diff"]},
                    "RequiredFileIds": {"file_ids": {"$has": file_ids}},
                    "CommonEvents": {"$event": list(events)},
                },
                'rules': [
                    {"if": "DiffEvent & RequiredFileIds", "do": "send_bright"},
                    {"if": "CommonEvents", "do": "send_bright"},
                    {"do": "skip"},
                ],
            }
        else:
            return {
                'vars': {
                    "CommonEvents": {"$event": list(events)},
                },
                'rules': [
                    {"if": "CommonEvents", "do": "send_bright"},
                    {"do": "skip"},
                ]
            }

    def mobile_subscribe(self, uid, app_token, events=None, file_ids=None):
        """
        Сделать мобильную подписку

        :param uid: uid пользователя, которого надо подписать
        :param app_token: токен, получаемый от моб. устройств(формат как у Пушкина).
        :param events: список событий, на которые хотят подписаться
        :param file_ids: список file_ids. Если 'diff' in events, то дополнительный фильтр по file_ids.
        """
        xiva_filter = self._create_xiva_filter_for_apps(events=events, file_ids=file_ids)
        self.subscriber.subscribe_app(uid, app_token, xiva_filter=xiva_filter)

    def mobile_unsubscribe(self, uid, app_token):
        self.subscriber.unsubscribe_app(uid, app_token)

    def subscribe(self, uid, callback, options):
        """Подписка пользователя на push по callback-у"""
        self.subscriber.subscribe_url(uid, callback)

    def send(self, data):
        action_name = data.get('action_name', 'default')
        action = actions.get(action_name)
        action.controller = self
        pushes = action.process(data)

        processed_uids = set()
        for item in pushes:
            uid = item.get('uid')
            push_class = item.get('class', 'other')
            xiva_data = item.pop('xiva_data')
            repack = item.pop('repack', None)
            file_ids = action.extract_fids(xiva_data)
            self._process_and_send_push(uid, xiva_data, push_class,
                                        file_ids=file_ids, extra_push_data=item, repack=repack)

            if uid not in processed_uids:
                action.complete_uid(uid)
                processed_uids.add(uid)

    def _process_and_send_push(self, uid, xiva_data, push_class, file_ids=None, extra_push_data=None, repack=None):
        if not isinstance(extra_push_data, dict):
            extra_push_data = {}

        keys = {}
        try:
            localized_msg = self._get_localized_msg(uid, xiva_data, push_class)
            if localized_msg is not None:
                keys.update(localized_msg)
        except Exception as e:
            error_log.exception("Error occurred during fetching localized message %s", e)
        if file_ids:
            keys.update({'file_ids': list(file_ids)})

        connection_id = extra_push_data.get('connection_id')

        """
        В данном параметре содержится перечень имен событий (event), по которым данный метод
        должен игнорировать to_json() для избежания двойного экранирования.
        Является обходным маневром для мобильных пушей, которые выставляет djfs-albums. В ином случае двойное
        экранирование препятствует доставке пушей.
        """
        if push_class in settings.push['events_for_ignore_json_encoding']:
            self.sender.send(uid, push_class, xiva_data, keys=keys, connection_id=connection_id, repack=repack)
        else:
            self.sender.send(uid, push_class, to_json(xiva_data), keys=keys, connection_id=connection_id, repack=repack)

    def _get_localized_msg(self, uid, xiva_data, push_class):
        """
        Порт костыля из пушкина
        """
        if push_class not in (SHARE_INVITE_NEW, SPACE_IS_LOW, SPACE_IS_FULL):
            return None
        try:
            user_locale = passport.public_userinfo(uid)['locale']
        except KeyError:
            user_locale = DEFAULT_LOCALE

        if push_class == SHARE_INVITE_NEW and xiva_data['root']['parameters']['for'] == 'owner':
            return None
        elif push_class == SHARE_INVITE_NEW and xiva_data['root']['parameters']['for'] == 'actor':
            owner = ''
            folder = ''
            for value in xiva_data['values']:
                if value['tag'] == 'owner':
                    owner = value['parameters']['name']
                elif value['tag'] == 'folder':
                    folder = value['parameters']['name']
            return {'localized_msg': PUSH_TRANSLATIONS[push_class][user_locale] % {'owner': owner, 'folder': folder}}
        elif push_class == SPACE_IS_LOW:
            free = int(xiva_data['root']['parameters']['free']) / (2 ** 20)
            return {'localized_msg': PUSH_TRANSLATIONS[push_class][user_locale] % {'free': str(free)}}
        elif push_class == SPACE_IS_FULL:
            return {'localized_msg': PUSH_TRANSLATIONS[push_class][user_locale]}

    def is_notification_sent(self, uid, key):
        from mpfs.core.user.base import User
        user = User(uid)
        value = user.states.get(key, PUSH_NAMESPACE)
        if value:
            result = value > time.time() - settings.push['low_space_delay']
        else:
            result = False
        return result

    def set_notification_sent(self, uid, key):
        from mpfs.core.user.base import User
        user = User(uid)
        user.set_state(key, int(time.time()), PUSH_NAMESPACE)
