# -*- coding: utf-8 -*-
'''

MPFS-QUEUE
Центральный интерфейс обращения к очереди
Разделяет задачи на нужные очереди

'''
import time

from copy import copy

import mpfs.engine.process
from mpfs.common.static import codes
from mpfs.common.static.tags.queue import QueueNames

from mpfs.common.util import to_json
from mpfs.config import settings
from mpfs.common.errors import MPFSRootException

from mpfs.core.services.queller_service import http_queller
from mpfs.engine.queue2.utils import get_master_dc


FEATURE_TOGGLES_EXECUTE_OPERATIONS_IN_MASTER_DC = settings.feature_toggles['execute_operations_in_master_dc']
QUEUE2_PUT_TASKS_IN_NEW_QUEUE = settings.queue2['put_tasks_in_new_queue']
QUEUE2_JTYPES_FOR_MASTER_DC = settings.queue2['jtypes_for_master_dc']
QUEUE2_TYPES_FOR_MASTER_DC = settings.queue2['types_for_master_dc']
QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT = settings.queue2['async_task_data_size_limit']


log = mpfs.engine.process.get_default_log()


class QueueDispatcher(object):
    """
    Диспетчер очередей
    """
    DEFAULT_QUEUE = 'queue_default'

    @staticmethod
    def _get_uid_from_params(params):
        # магия по получению уида из данных операции, нужно, чтобы положить данные для таски на шард!
        # чтобы не нагружать одну базу - сделано так, потому что в тасках уид явно не везде передается и это печально
        uid = None
        if isinstance(params, dict) and 'uid' in params:
            uid = str(params['uid'])
        elif isinstance(params, list) and len(params) and isinstance(params[0], dict) and 'uid' in params[0]:
            uid = str(params[0]['uid'])
        return uid

    def put(self, command, jtype, delay=None, stime=None, priority=None, deduplication_id=None, robust=False,
            op_type=None):
        # импорты из queue2.celery должны быть тут, иначе может не работать: mpfs, воркеры celery или админские скрипты
        from mpfs.engine.queue2.job_types import QUEUE2_JOB_TYPES_MAP
        from mpfs.engine.queue2.celery import TaskData
        from mpfs.engine.queue2.celery import KWARGS_DEDUPLICATION_ID_PARAMETER

        if QUEUE2_PUT_TASKS_IN_NEW_QUEUE and jtype in QUEUE2_JOB_TYPES_MAP:
            celery_task = QUEUE2_JOB_TYPES_MAP[jtype]
            if not isinstance(command, dict):
                raise TypeError('dict expected as command parameter (command type: %s, command: %s, jtype: %s)' %
                                (type(command), command, jtype))
            if 'context' in command and celery_task.app.conf.CELERY_ALWAYS_EAGER is False:
                raise RuntimeError('context is reserved parameter (jtype: %s, command: %s)' % (jtype, command))

            extra_options = {}
            if delay:
                extra_options['countdown'] = delay
            if stime:
                delay = int(stime - time.time())
                if delay > 0:
                    extra_options['countdown'] = delay

            # apply_async добавляет context в изначальный словарь и может его изменить:
            # https://st.yandex-team.ru/CHEMODAN-29853
            params = copy(command)

            # из всего этого бардака полезное только обработка operation_failing - остальное это из-за https://st.yandex-team.ru/CHEMODAN-30601
            if jtype == 'xiva' or jtype == 'notify' or jtype == 'call_url':
                params_length = len(to_json(params))
                if params_length > QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT:
                    log.info('serialized json data exceedes limit (%d > %d)' % (params_length,
                                                                                QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT))
                    uid = self._get_uid_from_params(params)
                    task_data = TaskData(uid=uid, data=params)
                    params = {'task_data': task_data}
            elif jtype == 'search' or jtype == 'search_photoslice' or \
                 jtype == 'search_group' or jtype == 'search_photoslice_group':
                data_length = len(to_json(params['data']))
                if data_length > QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT:
                    log.info('serialized json data exceedes limit (%d > %d)' % (data_length,
                                                                                QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT))
                    data = params.pop('data')
                    uid = self._get_uid_from_params(data)
                    task_data = TaskData(uid=uid, data=data)
                    params['task_data'] = task_data
                    params['data'] = None
            elif jtype == 'post_remove':
                params_length = len(to_json(params))
                if params_length > QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT:
                    log.info('serialized json data exceedes limit (%d > %d)' % (params_length,
                                                                                QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT))
                    uid = self._get_uid_from_params(params)
                    removed = params.pop('removed')
                    extra = params.pop('extra', None)
                    task_data = TaskData(uid=uid, data={'removed': removed, 'extra': extra})
                    params['task_data'] = task_data
                    params['removed'], params['extra'] = None, None
            elif jtype == 'symlinks_remove' or jtype == 'symlinks_update':
                params_length = len(to_json(params))
                if params_length > QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT:
                    log.info('serialized json data exceedes limit (%d > %d)' % (params_length,
                                                                                QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT))
                    uid = self._get_uid_from_params(params)
                    symlinks = params.pop('symlinks')
                    task_data = TaskData(uid=uid, data=symlinks)
                    params['task_data'] = task_data
                    params['symlinks'] = None
            elif jtype == 'process_device_install':
                params_length = len(to_json(params))
                if params_length > QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT:
                    log.info('serialized json data exceedes limit (%d > %d)' % (params_length,
                                                                                QUEUE2_ASYNC_TASK_DATA_SIZE_LIMIT))
                    uid = self._get_uid_from_params(params)
                    info = params.pop('info')
                    task_data = TaskData(uid=uid, data=info)
                    params['task_data'] = task_data
                    params['info'] = None
            elif jtype == 'operation_failing':
                # https://st.yandex-team.ru/CHEMODAN-30400
                # celery не может сериализовать объекты типа MPFSRootException в JSON (а монга может через repr),
                # поэтому делаем тут явно
                error = params.get('error', None)
                if isinstance(error, MPFSRootException):
                    params['error'] = error.repr()
                elif isinstance(error, dict):
                    params['error'] = error
                else:
                    # если словили что-то другое, вставляем заглушку:
                    params['error'] = {
                        'message': '%r' % error,
                        'code': codes.MPFS_ERROR,
                        'response': codes.INTERNAL_ERROR,
                    }

            if deduplication_id is not None:
                params[KWARGS_DEDUPLICATION_ID_PARAMETER] = deduplication_id

            if robust:
                master_dc = None
                if (FEATURE_TOGGLES_EXECUTE_OPERATIONS_IN_MASTER_DC and
                            jtype in QUEUE2_JTYPES_FOR_MASTER_DC and
                            op_type in QUEUE2_TYPES_FOR_MASTER_DC):
                    master_dc = get_master_dc(uid=params.get('uid'))

                # не поддерживает delay, countdown, eta, expires, time_limit, link, link_error
                task = http_queller.send_celery_task(celery_task, kwargs=params, master_dc=master_dc, **extra_options)
                log.info('queue2 robust put job %s with id %s' % (task['task'], task['id']))
            else:
                if jtype in settings.queue2['secondary_submit']['jtypes']:
                    extra_options['queue'] = QueueNames.SECONDARY_SUBMIT

                task = celery_task.apply_async(kwargs=params, **extra_options)
                log.info('queue2 put job %s with id %s' % (task.task_name, task.id))
        else:
            raise NotImplementedError("Use queue2")


mpfs_queue = QueueDispatcher()
