# -*- coding: utf-8 -*-
import csv
import datetime
import time

import click

from mpfs.config import settings


@click.command('get_period_user_stats', help=u"Собрать отчет по абонентской базе по пользователю за период. Подробнее здесь https://st.yandex-team.ru/CHEMODAN-43903 п.1")
@click.argument('begin_date_str')
@click.argument('end_date_str')
def get_period_user_stats(begin_date_str, end_date_str):
    import yt.wrapper as yt
    from nile.api.v1 import clusters
    from mpfs.core.mrstat.stat_utils import set_yt_proxy

    def aggregate_period_statistics(yt_table_paths):
        info_by_uid_pid = {}
        for row in yt.read_table(yt_table_paths, format='json'):
            key = (row['uid'], row['pid'])
            if key not in info_by_uid_pid:
                # Если за период не было активаций услуги, то в качестве первого дня активации надо указать последний
                # день аткивации на момент начала периода. Это значение храним в first_activation_timestamp. Если по
                # ходу прохода по периодам не встретим ни одной ктивации, то будем использовать это значение
                data = {
                    'active_days': 0,
                    'first_activation_timestamp': row.get('last_activation_timestamp'),
                    'period_first_activation_timestamp': None,
                    'period_last_deactivation_timestamp': row.get('last_deactivation_timestamp'),
                    'last_status': row.get('last_status'),
                }
                info_by_uid_pid[key] = data

            activation_timestamp = row.get('last_activation_timestamp')
            if (activation_timestamp != info_by_uid_pid[key]['first_activation_timestamp']
                    and info_by_uid_pid[key]['period_first_activation_timestamp'] is None):
                info_by_uid_pid[key]['period_first_activation_timestamp'] = activation_timestamp

            info_by_uid_pid[key]['period_last_deactivation_timestamp'] = row.get('last_deactivation_timestamp')
            info_by_uid_pid[key]['last_status'] = row.get('last_status')

            if row.get('active') or row.get('activate_deactivate'):
                info_by_uid_pid[key]['active_days'] += 1

        for k in info_by_uid_pid.iterkeys():
            info_by_uid_pid[k]['period_first_activation_timestamp'] = (
                    info_by_uid_pid[k].get('period_first_activation_timestamp')
                    or info_by_uid_pid[k].get('first_activation_timestamp'))
        return info_by_uid_pid

    def update_info_with_last_actions(yt_table, info_by_uid_pid):
        for row in yt.read_table(yt_table, format='json'):
            key = (row['uid'], row['pid'])
            if key not in info_by_uid_pid:
                continue
            info_by_uid_pid[key]['last_action_timestamp'] = row.get('action_timestamp')
            info_by_uid_pid[key]['last_action'] = row.get('action')
            info_by_uid_pid[key]['rostelecom_uid'] = row.get('rostelecom_uid')
        return info_by_uid_pid

    set_yt_proxy()
    # в отчетах rostelecom_active_services по дате yyyy-mm-dd лежит статистика за предыдущий день, поэтому прибавляем
    # один день
    begin_date = datetime.datetime.strptime(begin_date_str, '%Y-%m-%d').date() + datetime.timedelta(days=1)
    end_date = datetime.datetime.strptime(end_date_str, '%Y-%m-%d').date() + datetime.timedelta(days=1)
    table_paths = '//home/mpfs-stat/storage/rostelecom_active_services/{%s..%s}' % (begin_date, end_date)
    tmp_table_path = '//tmp/mpfs-get-period-user-stats-%s' % time.time()
    last_stat_table_path = '//home/mpfs-stat/storage/rostelecom_active_services/%s' % datetime.date.today()

    cluster = clusters.Hahn(settings.mrstat['yt_token'])
    job = cluster.job()
    job.table(table_paths) \
        .sort('timestamp') \
        .put(tmp_table_path)
    job.run()

    info_by_uid_pid = aggregate_period_statistics(tmp_table_path)
    info_by_uid_pid = update_info_with_last_actions(last_stat_table_path, info_by_uid_pid)

    with open('get_period_user_stats_result.csv', 'wb') as f:
        csv_writer = csv.writer(f)
        csv_writer.writerow(
            ['RTK_id', 'uid', 'product_name', 'active_days', 'first_activation_timestamp',
             'last_deactivation_timestamp', 'last_action', 'last_action_timestamp', 'last_status'])
        for k, v in info_by_uid_pid.iteritems():
            csv_writer.writerow(
                [v.get('rostelecom_uid'), k[0], k[1], v.get('active_days'), v.get('period_first_activation_timestamp'),
                 v.get('period_last_deactivation_timestamp'), v.get('last_action'), v.get('last_action_timestamp'),
                 v.get('last_status')])
