# -*- coding: utf-8 -*-
import datetime
import time

from mpfs.core.billing import Product
from mpfs.core.billing.processing.common import simple_delete_service, simple_create_service
from mpfs.core.metastorage.control import disk_info
from mpfs.common import errors
from mpfs.core.rostelecom_unlim.constants import (
    ROSTELECOM_PID_TO_SERVICE_KEY,
    ROSTELECOM_UNLIM_3,
    ROSTELECOM_SUBSTITUTE_PRODUCTS,
    ROSTELECOM_UNLIM_3_TEST
)
from mpfs.core.rostelecom_unlim.errors import RostelecomUnlimFrozenServiceNotFound


class FreezeManager(object):
    DISK_INFO_PATH = '/rostelecom_frozen/'

    @classmethod
    def freeze_service(cls, client, service):
        rostelecom_service_key = ROSTELECOM_PID_TO_SERVICE_KEY.get(service.pid, service.pid)
        key = cls.DISK_INFO_PATH + rostelecom_service_key

        try:
            disk_info.make_folder(client.uid, cls.DISK_INFO_PATH, '')
        except errors.StorageFolderAlreadyExist:
            pass

        data = {}
        if rostelecom_service_key in ROSTELECOM_SUBSTITUTE_PRODUCTS:
            current_timestamp = int(time.time())
            freeze_day_is_active = datetime.datetime.fromtimestamp(current_timestamp).date() \
                == datetime.datetime.fromtimestamp(service.ctime).date()
            data = {
                'pid': service.pid,
                'prev_btime': service.btime,
                'freeze_time': current_timestamp,
                'freeze_day_is_active': freeze_day_is_active,
            }
            if service.pid in (ROSTELECOM_UNLIM_3, ROSTELECOM_UNLIM_3_TEST):
                data['size'] = int(service.product.attributes.amount)
        disk_info.put(client.uid, key, data)
        simple_delete_service(client, service, send_email=False)

    @classmethod
    def unfreeze_service(cls, client, rostelecom_service_key):
        key = cls.DISK_INFO_PATH + rostelecom_service_key
        fetched_info = disk_info.find_one_by_field(client.uid, {'key': key})
        if not fetched_info:
            raise RostelecomUnlimFrozenServiceNotFound()

        if rostelecom_service_key in ROSTELECOM_SUBSTITUTE_PRODUCTS:
            pid = fetched_info['data']['pid']

            inactive_days = (datetime.date.today()
                             - datetime.datetime.fromtimestamp(fetched_info['data']['freeze_time']).date())
            if fetched_info['data']['freeze_day_is_active'] and inactive_days:
                # Если день заморозки был активным, и при этом текущая разморозка происходит не в тот же день, что и
                # предыдущая заморозка. то декрементим кол-во нактивных дней
                inactive_days = inactive_days - datetime.timedelta(days=1)
            new_btime = int(fetched_info['data']['prev_btime'] + inactive_days.total_seconds())
            attributes = {'service.btime': new_btime}

            if pid in (ROSTELECOM_UNLIM_3, ROSTELECOM_UNLIM_3_TEST):
                attributes['product.amount'] = int(fetched_info['data']['size'])
        else:
            pid = rostelecom_service_key
            attributes = None

        product = Product(pid)
        unfrozen_service = simple_create_service(client, product, attributes=attributes)
        disk_info.remove(client.uid, key)
        return unfrozen_service

    @classmethod
    def is_service_frozen(cls, uid, rostelecom_service_key):
        return bool(cls.get_frozen_service_info(uid, rostelecom_service_key))

    @classmethod
    def get_frozen_service_info(cls, uid, rostelecom_service_key):
        key = cls.DISK_INFO_PATH + rostelecom_service_key
        fetched_info = disk_info.find_one_by_field(uid, {'key': key})
        return fetched_info

    @classmethod
    def remove(cls, uid, rostelecom_service_key):
        key = cls.DISK_INFO_PATH + rostelecom_service_key
        disk_info.remove(uid, key)
