# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.common.static.tags.billing import *
from mpfs.core.billing.service import ServiceList
from mpfs.core.rostelecom_unlim import constants
from mpfs.core.rostelecom_unlim import logic
from mpfs.core.rostelecom_unlim.errors import RostelecomIncorrectNotFound

default_log = mpfs.engine.process.get_default_log()


def _get_services_by_parent_service_key(client, service_key):
    pids = constants.ROSTELECOM_SUBPRODUCTS.get(service_key, [service_key])
    return ServiceList(**{CLIENT: client, PRODUCT: pids})


def rostelecom_activate(req):
    """
    Активирует услугу для Ростелекома: https://wiki.yandex-team.ru/disk/mpfs/meps/mep-039/#novyeprodukty

    Можно активировать услуги с service_key=rostelecom_unlim и rostelecom_vas_* (см. список ROSTELECOM_SERVICE_KEYS)
    Для запроса с service_key=rostelecom_unlim активируется услуга с service_key=rostelecom_unlim_1
    Если услуга уже создана, будет ошибка.
    Если услуга не из списка ROSTELECOM_SERVICE_KEYS, будет ошибка.

    Query string аргументы:
    * uid [обязательный] - uid пользователя
    * service_key [обязательный] - id продукта из списка ROSTELECOM_SERVICE_KEYS
    * rostelecom_uid [необязательный] - id пользователя в системе РТК. Нужен для составления отчетов. В будущем будет
      сделан обязательным
    """
    if req.service_key not in constants.ROSTELECOM_SERVICE_KEYS:
        raise RostelecomIncorrectNotFound()
    return logic.rostelecom_activate(req.uid, req.service_key, req.rostelecom_uid)


def rostelecom_deactivate(req):
    """
    Деактивирует услугу для Ростелекома: https://wiki.yandex-team.ru/disk/mpfs/meps/mep-039/#novyeprodukty

    Можно деактивировать активные и замороженные услуги с service_key=rostelecom_unlim и rostelecom_vas_*
    (см. список ROSTELECOM_SERVICE_KEYS)
    Для запроса с service_key=rostelecom_unlim деактивируется любая из услуг с service_key=rostelecom_unlim_*
    Если услуга не активна, будет ошибка.
    Если услуга не из списка ROSTELECOM_SERVICE_KEYS, будет ошибка.
    Если для service_key=rostelecom_unlim находится больше одной услуги, то все они будут деактивированы и в default-лог
    будет добавлена запись.

    Query string аргументы:
    * uid [обязательный] - uid пользователя
    * service_key [обязательный] - id продукта из списка ROSTELECOM_SERVICE_KEYS
    * rostelecom_uid [необязательный] - id пользователя в системе РТК. Нужен для составления отчетов. В будущем будет
      сделан обязательным
    """
    if req.service_key not in constants.ROSTELECOM_SERVICE_KEYS:
        raise RostelecomIncorrectNotFound()
    return logic.rostelecom_deactivate(req.uid, req.service_key, req.rostelecom_uid)


def rostelecom_unfreeze(req):
    """
    Разморжаивает услугу.

    Разморозить можно только замороженные услуги. При разморозке выставляется следующее время апгрейда услуги
    (опеделяется btime'ом)

    Query string аргументы:
    * uid [обязательный] - uid пользователя
    * service_key [обязательный] - id продукта из списка ROSTELECOM_SERVICE_KEYS
    * rostelecom_uid [необязательный] - id пользователя в системе РТК. Нужен для составления отчетов. В будущем будет
      сделан обязательным
    """
    if req.service_key not in constants.ROSTELECOM_SERVICE_KEYS:
        raise RostelecomIncorrectNotFound()
    return logic.rostelecom_unfreeze(req.uid, req.service_key, req.rostelecom_uid)


def rostelecom_freeze(req):
    """
    Заморжаивает услугу.

    Заморозить можно только активные услуги.

    Query string аргументы:
    * uid [обязательный] - uid пользователя
    * service_key [обязательный] - id продукта из списка ROSTELECOM_SERVICE_KEYS
    * rostelecom_uid [необязательный] - id пользователя в системе РТК. Нужен для составления отчетов. В будущем будет
      сделан обязательным
    """
    if req.service_key not in constants.ROSTELECOM_SERVICE_KEYS:
        raise RostelecomIncorrectNotFound()
    return logic.rostelecom_freeze(req.uid, req.service_key, req.rostelecom_uid)


def rostelecom_list_services(req):
    """
    Показать статус всех ростелекомовских услуг

    Query string аргументы:
    * uid [обязательный] - uid пользователя
    """
    return logic.rostelecom_list_services(req.uid)
