# -*- coding: utf-8 -*-
import mpfs.engine.process
from mpfs.common.static import SPACE_1TB

from mpfs.common.static.tags.billing import *
from mpfs.core.billing import api
from mpfs.core.billing.client import Client
from mpfs.core.billing.processing.common import simple_delete_service, calculate_billing_time
from mpfs.core.billing.service import Service, ServiceList
from mpfs.core.rostelecom_unlim import constants
from mpfs.core.rostelecom_unlim import errors
from mpfs.core.rostelecom_unlim.freeze_manager import FreezeManager


default_log = mpfs.engine.process.get_default_log()


def rostelecom_activate(uid, rostelecom_service_key, rostelecom_uid=None):
    if _get_services_by_parent_service_key(Client(uid), rostelecom_service_key):
        raise errors.RostelecomUnlimServiceIsActivated()
    if FreezeManager.is_service_frozen(uid, rostelecom_service_key):
        raise errors.RostelecomUnlimServiceIsBlocked()
    effective_pid = constants.ROSTELECOM_SUBSTITUTE_PRODUCTS.get(rostelecom_service_key, rostelecom_service_key)
    service = api.service_create(uid, effective_pid, ROSTELECOM_UNLIM)
    _log_transaction_results(uid=uid, pid=effective_pid, action='activate', status_before='inactive',
                             status_after='active', rostelecom_uid=rostelecom_uid)
    return {SID: service.sid, BTIME: service.btime}


def rostelecom_deactivate(uid, rostelecom_service_key, rostelecom_uid=None):
    client = Client(uid)
    services_to_delete = _get_services_by_parent_service_key(client, rostelecom_service_key)
    if not services_to_delete:
        frozen_service_info = FreezeManager.get_frozen_service_info(client.uid, rostelecom_service_key)
        if not frozen_service_info:
            raise errors.RostelecomUnlimServiceIdDeactivated()
        FreezeManager.remove(client.uid, rostelecom_service_key)
        _log_transaction_results(uid=uid, pid=frozen_service_info['data'].get(PID, rostelecom_service_key),
                                 action='deactivate', status_before='frozen', status_after='inactive',
                                 rostelecom_uid=rostelecom_uid)
        return

    if len(services_to_delete) > 1:
        log_services = ', '.join('%s:%s' % (service[PID], service[SID]) for service in services_to_delete)
        default_log.info('Found more than one services for service_key=%s: %s' % (rostelecom_service_key, log_services))
    for service in services_to_delete:
        simple_delete_service(client, Service(service[SID]))
    _log_transaction_results(uid=uid, pid=services_to_delete[0][PID], action='deactivate', status_before='active',
                             status_after='inactive', rostelecom_uid=rostelecom_uid)


def rostelecom_unfreeze(uid, rostelecom_service_key, rostelecom_uid=None):
    client = Client(uid)
    if _get_services_by_parent_service_key(client, rostelecom_service_key):
        raise errors.RostelecomUnlimServiceIsActivated()

    try:
        service = FreezeManager.unfreeze_service(client, rostelecom_service_key)
    except errors.RostelecomUnlimFrozenServiceNotFound:
        raise errors.RostelecomUnlimServiceIdDeactivated()
    _log_transaction_results(uid=uid, pid=service.pid, action='unfreeze', status_before='frozen',
                             status_after='active', rostelecom_uid=rostelecom_uid)


def rostelecom_freeze(uid, rostelecom_service_key, rostelecom_uid=None):
    client = Client(uid)
    found_services_dicts = sorted(_get_services_by_parent_service_key(client, rostelecom_service_key),
                                  key=lambda x: x[PRODUCT].attributes.amount, reverse=True)
    if len(found_services_dicts) > 1:
        log_services = ', '.join('%s:%s' % (service[PID], service[SID]) for service in found_services_dicts)
        default_log.info('Found more than one services for service_key=%s: %s' % (rostelecom_service_key, log_services))
        for service in found_services_dicts[1:]:
            simple_delete_service(client, Service(service[SID]))
    elif not found_services_dicts:
        if FreezeManager.is_service_frozen(uid, rostelecom_service_key):
            raise errors.RostelecomUnlimServiceIsBlocked()
        raise errors.RostelecomUnlimServiceIdDeactivated()

    FreezeManager.freeze_service(client, Service(found_services_dicts[0][SID]))
    _log_transaction_results(uid=uid, pid=found_services_dicts[0][PID], action='freeze', status_before='active',
                             status_after='frozen', rostelecom_uid=rostelecom_uid)


def process_billing_for_rostelecom_unlim_service(client, service):
    if service.pid in constants.ROSTELECOM_UPGRADE_MAP:
        new_pid = constants.ROSTELECOM_UPGRADE_MAP[service.pid]
        new_service = api.service_create(client.uid, new_pid, ROSTELECOM_UNLIM, send_email=False)
        simple_delete_service(client, service, send_email=False)
        default_log.info('Rostelecom service upgraded old_sid=%s old_pid=%s new_sid=%s new_pid=%s uid=%s' %
                         (service.sid, service.pid, new_service.sid, new_pid, client.uid))
        return
    service.set_attribute('product.amount', int(service.product.attributes.amount) + SPACE_1TB)
    btime, lbtime = calculate_billing_time(service, service.btime)
    service.set(BTIME, btime)
    service.set(LASTBTIME, lbtime)


def rostelecom_list_services(uid):
    client = Client(uid)
    service_statuses = {}
    for service_key in constants.ROSTELECOM_SERVICE_KEYS:
        for pid in constants.ROSTELECOM_SUBPRODUCTS.get(service_key, [service_key]):
            service_statuses[pid] = 'inactive'

    for service_key in constants.ROSTELECOM_SERVICE_KEYS:
        frozen_service_info = FreezeManager.get_frozen_service_info(uid, service_key)
        if frozen_service_info:
            service_statuses[frozen_service_info['data'].get(PID, service_key)] = 'blocked'

    for pid in service_statuses.keys():
        active_services = ServiceList(**{CLIENT: client, PRODUCT: pid})
        if active_services:
            service_statuses[pid] = 'active'

    return service_statuses


def _get_services_by_parent_service_key(client, service_key):
    pids = constants.ROSTELECOM_SUBPRODUCTS.get(service_key, [service_key])
    return ServiceList(**{CLIENT: client, PRODUCT: pids})


def _log_transaction_results(uid, pid, action, status_before, status_after, rostelecom_uid):
    log_data = {
        'uid': uid,
        'pid': pid,
        'action': action,
        'status_before': status_before,
        'status_after': status_after,
        'rostelecom_uid': rostelecom_uid
    }
    default_log.info('rostelecom_unlim: status_before=%(status_before)s'
                     ' status_after=%(status_after)s'
                     ' action=%(action)s'
                     ' uid=%(uid)s'
                     ' rostelecom_uid=%(rostelecom_uid)s'
                     ' pid=%(pid)s' % log_data)
