# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис для работы с адресной книгой

"""
import urlparse

import mpfs.engine.process

from mpfs.common.util import from_json
from mpfs.core.services.common_service import Service
from mpfs.core.services.passport_service import Passport
from mpfs.core.services.directory_service import DirectoryService


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log('abook')

passport = Passport()


class AbookService(Service):
    name = 'abook'
    log = service_log

    list_parameters = 'v1/searchContacts?uid=%(uid)s&format=jsonx'
    list_pdd_parameters = 'v1/searchContacts?uid=%(uid)s&mixin=%(mixin)s&format=jsonx&sort-fields=-ws_type,sortname'
    add_email_parameters = 'compat/colabook_feed_addrdb?to=%(email)s&uid=%(uid)s'

    def get_contacts(self, uid, append_groups=False):
        """
        Выбираем контакты из abook

        Он отвечает примерно так:
        {
            "contact": [
                {
                    "cid": 2,
                    "email": "mervekrks69@gmail.com",
                    "id": 1,
                    "last_usage": 1398155252,
                    "mcid": 2,
                    "name": {
                        "first": "Merve",
                        "full": "Merve Karaka\u015f",
                        "last": "Karaka\u015f"
                    },
                    "phone": [],
                    "tag": [],
                    "tags": [],
                    "usage_count": 6
                },
            ],
            "count": 3,
            "pager": {
                "items-count": 3
            }
        }


        Наружу мы выдаем так:
        [
            {
                'name': 'Merve Karaka\u015f',
                'email': 'mervekrks69@gmail.com,
                'avatar': '',
                'locale': '',
            }
        ]

        :param uid: uid юзера
        :return: итератор по контактам
        """
        userinfo = passport.userinfo(uid)

        args = {'uid': uid}

        if userinfo['pdd'] and userinfo['pdd_domain']:
            args['mixin'] = userinfo['pdd_domain']
            parameters = self.list_pdd_parameters % args
        else:
            parameters = self.list_parameters % args

        url = self.base_url + parameters
        response = self.open_url(url)

        contacts = []
        contact_hashes = set()
        for each in from_json(response).get('contact', []):
            contact, contact_hash = self.process_contact(each, append_groups)
            if contact is None:
                continue
            if contact_hash in contact_hashes:
                continue
            contact_hashes.add(contact_hash)
            contacts.append(contact)
        return contacts

    def process_contact(self, entry, append_groups):
        name = entry.get('name', {}).get('full', '')
        email = entry.get('email', '')

        contact_id_name, contact_id = 'userid', email
        contact_hash = name + email

        contact = {
            'name': '',
            'avatar': '',
            'locale': '',
        }

        if 'ya_directory' in entry:
            if 'type' not in entry['ya_directory']:
                return None, None
            contact_type = entry['ya_directory']['type']

            if contact_type not in ('department', 'group', 'user'):
                return None, None

            if append_groups and (contact_type == 'department' or contact_type == 'group'):
                directory_id = entry['ya_directory']['id']
                org_id = entry['ya_directory']['org_id']
                if entry['ya_directory'].get('org_name'):
                    name = DirectoryService.format_group_name(entry['ya_directory']['org_name'],
                                                              entry.get('name', {}).get('first'))
                contact_id_name = 'groupid'
                contact_id = DirectoryService.encode_group_entity(org_id, contact_type, directory_id)
                contact_hash = str(org_id) + contact_type + str(directory_id)

            contact['avatar'] = self._get_avatar_url(entry.get('photo_partial_url', None))

            self._append_ya_directory_data(contact, contact_type, entry)

        contact['name'] = name
        contact[contact_id_name] = contact_id
        return contact, contact_hash

    @staticmethod
    def _get_avatar_url(photo_partial_url):
        """
        Превращает частичный url от abook'а в полноценный url с размером и схемой

        :param photo_partial_url: частичный url от abook, например "//yapic.yandex.ru/get/1130000000213987/"
        :return: полный url, например "https://yapic.yandex.ru/get/1130000000213987/islands-200"
        """
        if photo_partial_url is None:
            return ''

        parsed_url = urlparse.urlparse(photo_partial_url)

        path = parsed_url.path
        if not path.endswith('/'):
            path += '/'
        path += 'islands-200'

        url_parts = urlparse.ParseResult(scheme='https', netloc=parsed_url.netloc, path=path,
                                         params='', query='', fragment='')
        return url_parts.geturl()

    @staticmethod
    def _append_ya_directory_data(contact, contact_type, entry):
        contact['type'] = contact_type
        contact['is_ya_directory'] = True
        if 'email' in entry:
            contact['email'] = entry['email']

        if contact_type == 'user':
            if 'title' in entry:
                contact['title'] = entry['title']
            if 'department' in entry:
                contact['department'] = entry['department']
        elif contact_type == 'group':
            if 'summary' in entry:
                contact['summary'] = entry['summary']
            if 'org_name' in entry['ya_directory']:
                contact['org_name'] = entry['ya_directory']['org_name']
            if entry.get('name', {}).get('first'):
                contact['org_group_name'] = entry.get('name', {}).get('first')
        elif contact_type == 'department':
            if 'department' in entry:
                contact['department'] = entry['department']
            if 'org_name' in entry['ya_directory']:
                contact['org_name'] = entry['ya_directory']['org_name']
            if entry.get('name', {}).get('first'):
                contact['org_group_name'] = entry.get('name', {}).get('first')

    def add_email(self, uid, email):
        args = {
            'uid': uid,
            'email': email,
        }
        url = self.base_url + self.add_email_parameters % args
        self.open_url(url)
