# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис для работы с aviary

"""
import hashlib
import random
import time
import urllib

from mpfs.common import errors
from mpfs.core.bus import Bus
from mpfs.core.services.kladun_service import ExtractRotation

import mpfs.engine.process
from mpfs.engine.http import client

from mpfs.common.util import to_json, from_json
from mpfs.core.services.common_service import Service

from mpfs.core.user.constants import PUBLIC_UID

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log('aviary')

JOB_CREATED_CODE = 1


class Aviary(Service):

    name = 'aviary'
    api_error = errors.AviaryNoResponse
    log = service_log

    def __init__(self, *args, **kwargs):
        super(Aviary, self).__init__(*args, **kwargs)
        self.request = None
        for k, v in kwargs.iteritems():
            setattr(self, k, v)
        self.fs = Bus(request=self.request)

    def _get_image_rotation(self, uid, file_mid):
        service = ExtractRotation()
        exif = service.post_request({
            'uid': uid,
            'oid': None,
            'mulca-id': file_mid
        })
        return exif['rotation']

    def get_job_status(self, aviary_job_id):
        parameters = self.get_common_paramenters()
        parameters.update({'jobid': aviary_job_id})
        url = self.base_url + "/render?" + urllib.urlencode(parameters)

        response = client.open_url(url, log=service_log, timeout=self.timeout,
                                   api_err=self.api_error, method="GET")

        self.log.info(response)

        if response is None:
            raise errors.AviaryNoResponse()

        try:
            response = from_json(response)
        except Exception:
            raise errors.AviaryBadResponse()

        if response.get('jobStatus') is None:
            raise errors.AviaryBadResponse()

        return response

    def render(self, resource, actionlist, override_url=None):
        parameters = self.get_aviary_render_parameters(resource, actionlist, override_url=override_url)
        url = self.base_url + "/render"

        response = client.open_url(url, log=service_log, url_data=parameters, timeout=self.timeout, api_err=self.api_error, method="POST")

        self.log.info(response)

        if response is None:
            raise errors.AviaryNoResponse()

        try:
            response = from_json(response)
        except Exception:
            raise errors.AviaryBadResponse()

        if int(response.get('code')) != JOB_CREATED_CODE:
            raise errors.AviaryBadResponse()

        return response.get('id')

    def get_common_paramenters(self):
        salt = hashlib.md5(str(random.random()) + str(random.random())).hexdigest()
        timestamp = int(time.time())
        signature = hashlib.sha1(self.api_key + self.api_secret + str(timestamp) + salt).hexdigest()

        return {
            'apikey': self.api_key,
            'timestamp': timestamp,
            'salt': salt,
            'encryptionmethod': 'sha1',
            'signature': signature,
        }

    def get_aviary_render_parameters(self, resource, actionlist, override_url=None):
        file_mid = resource.info()['this']['meta']['file_mid']
        rotation = self._get_image_rotation(resource.address.uid, file_mid)

        if rotation != 0:
            actionlist_json = from_json(actionlist)

            actionlist_json['actionlist'] = [{
                    "action": "rotate90",
                    "angle": rotation,
                    "flatten": True
                }] + actionlist_json['actionlist'] + [{
                    "action": "rotate90",
                    "angle": -rotation,
                    "flatten": True
                }]

            actionlist = to_json(actionlist_json)

        parameters = self.get_common_paramenters()

        parameters['fileformat'] = 'jpeg'
        parameters['actionlist'] = actionlist
        if self.url_override_allowed and override_url is not None:
            parameters['origurl'] = override_url
        else:
            index = resource.name.rfind('.')
            if index < 0:
                name = 'filename_for_aviary'
            else:
                name = 'filename_for_aviary' + resource.name[index:]
            parameters['origurl'] = resource.get_url_with_hash(override_uid=PUBLIC_UID, override_name=name, hash='for_aviary', expire_seconds=self.expire_seconds)['file']

        return parameters
