# -*- coding: utf-8 -*-
"""
Код для работы с onetime task на базинге
DEPRECATED: https://st.yandex-team.ru/CHEMODAN-36976, Для запуска задач на mworker используй queller_service

Как создать новый "onetime-task"
1) Создать класс задачи отнаследовавшись от OnetimeTask
2) Создать к этому классу CLI и положить его в папку со скриптами
3) Создать конфиг для базинги см. документацию - https://wiki.yandex-team.ru/disk/worker/#kakbystrodobavitneskolkoodnorazovyxzadachvvorker
4) Положить конфиг к другим конфигам onetime task-ов

Как использовать его из кода:
Если нужно разово(не пачками) ставить задачи, то используем OnetimeTask.insert_to_bazinga
Если надо ставить много задач одновременно, то:
1) Объеденяем объекты-задачи в массив(tasks)
2) Ставим пачкой задачи в базингу: BazingaInterface().bulk_create_tasks(tasks)
"""
import urllib
import json

from abc import ABCMeta, abstractmethod
from enum import Enum

import mpfs.engine.process

from mpfs.common.util import from_json, chunks2
from mpfs.core.services.common_service import Service


service_log = mpfs.engine.process.get_service_log('bazinga')


class TaskStatus(Enum):
    READY = 'ready'
    STARTING = 'starting'
    RUNNING = 'running'
    COMPLETED = 'completed'
    FAILED = 'failed'

    @classmethod
    def not_finished(cls):
        return [cls.READY, cls.RUNNING, cls.STARTING]


class BazingaInterface(Service):
    """Работа с базингой через HTTP API.

    https://wiki.yandex-team.ru/disk/worker/#ruchkipgtasksactioncontainer
    """
    name = 'bazinga'
    log = service_log
    MAX_BATCH_SIZE = 10000

    def create_task(self, task_obj):
        """Создать один таск в базингу

        :param OnetimeTask task_obj: задача, которую хотим выполнить на базинге
        """
        tasks_ids = self.bulk_create_tasks([task_obj])
        if tasks_ids:
            return tasks_ids[0]
        else:
            return None

    def bulk_create_tasks(self, tasks_obj):
        """Создание пачки тасков в базинге

        https://wiki.yandex-team.ru/disk/worker/#zapustitkuchuodnorazovyxzadach

        :param list tasks_obj: список с объектами OnetimeTask
        """
        result = []
        for tasks_chunk in chunks2(tasks_obj, chunk_size=self.MAX_BATCH_SIZE):
            result.extend(self._bulk_create_tasks(tasks_chunk))
        return result

    def _bulk_create_tasks(self, tasks_obj):
        if not tasks_obj:
            return []

        body = {
            "tasks": [
            ]
        }
        for task in tasks_obj:
            if not isinstance(task, OnetimeTask):
                raise TypeError('"OnetimeTask" objects required. Got: %r' % task)
            command_parameters = task.build_command_parameters()
            if not isinstance(command_parameters, list):
                raise TypeError('`build_command_parameters` must return `list`. Got: %r' % command_parameters)

            task = {
                "taskId": task.BAZINGA_TASK_NAME,
                "parameters":  json.dumps({'commandParameters': command_parameters})
            }
            body['tasks'].append(task)
        body = json.dumps(body)

        url = "%s%s" % (self.base_url, '/tasks/add-tasks')

        response = self.open_url(url, pure_data=body, method='POST', headers={"Content-type": "application/json"})
        resp_json = from_json(response)

        if 'result' in resp_json:
            return resp_json['result']['jobsIds']
        else:
            raise self.api_error(response)

    def get_tasks_count(self, task_cls, statuses=None):
        """Получение количества тасков в базинге

        https://wiki.yandex-team.ru/disk/worker/#poluchitkolichestvozapuskovodnorazovyxzadachpostatusu

        :param type task_cls: наследник OnetimeTask
        :param (list|TaskStatus) statuses: статус или массив статусов таска для фильтрации по ним. None - любой статус
        """
        if not issubclass(task_cls, OnetimeTask):
            raise TypeError('"OnetimeTask" subclass required. Got %r' % task_cls)
        statuses = self._validate_statuses(statuses)

        params = {
            'task': task_cls.BAZINGA_TASK_NAME,
            'status': [s.value for s in statuses],
        }
        query_string = urllib.urlencode(params, doseq=True)
        url = "%s%s?%s" % (self.base_url, '/tasks/get-tasks-count', query_string)

        response = self.open_url(url)
        resp_json = from_json(response)

        if 'result' in resp_json:
            return int(resp_json['result']['count'])
        else:
            raise self.api_error(response)

    def _validate_statuses(self, statuses):
        if statuses is None:
            return list(TaskStatus)
        elif isinstance(statuses, TaskStatus):
            statuses = [statuses]
        elif isinstance(statuses, (list, tuple)):
            for status in statuses:
                if not isinstance(status, TaskStatus):
                    raise TypeError('Expected `TaskStatus`. Got: "%s"' % status)
        else:
            raise TypeError('"statuses" should be `list` or `TaskStatus` type')
        return statuses


class OnetimeTask(object):
    """
    Абстрактный класс задач в базинге

    Правила для классов-реализаций:
        1. В конструктор передаются параметры запуска(ARGV CLI).
        2. В конструкторе не должно быть "тяжелых" действий.
        3. CLI к этому классу должен инициализировать объект и дергать метод run.
        4. То, что будет выполнятся на базинге кладем в метод "run".
    """
    __metaclass__ = ABCMeta

    BAZINGA_TASK_NAME = None
    """ID задачи в json конфиге. Обязательно переопределяем!"""
    _bazinga_interface = BazingaInterface()

    def __init__(self, *args, **kwargs):
        if not isinstance(self.BAZINGA_TASK_NAME, basestring):
            raise ValueError('"BAZINGA_TASK_NAME" must be string')
        super(OnetimeTask, self).__init__(*args, **kwargs)

    @abstractmethod
    def run(self):
        pass

    @abstractmethod
    def build_command_parameters(self):
        """
        Генерирует массив параметров, которые будут переданы базинге для запуска скрипта

        Пример:
            Хотим запустить такое:
            > /usr/sbin/mpfs-admin-storage-cleaner-worker.py ThreadingCleaningWorker 1234
            Функция должна отдать:
            > ['ThreadingCleaningWorker', '1234']
        """
        pass

    def insert_to_bazinga(self):
        """Поставить задачу(себя) в базингу"""
        return self._bazinga_interface.create_task(self)

    @classmethod
    def queue_len(cls):
        """Получение длины очереди задачи"""
        return cls._bazinga_interface.get_tasks_count(cls, TaskStatus.READY)

    @classmethod
    def not_finished_len(cls):
        """Число задач НЕ в терминальном состоянии"""
        return cls._bazinga_interface.get_tasks_count(cls, TaskStatus.not_finished())
