# -*- coding: utf-8 -*-
from types import NoneType

import mpfs.engine.process

from mpfs.common.util import from_json
from mpfs.common import errors
from mpfs.core.services.common_service import Service

service_log = mpfs.engine.process.get_service_log('big-brother')


class BigBrotherService(Service):
    """
    Сервис для общения с API Большого Брата
    """
    name = 'big_brother'
    api_error = errors.DataApiProfileBadResponse
    log = service_log
    base_url = None  # should be set in config
    timeout = None  # should be set in config

    def empty_regions(self):
        return {}

    def get_home_region_id(self, yandexuid):
        regions = self.get_regions(yandexuid)
        for item in regions.get('items', []):
            if item.get('slug', '') == 'home':
                return item.get('geo_id')
        return None

    def get_raw_data(self, yandexuid):
        headers = {
            'Cookie': 'yandexuid=' + str(yandexuid)
        }
        return self.open_url(self.base_url, method='GET', headers=headers, retry=False, return_status=True)

    def get_data(self, yandexuid):
        status_code, resp, headers = self.get_raw_data(yandexuid)
        if resp and not isinstance(resp, (dict, list, NoneType)):
            resp = from_json(resp)

        if not isinstance(resp, dict):
            resp = dict()

        return resp.get('data', [])

    def get_data_by_id(self, yandexuid, id):
        data = []
        for element in self.get_data(yandexuid):
            for segment in element.get('segment', []):
                if segment.get('id', '') == id:
                    data.append(segment.get('value'))
        return data

    def get_regions(self, yandexuid):
        """returns list of user regions in format

        {
            'geo_id': integer
            'slug': string, optional
        }
        """

        # 349 - id of segment with geo information
        for value in self.get_data_by_id(yandexuid, "349"):
            return self.parse_geo_value(value)

        return self.empty_regions()

    def parse_geo_id_weight(self, part):
        """parse string "geo_id:weight" to tuple (weight, geo_id)"""
        geo_id, weight = part.split(":")
        return int(weight), int(geo_id)

    def parse_geo_value(self, value):
        """parse value in format "geo_id1:weight1,geo_id2:weight2,geo_id3:weight3" """
        parts = value.split(",")
        regions = [self.parse_geo_id_weight(part) for part in parts]
        regions.sort(reverse=True)

        if len(regions) == 0:
            return self.empty_regions()

        items = [
            {
                'slug': 'home',
                'geo_id': regions[0][1]
            }
        ]

        # not more than 10 regions total
        for region in regions[1:10]:
            items += [
                {
                    'geo_id': region[1]
                }
            ]

        return {
            'flags': [],
            'items': items
        }

    BB_TO_APP_KEY_MAP = {
        "583": 1,
        "584": 2,
        "585": 3,
        "546": 10012,
        "547": 18313,
        "548": 18316,
        "549": 19531,
        "550": 19534,
        "551": 25204,
        "552": 24691,
        "553": 16807,
        "554": 18895,
        "555": 14836,
        "556": 4,
        "557": 28587,
        "558": 23107,
        "559": 23104,
        "560": 23101,
        "561": 23548,
        "562": 25378,
        "563": 10126,
        "564": 10129,
        "565": 10123,
        "566": 17914,
        "567": 25870,
        "568": 17569,
        "569": 22792,
        "570": 21925,
        "571": 28210,
        "572": 27952,
        "573": 9850,
        "574": 20740,
        "575": 9973,
        "576": 10180,
        "577": 10270,
        "578": 10267,
        "579": 22675,
        "580": 24616,
        "581": 10609,
        "582": 5746
    }

    APP_KEY_TO_NAME_MAP = {
        10012: "Афиша (Android)",
        18313: "Яндекс.Браузер Android Phone",
        18316: "Браузер Android Tablet",
        19531: "Яндекс.Браузер iOS Phone",
        19534: "Яндекс.Браузер iOS Tablet",
        25204: "Яндекс.Город Android",
        24691: "Яндекс.Город iOS",
        16807: "Yandex.Direct",
        18895: "Yandex.Disk",
        14836: "Yandex.Mail Android",
        4: "Yandex.Maps (Production)",
        28587: "Yandex.Maps Android",
        23107: "Yandex.Market Android",
        23104: "Yandex.Market IOS",
        23101: "Yandex.Market WP",
        23548: "Yandex.Master iOS",
        2: "Yandex.Metro (Production)",
        25378: "Яндекс.Музыка (NEW)",
        10126: "Яндекс.Музыка Android",
        10129: "Яндекс.Музыка WP",
        10123: "Яндекс.Музыка iOS",
        17914: "Yandex.News",
        25870: "Yandex.News Android",
        17569: "Yandex.Parking(Production)",
        22792: "Yandex.Shell",
        3: "Yandex.Taxi (Production)",
        21925: "Яндекс.Транспорт (Production)",
        1: "Yandex.Store",
        28210: "Yandex.Weather",
        27952: "Yandex.Trains Android",
        9850: "Yandex.Trains IOS",
        20740: "Yandex.Translate for Android",
        9973: "Yandex.Translate for WP",
        10180: "Yandex.Translate for iOS",
        10270: "УгадайКино! (Production)",
        10267: "КиноПоиск (Production)",
        22675: "library Account Manager (Production)",
        24616: "Яндекс.ЕГЭ (прод)",
        10609: "Яндекс.Медали (Production)",
        5746: "Yandex.Search for WP"
    }

    def parse_mobile_app_value(self, value):
        """parse from key in big brother to

        {
            'big_brother_id': integer
            'app_key': integer,
            'name': string
        }
        """
        if value not in self.BB_TO_APP_KEY_MAP:
            return {
                'big_brother_id': int(value)
            }
        else:
            app_key = self.BB_TO_APP_KEY_MAP[value]
            name = self.APP_KEY_TO_NAME_MAP[app_key]
            return {
                'big_brother_id': int(value),
                'app_key': app_key,
                'name': name
            }

    def get_yandex_mobile_applications(self, yandexuid):
        """returns list of mobile applications for given yandexuid in format

        {
            'big_brother_id': integer
            'app_key': integer,
            'name': string
        }
        """
        return [self.parse_mobile_app_value(value) for value in self.get_data_by_id(yandexuid, "319")]

big_brother = BigBrotherService()
