# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис сокращения урлов

"""
import re
from urlparse import urlsplit, urlunsplit, urljoin, urlparse, urlunparse

import mpfs.engine.process
import mpfs.common.errors as errors

from mpfs.core.services.common_service import Service
from mpfs.common.util import unquote_qs_param

service_log = mpfs.engine.process.get_service_log('clck')


class Clck(Service):
    name = 'clck'
    api_error = errors.ClckNoResponse
    log = service_log
    FULL_LINK_RE = re.compile(r'^https?://.+[\?&](hash|key)=(?P<public_key>[^&]+)')
    short_url_path_re = re.compile(r'^/(d|i|a|j)/[\w-]{10,16}$')

    url_types = {
        'default': 'yadisk',
        'image': 'yadisk-image',
        'album': 'yadisk-album',
        'telemost-join': 'telemost-join',
    }

    def generate(self, url, mode='default'):
        """
        Генерация урла
        """
        data = {
            'url': url,
            'type': 'yadisk'
        }

        if self.typed_urls:
            data['type'] = self.url_types.get(mode, 'yadisk')

        link = self.open_url(self.base_url, data)
        if not self.is_valid_short_url(link):
            raise ValueError("Bad short url %s" % link)
        chunks = link.split('/')
        _id = chunks.pop()
        return _id, link

    @classmethod
    def is_valid_short_url(cls, short_url):
        """Проверяет урлы получаемые от кликера

        в базе могут быть short_url другого формата
        """
        parsed_url = urlparse(short_url)
        # нет протокола, хоста или пути
        if not all((parsed_url.scheme, parsed_url.netloc, parsed_url.path)):
            return False
        if not cls.short_url_path_re.search(parsed_url.path):
            return False
        return True

    def resolve(self, short_url):
        """Алиас к short_url_to_full_url. Лучше не использовать"""
        return self.short_url_to_full_url(short_url)

    def short_url_to_full_url(self, short_url):
        """Резолвит короткую ссылку в длинную"""
        _, _, path, query, fragment = urlsplit(short_url)
        url = urlunsplit((None, None, path, query, fragment))
        url = urljoin(self.base_url, url)
        status, _, headers = self.open_url(url, follow_redirects=False, return_status=True)
        if status in (301, 302):
            for k, v in headers.iteritems():
                if k.lower() == 'location':
                    return v

    def full_url_to_public_hash(self, full_url):
        """Резолвит длинную ссылку в public_hash"""
        match = self.FULL_LINK_RE.match(full_url)
        if match:
            public_hash = match.group('public_key')
            # ключ скорее всего за urlencode'н поэтому ещё раз декодим
            return unquote_qs_param(public_hash)

    def _parse_short_url(self, short_url):
        sep = '/'
        parsed_url = urlparse(short_url)
        path = parsed_url.path or ''
        path_parts = path.split(sep, 3)
        if len(path_parts) <= 2:
            # https://yadi.sk/ or https://yadi.sk
            raise ValueError('Wrong URL format.')

        if len(path_parts) == 3:
            if not path_parts[-1]:
                # https://yadi.sk/d/
                raise ValueError('Wrong URL format.')

        # отрезаем относительный путь и резолвим короткую ссылку, добавляем относительный путь к полученному хешу
        if len(path_parts) == 3:
            # https://yadi.sk/d/EZiDUapAympaL
            empty_str, type_, hash_ = path_parts
            relative_path = ''
        else:
            # https://yadi.sk/d/EZiDUapAympaL/ or https://yadi.sk/d/EZiDUapAympaL/path/to/relative[/]
            empty_str, type_, hash_, relative_path = path_parts

        return parsed_url, sep, empty_str, type_, hash_, relative_path

    def short_url_to_public_hash(self, short_url):
        """
        Резолвит короткую ссылку в публичный хеш.

        Умеет работать с короткими ссылками с относительным путем (https://yadi.sk/d/EZiDUapAympaL/sub_folder.).
        """
        parsed_url, sep, empty_str, type_, hash_, relative_path = self._parse_short_url(short_url)
        parsed_url_lst = list(parsed_url)
        parsed_url_lst[2] = sep.join((empty_str, type_, hash_)) + sep
        full_url = self.short_url_to_full_url(urlunparse(parsed_url_lst))
        if full_url:
            public_hash = self.full_url_to_public_hash(full_url)
            if relative_path:
                public_hash += (':%s%s' % (sep, relative_path))
            return public_hash

    def check_short_url_is_album_url(self, short_url):
        """Проверить является ли короткая ссылка ссылкой на альбом."""
        return urlparse(short_url).path.startswith('/a/')


clck = Clck()
