# -*- coding: utf-8 -*-
"""

MPFS
CORE

Базовый сервис

"""
import re
import operator
import hashlib
import hmac
import urllib
import traceback
import sys
from urllib2 import HTTPError

import requests
import time

from collections import defaultdict

import mpfs.engine.process
import mpfs.common.errors as errors

from mpfs.common.util.logger import headers_to_log
from mpfs.common.util.natsort import natcasecmp
from mpfs.engine.process import get_error_log
from mpfs.engine.http import client as http_client
from mpfs.engine.xmlrpc import client as xmlrpc_client
from mpfs.config import settings

service_log = mpfs.engine.process.get_default_log()
requests_log = mpfs.engine.process.get_requests_log()

MAIL_RE = re.compile('([\w,\.\-]+@[\w,\.\-]+)\s*')

SERVICES_TVM_2_0_CLIENT_ID = settings.services['tvm_2_0']['client_id']
SERVICES_TVM_2_0_ENABLED = settings.services['tvm_2_0']['enabled']
FEATURE_TOGGLES_TVM_DAEMON_USAGE_ENABLED = settings.feature_toggles['tvm_daemon_usage_enabled']
SYSTEM_HTTP_RETRIES = settings.system['http_retries']


class RequestsPoweredServiceBase(object):
    REQUESTS_LOG_PATTERN = '%(method)s "%(url_message)s" %(response_code)s %(bytes_sent)s %(bytes_received)s %(elapsed_time).3f'
    static_headers = {}

    def __init__(self):
        self.base_url = None
        self.timeout = None
        self.connection_timeout = None
        self.http_retries = SYSTEM_HTTP_RETRIES
        self.send_tvm_ticket = None
        self.send_cloud_request_id = None
        self.tvm_2_0 = None
        self.recreate_connection_on_5xx = True
        self.recreate_connection_on_exception = True

        # Решаем проблему с получением ТВМ тикетов в тестинге через продовый ятимный BB
        # https://st.yandex-team.ru/CHEMODAN-38321#1508846667000
        self.get_tvm_ticket = True

        # number of pools (per host)
        # https://laike9m.com/blog/requests-secret-pool_connections-and-pool_maxsize,89/
        self.pool_connections = None

        # number of connections in pool (to same host)
        # https://laike9m.com/blog/requests-secret-pool_connections-and-pool_maxsize,89/
        self.pool_maxsize = None

        # a list of headers that won't be logged
        # lowercase
        self.not_logged_headers = None

        self.__apply_config('ServiceBase')
        self.__apply_config(self.__class__.__name__)

        self._initialize_session()

    def _initialize_session(self):
        self._session = requests.session()
        self._session.mount('http://', requests.adapters.HTTPAdapter(pool_connections=self.pool_connections, pool_maxsize=self.pool_maxsize))
        self._session.mount('https://', requests.adapters.HTTPAdapter(pool_connections=self.pool_connections, pool_maxsize=self.pool_maxsize))
        self._session.verify = False

    def __apply_config(self, name):
        config = settings.services.get(name, {})
        for attr, value in config.iteritems():
            if not hasattr(self, attr) or (attr == 'tvm_2_0' and not value.get('client_id')):
                raise Exception("Can't configure services.%s.%s in %s: no such attribute" % (name, attr, self.__class__.__name__))
            setattr(self, attr, value)

    def _exception_from_response(self, response):
        return HTTPError(response.request.url, response.status_code, response.content, None, None)

    def get_service_headers(self, headers, *args, **kwargs):
        return headers or {}

    def request(self, method, relative_url, params=None, data=None, headers=None, cookies=None, base_url=None,
                timeout=None, connection_timeout=None, recreate_connection_on_5xx=None,
                recreate_connection_on_exception=None, send_tvm_ticket=None, auth=None, http_retries=None):
        """Метод не из потомков использовать только по специальному разрешению @akinfold.

        :param str method: HTTP method
        :param str relative_url: часть после self.base_url
        :param str base_url: entry point (default: `self.base_url`)
        :param dict params: query-string параметры
        :param dict|str data: если dict, то данные будут отправлены в form-encoded виде
                     если str, то будет отправлена строка как есть
        :param dict headers:
        :param dict cookies:
        :param int|float timeout: в секундах
        :param int|float connection_timeout: в секундах - таймаут на соединение
        :param bool send_tvm_ticket: переопределяет self.send_tvm_ticket на один запрос
        :param bool recreate_connection_on_5xx: пересоздавать сессию, чтобы балансер выбрал хост заново при пятисотках
        :param bool recreate_connection_on_exception: пересоздавать сессию, чтобы балансер выбрал хост заново
                     при исключениях
        :param auth: объект авторизации, передается без изменений в конструктор Request
        :param http_retries: количество попыток выполнить запрос (default: http_retries из конфига сервиса)

        :rtype: requests.Response
        """
        headers = self.get_service_headers(headers, timeout)
        if self.static_headers:
            headers.update(self.static_headers)

        # Добавление TVM 2.0 тикета
        if SERVICES_TVM_2_0_ENABLED and self.tvm_2_0:
            user_ticket = mpfs.engine.process.get_tvm_2_0_user_ticket()
            if user_ticket:
                headers['X-Ya-User-Ticket'] = user_ticket.value()

            if FEATURE_TOGGLES_TVM_DAEMON_USAGE_ENABLED:
                from mpfs.core.services.tvm_daemon_service import TVMDaemonService
                service_ticket = TVMDaemonService().get_service_ticket_for_client(self.tvm_2_0['client_id'])
            else:
                service_ticket = mpfs.engine.process.get_tvm_2_0_service_ticket_for_client(self.tvm_2_0['client_id'])
            headers['X-Ya-Service-Ticket'] = service_ticket.value()

        if send_tvm_ticket is None:
            send_tvm_ticket = self.send_tvm_ticket
        if send_tvm_ticket:
            tvm_ticket = mpfs.engine.process.get_external_tvm_ticket()
            if tvm_ticket is None:
                tvm_ticket = mpfs.engine.process.get_tvm_ticket()
            if tvm_ticket:
                headers['Ticket'] = tvm_ticket.value()

        if self.send_cloud_request_id:
            cloud_request_id = mpfs.engine.process.get_cloud_req_id()
            if cloud_request_id:
                headers['Yandex-Cloud-Request-ID'] = cloud_request_id

        if isinstance(data, unicode):
            data = data.encode('utf-8')

        error = None
        if recreate_connection_on_5xx is None:
            recreate_connection_on_5xx = self.recreate_connection_on_5xx
        if recreate_connection_on_exception is None:
            recreate_connection_on_exception = self.recreate_connection_on_exception
        if base_url is None:
            base_url = self.base_url
        if http_retries is None:
            http_retries = self.http_retries
        for try_count in xrange(http_retries):
            start_time = time.time()
            try:
                headers['X-Request-Attempt'] = str(try_count)
                request = requests.Request(
                    method, base_url + relative_url, params=params, data=data, headers=headers, cookies=cookies,
                    auth=auth)
                prepared_request = self._session.prepare_request(request)
                response = self._session.send(
                    prepared_request,
                    timeout=(connection_timeout or self.connection_timeout, timeout or self.timeout),
                )
                end_time = time.time()
                self.__requests_log_response(response, end_time - start_time)
                if response.ok:
                    return response
                else:
                    error = self._exception_from_response(response)
                    # не ретраим 4xx
                    if 400 <= response.status_code <= 499:
                        break
                    elif 500 <= response.status_code <= 599 and recreate_connection_on_5xx:
                        self._session.close()
                        self._initialize_session()
            except Exception as e:
                end_time = time.time()
                url = self._transform_url_for_logging(prepared_request.url)
                msg = self._transform_url_for_logging(str(e))
                exc = self._transform_url_for_logging(traceback.format_exc())
                get_error_log().error('failed to open "%s", exception: "%s", trace: "%s"' % (url, msg, exc))

                self.__requests_log_exception(prepared_request, e, end_time - start_time)
                if try_count + 1 >= self.http_retries:  # last iteration
                    raise sys.exc_info()[0], msg, sys.exc_info()[2]
                if recreate_connection_on_exception:
                    self._session.close()
                    self._initialize_session()

        if error:
            raise error
        else:
            raise Exception('This must be unreachable')

    def __get_request_log_string_parameters(self, prepared_request):
        method = prepared_request.method
        url = self._transform_url_for_logging(prepared_request.url)

        headers = prepared_request.headers
        if headers:
            headers = headers_to_log(headers, self.not_logged_headers)

        body = prepared_request.body
        if body:
            bytes_sent = len(prepared_request.body)
        else:
            bytes_sent = 0

        url_message = url
        if headers:
            url_message = url_message + ' headers:' + headers
        if body:
            url_message = url_message + ' body:' + body

        return {'method': method, 'url': url, 'headers': headers, 'body': body, 'bytes_sent': bytes_sent, 'url_message': url_message}

    @staticmethod
    def _transform_url_for_logging(url):
        return url

    def __requests_log_response(self, response, elapsed_time):
        log_parameters = self.__get_request_log_string_parameters(response.request)
        log_parameters['elapsed_time'] = elapsed_time
        log_parameters['response_code'] = response.status_code

        if response.content:
            log_parameters['bytes_received'] = len(response.content)
        else:
            log_parameters['bytes_received'] = 0

        requests_log.info(self.REQUESTS_LOG_PATTERN % log_parameters)

    def __requests_log_exception(self, prepared_request, exception, elapsed_time):
        """Для ошибок логируем строку в requests log с 9xx статусом."""
        log_parameters = self.__get_request_log_string_parameters(prepared_request)
        log_parameters['elapsed_time'] = elapsed_time
        log_parameters['bytes_received'] = 0

        self.__define_response_code(exception, log_parameters)

        requests_log.info(self.REQUESTS_LOG_PATTERN % log_parameters)

    @staticmethod
    def __define_response_code(exception, log_parameters):
        if isinstance(exception, requests.exceptions.ConnectTimeout):
            log_parameters['response_code'] = 901
        elif isinstance(exception, requests.exceptions.ReadTimeout):
            log_parameters['response_code'] = 902
        elif isinstance(exception, requests.exceptions.HTTPError):
            log_parameters['response_code'] = 903
        elif isinstance(exception, requests.exceptions.SSLError):
            log_parameters['response_code'] = 904
        elif isinstance(exception, requests.exceptions.ProxyError):
            log_parameters['response_code'] = 905
        elif isinstance(exception, requests.exceptions.TooManyRedirects):
            log_parameters['response_code'] = 906
        else:
            log_parameters['response_code'] = 900


class Service(object):
    name = ''
    base_url = None
    api_error = errors.APIError
    log = service_log
    _response_patches = defaultdict(defaultdict)
    tvm_2_0 = None
    http_retries = None

    def __init__(self, *args, **kwargs):
        self.send_tvm_ticket = None

        config = settings.services['common'].copy()
        config.update(settings.services.get(self.name, {}))
        for attr, value in config.iteritems():
            if attr == 'tvm_2_0' and not value.get('client_id'):
                raise Exception('Can\'t configure services.%s.%s in %s: no such attribute' % (self.name, attr, self.__class__.__name__))
            setattr(self, attr, value)

    def set_response_patch(self, method, url_pattern, status_code, data='', headers=None):
        """
        Метод задаёт значение следующего ответа сервиса на запрос URL'а.
        """
        headers = headers or {}
        self._response_patches[method][re.compile(url_pattern, re.IGNORECASE)] = (status_code, data, headers)

    def clean_response_patches(self):
        self._response_patches = defaultdict(defaultdict)

    def pop_response_patch(self, method, url):
        patches = self._response_patches.get(method, {})
        for pattern, resp in patches.iteritems():
            if pattern.match(url) or pattern.match(urllib.unquote(url)):
                # unquote делаем для запросов вида .../profile/geopoints/work%3Dhome%3D0?__uid=128280859
                return patches.pop(pattern)  # удаляем отработавший паттерн

    def open_url(self, url, post={}, cookie={}, **kwargs):
        """
        :param string url: url на который необходимо отправить запрос
        :param dict post: данные для отправки
        :param dict cookie: cookie
        :param Exception|None api_err: класс исключения, которое нужно выбросить в случае ошибки. При передаче в kwargs
        параметров return_status/rais нужно передавать в api_err None для того, чтобы эти параметры использовались при
        обработке ошибок внутри http_client.open_url
        :param kwargs: дополнительные параметры для передачи в http_client.open_url
        :return: response
        """
        if self._response_patches:
            response_patch = self.pop_response_patch(kwargs.get('method', 'GET'), url)
            if response_patch:
                kwargs['response_patch'] = response_patch

        if kwargs.get('http_retries') is None:
            kwargs['http_retries'] = self.http_retries
        if kwargs.get('timeout') is None:
            kwargs['timeout'] = self.timeout
        kwargs['pass_crid'] = self.pass_cloud_request_id
        if 'api_err' not in kwargs:
            kwargs['api_err'] = self.api_error

        # Смотрим из конфигурации нужно ли слать TVM 1.0 тикет
        if self.send_tvm_ticket is not None:
            kwargs['need_tvm_ticket'] = self.send_tvm_ticket

        # Добавление TVM 2.0 тикета
        headers = kwargs.get('headers', {})
        if SERVICES_TVM_2_0_ENABLED and self.tvm_2_0:
            user_ticket = mpfs.engine.process.get_tvm_2_0_user_ticket()
            if user_ticket:
                headers['X-Ya-User-Ticket'] = user_ticket.value()

            if FEATURE_TOGGLES_TVM_DAEMON_USAGE_ENABLED:
                from mpfs.core.services.tvm_daemon_service import TVMDaemonService
                service_ticket = TVMDaemonService().get_service_ticket_for_client(self.tvm_2_0['client_id'])
            else:
                service_ticket = mpfs.engine.process.get_tvm_2_0_service_ticket_for_client(self.tvm_2_0['client_id'])
            headers['X-Ya-Service-Ticket'] = service_ticket.value()

            kwargs['headers'] = headers

        response = http_client.open_url(url, self.log, post, cookie, **kwargs)

        if response is None:
            raise self.api_error()
        elif response:
            if kwargs.get('return_status', False):
                self.log_response(response[1])
            else:
                self.log_response(response)
            return response

    def xmlrpc_call(self, method, *args):
        response = xmlrpc_client.call(
            self.base_url,
            method,
            self.log,
            self.timeout,
            self.api_error,
            *args
        )

        if response is None:
            raise self.api_error()
        elif response:
            self.log.debug(response)
            return response

    def log_response(self, response):
        self.log.debug(response)

    def process_response(self, action, data):
        raise errors.MPFSNotImplemented()

    def xml2dict(self, element):
        if not element.get('id'):
            return {}
        etype = {'file' : 'file', 'folder' : 'dir'}
        res = {
            'type'  : etype.get(element.tag),
            'name'  : element.get('name'),
            'ctime' : element.get('ctime'),
            'id'    : element.get('id'),
            'link'  : element.get('link'),
            'mimetype': element.get('type'),
            'size'    : element.get('size'),
        }
        res = dict(filter(lambda(x, y): y, res.iteritems()))
        meta_data = element.findall('metainfo')
        #===================================================================
        if not isinstance(meta_data, list) or not meta_data:
            meta_data = element.findall('meta')
        else:
            meta_data = meta_data[0].findall('meta')
        #===================================================================
        all_meta = {}
        for meta in meta_data:
            if meta.attrib['name'] in ('to', 'from', 'cc', 'bcc'):
                all_meta[meta.attrib['name']] = self.parse_addr(meta.attrib['value'])
            else:
                all_meta[meta.attrib['name']] = meta.attrib['value']
        res['meta'] = all_meta
        return res

    def parse_addr(self, addr):
        result = []

        def _clean(inp):
            return inp.replace('<', '').replace('>', '').replace(',', ' ').replace( \
            '"', '').replace("'", "").replace('  ', ' ').replace('\\', '')
        def pairwise(iterable):
            itnext = iter(iterable).next
            while 1:
                yield itnext(), itnext()

        addr = _clean(addr)
        sp = MAIL_RE.split(addr)
        address_dict = dict([x[::-1] for x in pairwise(sp)])
        result = map(lambda (k, v): { 'email' : k, 'name' : v }, address_dict.iteritems())

        return result

    def sort(self, heap, sort_field, order):
        order = not bool(order)
        sort_field = sort_field or 'name'
        files = filter(lambda res: res['type'] == 'file', heap)
        folders = filter(lambda res: res['type'] == 'dir', heap)

        def _sort(seq, s_field):
            if not seq:
                return []
            if s_field not in seq[0]:
                s_field = 'name'
            if s_field == 'name':
                seq.sort(natcasecmp, key=operator.itemgetter(s_field), reverse=order)
                result = seq[:]
            else:
                result = []
                seq.sort(key=operator.itemgetter(s_field), reverse=order)

                def countDuplicatesInList(dupedList):
                    uniqueSet = set(item for item in dupedList)
                    return dict((item, dupedList.count(item)) for item in uniqueSet)

                sorted_values = countDuplicatesInList(list(i[s_field] for i in seq))
                approved_elements = set()

                for _elem in seq:
                    if _elem[s_field] not in approved_elements:
                        if sorted_values[_elem[s_field]] > 1:
                            _res = filter(lambda res: res[s_field] == _elem[s_field], seq)
                            _res.sort(natcasecmp, key=operator.itemgetter('name'), reverse=order)
                            result.extend(_res)
                            approved_elements.add(_elem[s_field])
                        else:
                            result.append(_elem)

            return result

        return _sort(folders, sort_field) + _sort(files, sort_field)

    def apply_filter(self, resource_data, fltr):
        '''
        assert fltr values equal to src
        '''
        def _filt(fkys, dkys, dta):
            if not fkys or not dta:
                return True
            elif fkys.issubset(dkys):
                gttr = operator.itemgetter(*fkys)
                return gttr(dta) == gttr(fltr)
            else:
                return False
        if fltr:
            res_set = set(resource_data)
            flt_set = set(fltr)
            meta_keys = flt_set - res_set
            main_keys = flt_set - meta_keys
            return _filt(main_keys, res_set, resource_data) and _filt(meta_keys, res_set, resource_data.get('meta', {}))
        else:
            return True

    def start(self, uid):
        pass

    def commit(self, uid):
        pass

    def rollback(self, uid):
        pass

    def diff(self, uid, path, version):
        raise errors.MPFSNotImplemented()

    def update_counter(self, uid, value, version=None):
        pass

    def drop(self, uid):
        pass

    def get_version(self, uid):
        return ''

    def make_token(self, items):
        string_parts = []
        for item in items:
            if isinstance(item, (int, long)) or not item or item is None:
                item = str(item)
            string_parts.append(item)

        token = '-'.join(string_parts)
        if isinstance(token, unicode):
            token = token.encode('utf-8')

        return token

    def signed_token(self, *args):
        '''
        Собирает из аргументов строку вида SECRET_KEY-A-B-C-***-Z
        и выдает md5 от этой нечисти
        '''
        items = [self.secret_key]
        items.extend(list(args))
        token = self.make_token(items)

        hashDigest = hashlib.md5()
        hashDigest.update(token)
        return hashDigest.hexdigest()

    def signed_token_v2(self, *args):
        token = self.make_token(list(args))
        return hmac.new(self.secret_key, token, hashlib.sha256).hexdigest()

    def get_data_for_uid(self, uid, doc):
        """Формирует словарь для инициализации ресурса текущего сервиса
        из данных ``doc`` записи из базы.

        При необходмости в данные добавляется версия для ``uid``.

        :type uid: str
        :type doc: dict
        :rtype: dict
        """
        return doc


class StorageService(Service):

    def __init__(self, *args, **kwargs):
        Service.__init__(self, *args, **kwargs)
        self.available_filters = ()
        self.available_bounds = {}
        self.request_processing_fields = {
            'filters' : {},
            'bounds' : {},
        }
        """
        Поля которые могут быть обработаны сервисом.
        Дело в том, что некоторые сервисы не умеют, например, сортировку, фильтрацию или постраничный вывод,
        тогда мы вынуждены запрашивать всё, и уже на стороне MPFS сортировать/фильтровать/резать данные.
        Поэтому в этом атрибуте будут содержаться только те поля которые можно обработать на стороне сервиса.
        Поля которые мы должны обработать в MPFS будут помещены в `model.form.args`.
        """
        self.available_sort_fields = {}

    def split_filters(self, model):
        '''
        Разделение внешних и внутренних фильтров
        '''
        if model.request and model.form.args and model.form.args.forward:
            self.request_processing_fields = {
                'filters' : {},
                'bounds' : {},
            }
            bounds = {}
            remote_filter_fields = set(self.available_filters) - (set(self.available_filters) - set(model.form.args['filter']))
            remote_filter = dict(
                filter(
                    lambda (k, v): k in remote_filter_fields,
                    model.form.args['filter'].iteritems()
                )
            )

            filters = dict(
                filter(
                    lambda (k, v): k not in remote_filter_fields,
                    model.form.args['filter'].iteritems()
                )
            )

            self.request_processing_fields['filters'] = remote_filter

            if not filters and \
                'sort' in self.available_bounds and \
                model.form.args['bounds']['sort'] in self.available_sort_fields:
                for k, v in model.form.args['bounds'].iteritems():
                    if k in self.available_bounds:
                        if k == 'sort':
                            self.request_processing_fields['bounds'][self.available_bounds[k]] = self.available_sort_fields[v]
                        else:
                            self.request_processing_fields['bounds'][self.available_bounds[k]] = v
                    else:
                        bounds[k] = v
                model.form.args['bounds'] = bounds
            model.form.args['filter'] = filters

    def folder_content(self, model):
        self.split_filters(model)

    def inspect_storage(self, uid):
        raise errors.MPFSNotImplemented()

    def listing(self, *args, **kwargs):
        return self.folder_content(*args, **kwargs)

    def lock_set(self, uid, path, version=None):
        raise errors.MPFSNotImplemented()

    def lock_unset(self, uid, path, version=None):
        raise errors.MPFSNotImplemented()

    def lock_check(self, uid, path, version=None):
        raise errors.MPFSNotImplemented()

    @staticmethod
    def add_beginning_and_trailing_slashes(path):
        """Добавить в конец и начало пути слэши по надобности."""
        if not path.startswith('/'):
            path = '/' + path
        if not path.endswith('/'):
            path += '/'
        return path


class BaseFilter(object):

    def __init__(self, val):
        if isinstance(val, (str, unicode)):
            val = val.split(',')
        self.val = val

    def get(self):
        return ''

