# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис для получения списка хостов по группам/макросам и т.д. на Кондукторе
https://wiki.yandex-team.ru/Conductor/API/handles/

Пример запроса для получения списка хостов по группе:
    curl "https://c.yandex-team.ru/api/groups2hosts/disk_mpfs,disk_api?fields=id,fqdn&format=json"


"""
import urlparse
import socket
import time
import os
import cjson
import demjson

import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.util.persistent_dict import PersistentDict
from mpfs.core.services.common_service import Service
from mpfs.common.util import from_json
from mpfs.config import settings
from mpfs.core.signals import register_signal, add_timer
from mpfs.engine.process import Signal

SERVICES_CONDUCTOR_CACHE_FILE_PATH = settings.services['conductor']['cache_file_path']
SERVICES_CONDUCTOR_CACHE_UPDATE_PERIOD = settings.services['conductor']['cache_update_period']
SERVICES_CONDUCTOR_SERVICE_CHECK_INTERVAL = settings.services['conductor']['service_check_interval']

service_log = mpfs.engine.process.get_service_log('conductor')


class ConductorService(Service):
    name = 'conductor'
    log = service_log
    api_error = errors.ConductorError
    cache_filepath = SERVICES_CONDUCTOR_CACHE_FILE_PATH
    cache_file_format = 'json'

    def __init__(self, read_cache=True, *args, **kwargs):
        super(ConductorService, self).__init__(*args, **kwargs)
        self._cache_data = {
            'cache': {},
            'update_time': 0,
        }

    def get_conductor_item_by_ip(self, ip):
        return self.cache.get(ip)

    @classmethod
    def is_cache_file_exists(cls):
        return os.path.exists(cls.cache_filepath)

    @classmethod
    def is_conductor_auth_fallback_mode(cls, request):
        ip = request.remote_addr
        if ip == '::1' or ip == '127.0.0.1':
            # пускать запросы с локальной машины (по ipv6 и ipv4)
            return True
        elif not cls.is_cache_file_exists():
            # если файла с кешом нет на диске - пускаем всех со всеми скоупами
            # https://st.yandex-team.ru/CHEMODAN-28353
            cls.log.warning('Conductor cache file does not exist (%s). Allow access for internal API' % cls.cache_filepath)
            return True
        return False

    @property
    def cache(self):
        now = self._get_timestamp()
        if now > self._cache_data['update_time'] + SERVICES_CONDUCTOR_CACHE_UPDATE_PERIOD:
            self._reload_cache_from_file()
            self._cache_data['update_time'] = now  # обновляем в любом случае
        return self._cache_data['cache']

    def _reload_cache_from_file(self):
        cache_dict = PersistentDict(self.cache_filepath, flag='r', mode=None, format=self.cache_file_format)
        if not cache_dict:
            self.log.info('Reload cache from file failed')
            # продолжаем пользоваться старым кешом - новый (на диске) побитый, ждем следующего обновления
        else:
            self.cache = cache_dict

    @cache.setter
    def cache(self, value):
        self._cache_data['cache'] = value
        self._cache_data['update_time'] = self._get_timestamp()

    def get_hosts_by_group(self, groups):
        """
        Получаем список хостов по имени группы или списку групп

        :param groups: имя группы или список имен групп, хосты которых нужно получить
        :type groups: list | str
        :return: список хостов
        """
        if isinstance(groups, basestring):
            groups = [groups]

        groups_str = ','.join(groups)
        url = urlparse.urljoin(self.base_url, 'groups2hosts/%s?fields=id,fqdn&format=json' % groups_str)

        result = self.open_url(url)
        service_result = from_json(result)
        fqdn_list = [i['fqdn'] for i in service_result if 'fqdn' in i]

        return fqdn_list

    def get_group(self, group_name, request_fields=None):
        """
        Получаем список хостов по имени группы из кондуктора.

        :param group_name: имя группы без %.
        :param request_fields: Список полей, которые хотим получить. Если не указать, то вернется только FQDN.
                               Полный список можно узнать тут:
                               https://wiki.yandex-team.ru/Conductor/API/handles/#groups2hosts
        :return: список словарей, описывающих хост, например:
            [{"fqdn":"mpfs10g.disk.yandex.net"},{"fqdn":"mpfs10h.disk.yandex.net"}, ... ]
            или
            [{"fqdn":"mpfs10g.disk.yandex.net","datacenter_name":"myt5"},
             {"fqdn":"mpfs10h.disk.yandex.net","datacenter_name":"fol4"}, ... ]
        """
        fields = None
        if request_fields is not None:
            fields = ','.join(request_fields)

        if fields:
            url = urlparse.urljoin(self.base_url, 'groups2hosts/%s?fields=%s&format=json' % (group_name, fields))
        else:
            url = urlparse.urljoin(self.base_url, 'groups2hosts/%s?format=json' % group_name)

        result = self.open_url(url)
        response = from_json(result)
        return response

    def get_host_by_fqdn(self, fqdn):
        """
        Получаем информацию о хосте по его имени, нужно для проверки, есть ли такой хост в кондукторе

        :param fqdn: имя хоста
        :type fqdn: str
        :return: информация о хосте
        """
        url = urlparse.urljoin(self.base_url, 'hosts/%s?format=json' % fqdn)

        result = self.open_url(url)
        service_result = from_json(result)
        if isinstance(service_result, list) and len(service_result):
            return service_result[0]
        elif isinstance(service_result, dict) and len(service_result):
            return service_result
        else:
            return None

    @staticmethod
    def _get_ip_by_hostname(name):
        """
        Резолвит Возвращает ip или список ip адресов по данному хосту

        :param name: имя хоста
        :return: list | None
        """
        try:
            result = socket.getaddrinfo(name, None)
        except socket.error:
            return None

        ip_addresses = set(i[4][0] for i in result)
        return list(ip_addresses)

    @staticmethod
    def _get_timestamp():
        return int(time.time() * 1000)

    def _get_hosts_by_item(self, item):
        hosts = []

        if item.startswith('%'):
            group_name = item[1:]
            hosts = self.get_hosts_by_group(group_name)
        else:
            host = self.get_host_by_fqdn(item)
            if host is not None:
                hosts = [host]

        return hosts

    def fetch_hosts_from_server(self, items):
        """
        Получает на вход список объектов и идет с ними в кондуктор, резолвит группы в хосты, хосты проверяет, что
        присутствуют в кондукторе.

        :param items: список кондукторных объектов. Если начинается с %, то считаем, что это кондукторная группа.
                      Если не начинается с %, то считаем, что это хост.
        :return: словарь пар вида <ip адрес>: <сущность кондуктора> (сущность кондуктора - это группа или fqdn)
        :rtype: dict
        """
        conductor_item_to_ip = {}

        for item in items:
            try:
                hosts = self._get_hosts_by_item(item)
            except errors.ConductorError, e:
                if e.data and e.data.get('code') == 404:
                    # не добавляем в словарь, если такой сущности в кондукторе нет - тогда она будет удалена
                    pass
                else:
                    self.log.exception('Failed to get response for item %s' % item)
                    conductor_item_to_ip[item] = None  # пишем в словарь None, чтобы не удалялась предыдущая группа
            except (ValueError, demjson.JSONDecodeError, cjson.DecodeError):
                self.log.exception('Failed to decode json response for item %s' % item)
                conductor_item_to_ip[item] = None  # пишем в словарь None, чтобы не удалялась предыдущая группа
            else:
                # иначе резолвим ip адреса и добавляем их в словарь
                ip_addresses = []
                for host in hosts:
                    ip_list = self._get_ip_by_hostname(host)
                    if ip_addresses is not None:
                        ip_addresses.extend(ip_list)

                conductor_item_to_ip[item] = ip_addresses

        return conductor_item_to_ip

    @staticmethod
    def _clean_cache_for_item(item, persistent_dict):
        for key, value in persistent_dict.items():
            if value == item:
                del persistent_dict[key]

    def _update_cache_for_item(self, item, conductor_item_to_ip, persistent_dict):
        if item not in conductor_item_to_ip:
            self._clean_cache_for_item(item, persistent_dict)
        else:
            ip_addresses = conductor_item_to_ip[item]
            if not ip_addresses:
                pass  # не обновляем данные кеша, если пришел None или список пустой
            else:
                for ip in ip_addresses:
                    persistent_dict[ip] = item

    def update_cache(self, platform_auth):
        conductor_items = set()
        config_data = filter(lambda i: 'conductor' in i['auth_methods'], platform_auth)
        for props in config_data:
            if not props['enabled'] or 'conductor_items' not in props:
                continue

            items = props['conductor_items']
            conductor_items.update(items)

        try:
            conductor_item_to_ip = self.fetch_hosts_from_server(conductor_items)
        except Exception:
            self.log.exception('Fetch hosts from Conductor failed')
            return

        cache_dir = os.path.dirname(self.cache_filepath)
        if not os.path.exists(cache_dir):
            os.makedirs(cache_dir)

        with PersistentDict(self.cache_filepath, flag='c', mode=None, format=self.cache_file_format) as persistent_dict:
            for conductor_item in conductor_items:
                self._update_cache_for_item(conductor_item, conductor_item_to_ip, persistent_dict)
            for ip, item in persistent_dict.items():
                if item not in conductor_items:
                    del persistent_dict[ip]


def setup_conductor_cache_update():
    """Запустить периодическое обновление кеша.
    """
    register_signal(
        Signal.CONDUCTOR_CACHE,
        lambda signum: ConductorService(read_cache=False).update_cache(settings.platform['auth']),
    )
    add_timer(Signal.CONDUCTOR_CACHE, SERVICES_CONDUCTOR_SERVICE_CHECK_INTERVAL)
