# -*- coding: utf-8 -*-
from types import NoneType

import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.util import from_json, to_json
from mpfs.core.services.common_service import Service
from mpfs.platform.utils import quote_string

service_log = mpfs.engine.process.get_service_log('data-api')


class DataApiService(Service):
    """
    Сервис для общения с API профилей

    Документация по API: https://wiki.yandex-team.ru/disk/data-api/backend-api
    """
    name = 'data_api'
    api_error = errors.DataApiBadResponse
    log = service_log
    base_url = None  # should be set in config
    timeout = None  # should be set in config
    list_users_timeout = None  # should be set in config


class DataApiDocsService(DataApiService):
    service_url = '/api/process_generic_data/?http_method=%(http_method)s&resource_path=%(resource_path)s&__uid=%(uid)s'

    def __init__(self, resource_path, *args, **kwargs):
        super(DataApiDocsService, self).__init__(*args, **kwargs)
        self.resource_path = resource_path

    def build_url(self, uid, http_method, object_id=None):
        context = {
            'uid': uid,
            'http_method': http_method,
            'resource_path': self.resource_path if object_id is None else '%s/%s' % (self.resource_path, object_id),
        }
        urlencoded_context = self.urlencode_context(context)
        return '%s%s' % (self.base_url, self.service_url % urlencoded_context)


    def open_url(self, url, post={}, cookie={}, **kwargs):
        headers = {'Accept': 'application/json', 'Content-type': 'application/json'}
        return super(DataApiDocsService, self).open_url(url, post, cookie, method='POST', headers=headers, **kwargs)

    @staticmethod
    def urlencode_context(context):
        ret = {}
        for k, v in context.iteritems():
            ret[k] = quote_string(v)
        return ret

    def get_all_user_docs(self, uid):
        url = self.build_url(uid, 'GET')
        return self.request_service(url)['items']

    def delete_doc_item(self, uid, object_id):
        url = self.build_url(uid, 'DELETE', object_id=object_id)
        return self.request_service(url)

    def post_doc_item(self, uid, data):
        url = self.build_url(uid, 'POST')
        pure_data = to_json(data)
        return self.request_service(url, pure_data=pure_data)

    def request_service(self, url, **kwargs):
        status_code, resp, headers = self.open_url(url, return_status=True, **kwargs)
        if resp and not isinstance(resp, (dict, list, NoneType)):
            resp = from_json(resp)
        return self._get_result_from_response(resp)

    @staticmethod
    def _get_result_from_response(resp):
        return resp.get('result')

data_api = DataApiService()
