# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис Директория
https://api-internal-test.directory.ws.yandex.net/docs/

Примеры запросов для получения списка контактов для отдела:

    curl -H "X-UID: 1130000000156481" -H "X-USER-IP: 127.0.0.1" -H "X-Ya-Service-Ticket: <service-ticket>"
     -H "Accept: application/json" "https://api-internal-test.directory.ws.yandex.net/users/?department_id=5"

"""
import urlparse
import mpfs.engine.process

from collections import namedtuple

from mpfs.common.errors import APIError
from mpfs.core.services.common_service import Service
from mpfs.common.util import from_json, to_json
from mpfs.common.util.urls import update_qs_params


default_log = mpfs.engine.process.get_default_log()
service_log = mpfs.engine.process.get_service_log('directory')


class DirectoryContact(object):
    def __init__(self, user_dict):
        self.uid = str(user_dict['id'])
        self.email = user_dict['email']

    def __repr__(self):
        return "%s(uid=%s, email=%s)" % (self.__class__.__name__, self.uid, self.email)

    def __hash__(self):
        return hash(self.uid)

    def __eq__(self, other):
        return self.uid == other.uid


DirectoryOrganizationInfo = namedtuple('DirectoryOrganizationInfo', [
    'id',
    'is_paid',
    'disk_limit',
    'disk_usage',
])


DirectoryOrganizationLimit = namedtuple('DirectoryOrganizationLimit', [
    'limit',
    'free',
])


class MigrationInProcessException(APIError):
    pass


class DirectoryService(Service):
    name = 'directory'
    log = service_log

    def open_url(self, url, **kwargs):
        return super(DirectoryService, self).open_url(url, need_tvm_ticket=False, **kwargs)

    def get_organizations_by_uids(self, uids):
        uids = ','.join(uids)
        url = urlparse.urljoin(self.base_url, 'v11/organizations-by-uids/?uids=%s' % uids)
        return from_json(self.open_url(url))

    def get_group_contacts(self, org_id, group_id, user_ip=None):
        return self.get_contacts(org_id, 'recursive_group_id', group_id, user_ip)

    def get_department_contacts(self, org_id, department_id, user_ip=None):
        return self.get_contacts(org_id, 'recursive_department_id', department_id, user_ip)

    def get_contacts(self, org_id, entity_type, entity_id, user_ip=None, resource_relation_name=None):
        return list(self.iter_by_contacts(
            org_id, entity_type, entity_id, user_ip=user_ip,
            resource_relation_name=resource_relation_name
        ))

    def iter_by_contacts(self, org_id, entity_type, entity_id, user_ip=None, resource_relation_name=None, uid=None):
        """Проитерироваться по списку сотрудниов

        https://api-internal-test.directory.ws.yandex.net/docs/playground.html#spisok-sotrudnikov
        """
        headers = self._get_headers(user_ip=user_ip, organization_id=org_id, uid=uid)
        params = {
            entity_type: entity_id,
            'per_page': 100
        }
        if resource_relation_name is not None:
            params['resource_relation_name'] = resource_relation_name

        next_url = update_qs_params(urlparse.urljoin(self.base_url, '/users/'), params)
        while next_url:
            service_result = from_json(self.open_url(next_url, headers=headers))
            for user_dict in service_result['result']:
                yield DirectoryContact(user_dict)
            next_url = service_result['links'].get('next')

    def get_shared_folder_members(self, owner_uid, gid):
        """Получить из директории b2b-ых пользователей имеющих доступ в ОП

        :rtype: tuple(set, set)
        """
        resource_id = "%s:%s" % (owner_uid, gid)
        rw_contacts = set(self.iter_by_contacts(None, 'resource_id', resource_id, resource_relation_name='write', uid=owner_uid))
        ro_contacts = set(self.iter_by_contacts(None, 'resource_id', resource_id, resource_relation_name='read', uid=owner_uid))
        # на момент написания кода в директории множества пользователей с
        # доступом только на чтение и на чтение/запись пересекались
        ro_contacts -= rw_contacts
        return rw_contacts, ro_contacts

    @staticmethod
    def encode_group_entity(org_id, entity_type, entity_id):
        return '%s:%s:%s' % (org_id, entity_type, str(entity_id))

    @staticmethod
    def decode_group_entity(entity):
        org_id, entity_type, entity_id = entity.split(':')
        return org_id, entity_type, entity_id

    @staticmethod
    def format_group_name(org_name, entity_name):
        if entity_name:
            return '%s/%s' % (org_name, entity_name)
        return org_name

    def notify_user_activate_invite(self, owner_uid, gid, invited_uid, read_only=True):
        body = {
            'operations': [
                {
                    'operation_type': 'add',
                    'value': {
                        'name': 'read' if read_only else 'write',
                        'object_type': 'user',
                        'object_id': int(invited_uid),
                    },
                },
            ],
        }
        path = '/resources/%(owner_uid)s:%(gid)s/relations/bulk-update/' % {'owner_uid': owner_uid, 'gid': gid}
        url = urlparse.urljoin(self.base_url, path)
        headers = self._get_headers(uid=owner_uid)
        self.open_url(url, method='POST', headers=headers, pure_data=to_json(body))

    def get_organization_info(self, organization_id):
        """
        Пример запроса, который делает этот метод:
        curl -v 'https://api-internal-test.directory.ws.yandex.net/v5/organizations/125/?fields=disk_limit,disk_usage,subscription_plan' \
             -H 'X-Org-ID: 125' \
             -H 'X-Ya-Service-Ticket: <service-ticket>'
        """
        url = urlparse.urljoin(self.base_url, '/v5/organizations/%s/?fields=disk_limit,disk_usage,subscription_plan'
                               % str(organization_id))
        headers = self._get_headers(organization_id=organization_id)
        try:
            response = self.open_url(url, method='GET', headers=headers)
        except APIError as e:
            try:
                default_log.info('body: ' + e.data['text'])
                error_code = from_json(e.data['text'])['code']
            except Exception:
                raise e
            else:
                if error_code == 'not-ready':
                    raise MigrationInProcessException(e)
                raise e
        else:
            json = from_json(response)
            return DirectoryOrganizationInfo(id=str(json['id']),
                                             is_paid=json['subscription_plan'] == 'paid',
                                             disk_limit=json['disk_limit'],
                                             disk_usage=json['disk_usage'])

    def get_organization_infos(self):
        """
        Пример запроса, который делает этот метод:
        curl -v 'https://api-internal-test.directory.ws.yandex.net/v5/organizations/?show=all&fields=disk_limit,disk_usage,subscription_plan' \
             -H 'X-Ya-Service-Ticket: <service-ticket>'
        """
        url = urlparse.urljoin(self.base_url, '/v5/organizations/?show=all&fields=disk_limit,disk_usage,'
                                              'subscription_plan')
        headers = self._get_headers()
        while url:
            response = self.open_url(url, method='GET', headers=headers)
            json = from_json(response)
            for organization_info in json['result']:
                yield DirectoryOrganizationInfo(id=str(organization_info['id']),
                                                is_paid=organization_info['subscription_plan'] == 'paid',
                                                disk_limit=organization_info['disk_limit'],
                                                disk_usage=organization_info['disk_usage'])
            if 'links' in json and 'next' in json['links']:
                url = json['links']['next']
            else:
                url = None

    def get_organization_disk_limits(self, organization_id):
        """
        Пример запроса, который делает этот метод:
        curl -v 'https://api-internal-test.directory.ws.yandex.net/v5/disk-usage/' \
             -H 'X-Org-ID: 125' \
             -H 'X-Ya-Service-Ticket: <service-ticket>'
        """
        url = urlparse.urljoin(self.base_url, '/v5/disk-usage/')
        headers = self._get_headers(organization_id=organization_id)
        response = self.open_url(url, method='GET', headers=headers)
        json = from_json(response)
        return DirectoryOrganizationLimit(limit=json['limit'], free=json['free'])

    def set_organization_disk_usage(self, organization_id, usage):
        """
        Пример запроса, который делает этот метод:
        curl -v -X POST 'https://api-internal-test.directory.ws.yandex.net/v5/disk-usage/' \
             -d '{"usage": 10000}' \
             -H 'X-Org-ID: 125' \
             -H 'X-Ya-Service-Ticket: <service-ticket>' \
             -H 'Content-Type: application/json'
        """
        url = urlparse.urljoin(self.base_url, '/v5/disk-usage/')
        headers = self._get_headers(organization_id=organization_id)
        body = to_json({'usage': usage})
        try:
            response = self.open_url(url, method='POST', headers=headers, pure_data=body)
        except APIError as e:
            try:
                error_code = from_json(e.data['text'])['code']
            except Exception:
                raise e
            if error_code == 'not-ready':
                raise MigrationInProcessException(e)

        json = from_json(response)
        return DirectoryOrganizationLimit(limit=json['limit'], free=json['free'])

    def _get_headers(self, uid=None, organization_id=None, user_ip=None):
        user_ip = user_ip or '127.0.0.1'
        result = {
            'X-USER-IP': str(user_ip),
            'Accept': 'application/json',
            'Content-Type': 'application/json',
        }
        if uid:
            result['X-UID'] = str(uid)
        if organization_id:
            result['X-Org-ID'] = str(organization_id)
        return result
