# -*- coding: utf-8 -*-

import json
import re

from itertools import imap, groupby
from lxml import etree
from operator import itemgetter
from urllib import urlencode
from urlparse import parse_qs, urlparse

import mpfs.engine.process

from mpfs.common.util import from_json
from mpfs.config import settings
from mpfs.core.cache import cache_get, cache_set, cache_exists
from mpfs.core.services.common_service import Service
from mpfs.core.office.errors import OfficeUnsupportedActionError, OfficeUnsupportedExtensionError, OfficeError
from mpfs.core.user.constants import SUPPORTED_LOCALES, DEFAULT_LOCALE

OFFICE_SIZE_LIMITS = settings.office['size_limits']
OFFICE_DISCOVERY_URL = settings.office['discovery_url']
OFFICE_WOPI_RESOURCE_URL_TEMPLATE = settings.office['wopi_resource_url_template']

service_log = mpfs.engine.process.get_service_log('discovery')


class FakeDTDResolver(etree.Resolver):
    def resolve(self, url, id, context):
        return self.resolve_string('', context)


class DiscoveryService(Service):
    name = 'discovery'
    log = service_log
    APPS = ('Word', 'Excel', 'PowerPoint', 'WopiTest')
    ACTIONS = ('view', 'edit', 'editnew', 'getinfo')
    RE_TEMPLATE = re.compile('<[^=]+=[^&]+&>')
    CACHE_NAME = 'discovery'
    CACHE_ITEM_NAME = 'discovery'
    CACHE_EXPIRE = 0  # never
    _discovery = {}

    def __init__(self, timeout=None, retry=None):
        """
        :type timeout: None | float
        :type retry: None | bool
        """
        super(DiscoveryService, self).__init__()
        if timeout is not None:
            self.timeout = timeout
        if retry is not None:
            self.retry = retry

        self.parser = etree.XMLParser(remove_blank_text=True, resolve_entities=False, no_network=True)
        self.parser.resolvers.add(FakeDTDResolver())

    def get(self):
        """Вернуть закешированный словарь `_discovery`.

        :raises OfficeError: Если кеш не доступен
        :rtype: dict
        """
        if not self._discovery and not self._update_from_uwsgi_cache():
            raise OfficeError("No discovery available")
        return self._discovery

    def get_app_url(self, action, ext, locale=DEFAULT_LOCALE):
        """Получить из `_discovery` имя приложения и URL.

        :raises OfficeUnsupportedActionError: Если `action` мы не поддерживаем
                                              или оно недоступно в discovery.
        :raises OfficeUnsupportedExtensionError: Если `extension` мы не поддерживаем
                                              или оно недоступно в discovery.
        :type action: str
        :type ext: [str, None]
        :rtype: tuple[str, str]
        """
        if ext is not None:
            ext = ext.lower()
        apps = self.get().iteritems()
        app_exts = [(app, act[action]) for app, act in apps if action in act]
        if not app_exts:
            raise OfficeUnsupportedActionError()

        app_urls = ((a, e[ext]) for a, e in app_exts if ext in e)
        app, url = next(app_urls, (None, None))
        if not url:
            raise OfficeUnsupportedExtensionError()

        loc = '%s-%s' % (locale, SUPPORTED_LOCALES[locale])
        return app, self.resolve_qs_templates(url, {'ui': loc, 'rs': loc})

    def get_wopi_test_info(self, get_info_url, access_token, access_token_expires):
        test_cases = []

        query = parse_qs(urlparse(get_info_url).query)
        query.update({
            'access_token': access_token,
            'access_token_ttl': access_token_expires
        })
        url = get_info_url.split('?', 1)[0] + '?' + urlencode(query, doseq=1)
        try:
            test_cases = from_json(self.open_url(url))
        except Exception:
            pass
        return test_cases

    def resolve_qs_templates(self, url, query_args):
        """Удалить шаблоны из url и добавить аргументы в строку запроса.

        :type url: str
        :type query_args: dict
        :rtype: str
        """
        if '?' not in url:
            return url

        path, query = url.split('?', 1)
        query = parse_qs(re.sub(self.RE_TEMPLATE, '', query))
        query.update(query_args)
        return path + '?' + urlencode(query, doseq=1)

    def ensure_cache(self):
        """Убедиться, что `_discovery` закеширован, иначе установить кеш.

        :raises OfficeError: Если не удалось обновить кеш.
        """
        if self._discovery:
            return

        if not cache_exists(self.CACHE_ITEM_NAME, self.CACHE_NAME):
            try:
                xml = self.open_url(OFFICE_DISCOVERY_URL, timeout=self.timeout, retry=self.retry)
            except Exception:
                raise OfficeError('Unable to download discovery.xml')

            result = cache_set(
                self.CACHE_ITEM_NAME, json.dumps(self._from_xml(xml)), self.CACHE_EXPIRE, self.CACHE_NAME
            )
            if result is None:
                # например, не влез в буфер кеша
                # или нет такого кеша по имени
                raise OfficeError('Unable to set item: "%s" to cache: "%s"' % (self.CACHE_ITEM_NAME, self.CACHE_NAME))

        self._update_from_uwsgi_cache()

    def _update_from_uwsgi_cache(self):
        """Получить кеш из uwsgi и обновить ``_discovery``.

        Используется, когда какой-то воркер обновил uwsgi кеш.

        :rtype: dict
        """
        cache = cache_get(self.CACHE_ITEM_NAME, self.CACHE_NAME)
        if cache is not None:
            self._set_discovery(json.loads(cache))
        return self._discovery

    @classmethod
    def _set_discovery(cls, discovery):
        cls._discovery = discovery

    def _from_xml(self, xml):
        """Создать словарь на основе `xml` файла discovery.

        При этом отфильтровать приложения по `APP` и действия по `ACTIONS`.

        :type xml: str
        :rtype: dict
        :return: {'Word': {'view': {'doc': 'http://localhost/', ...}, ...}, ...}
        """
        discovery = dict.fromkeys(self.APPS)
        root = etree.fromstring(xml, self.parser)
        
        for app in discovery:
            xpath = 'net-zone/app[@name="%s"]/action' % app
            actions = (a.attrib for a in root.xpath(xpath))
            actions = (a for a in actions if a['name'] in self.ACTIONS)
            actions = imap(itemgetter('name', 'ext', 'urlsrc'), actions)
            groups = groupby(sorted(actions), itemgetter(0))
            discovery[app] = {a: dict(i[1:] for i in g) for a, g in groups}
        return discovery
