# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис хранилища

"""
import copy
import re
import traceback
import time

from heapq import merge
from operator import itemgetter, attrgetter
from collections import defaultdict
from itertools import ifilter, imap, izip

import mpfs.engine.process
import mpfs.common.errors as errors

from mpfs.common.errors.share import ShareNotFound
from mpfs.common.errors import SnapshotSharedFoldersTimeOut
from mpfs.config import settings
from mpfs.common.util import filetypes
from mpfs.common.util.ycrid_parser import YcridParser
from mpfs.common.static.tags import VIEWS_COUNTER
from mpfs.core.filesystem.dao.legacy import is_new_fs_spec_required
from mpfs.core.organizations.dao.organizations import OrganizationDAO
from mpfs.core.services.common_service import StorageService
from mpfs.core.services import zaberun_service, mulca_service, logreader_service
from mpfs.core.address import Address, GroupAddress
from mpfs.core.services.common_service import BaseFilter
from mpfs.core.filesystem.symlinks import Symlink
from mpfs.core.filesystem.resources.base import Resource
from mpfs.core.user.utils import ignores_shared_folders_space

from mpfs.core.metastorage.control import (
    disk,
    disk_info,
    link_data,
    tag_data,
    changelog,
    recount,
)

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

zaberun = zaberun_service.Zaberun()
mulca = mulca_service.Mulca()
logreader = logreader_service.LogreaderService()


ENABLE_WH_VERSION_FIX = settings.system['enable_wh_version_fix']
SYSTEM_SYSTEM_TRASH_SORT_BY_APPEND_TIME_LIMIT = settings.system['system']['trash_sort_by_append_time_limit']
B2B_SHARED_ORGANIZATION_SPACE_ENABLED = settings.b2b['shared_organization_space_enabled']
B2B_SHARED_ORGANIZATION_SPACE_RECALCULATION_DELAY = settings.b2b['shared_organiaztion_space_recalculation_delay']

FEATURE_TOGGLES_CHECK_DAV_MAX_FILES_LISTING_LIMIT = settings.feature_toggles['check_dav_max_files_listing_limit']
DAV_MAX_FILES_LISTING_LIMIT = settings.feature_toggles['dav_max_files_listing_limit']

class CtimeFilter(BaseFilter):

    def get(self):
        return {'data.mtime' : {'$gt' : int(self.val[0]), '$lte' : int(self.val[1])}}


class DiskFilter(BaseFilter):

    def __init__(self, val):
        self.val = val


class VisibleFilter(BaseFilter):

    def get(self):
        return {'data.visible' : {'$in': [self.val[0], int(self.val[0])]}}


class NameFilter(BaseFilter):

    def get(self):
        return {'key' : re.compile('.*/%s' % re.escape(self.val[0]))}


class MediatypeFilter(BaseFilter):

    def get(self):
        media_type_list = [filetypes.getGroupNumber(x) for x in self.val]
        return {'data.mt': {'$in': media_type_list}}


class PublicFilter(BaseFilter):

    def get(self):
        return {'$or': [{'data.meta' : ['public', int(self.val[0])]},
            {'data.public': int(self.val[0])}]}


class MtimeGreaterThenOrEqual(BaseFilter):
    def get(self):
        return {'data.mtime': {'$gte': int(self.val[0])}}


class MtimeLowerThenOrEqual(BaseFilter):
    def get(self):
        return {'data.mtime': {'$lte': int(self.val[0])}}


class MtimeInterval(BaseFilter):
    def get(self):
        if self.val[0] > self.val[1]:
            raise ValueError('"mtime_gte" must be lower then or equal "mtime_lte"')
        return {'data.mtime': {'$gte': int(self.val[0]), '$lte': int(self.val[1])}}


class ResourceType(BaseFilter):
    def get(self):
        return {'type': self.val[0]}


class ModifyUid(BaseFilter):
    def get(self):
        # формируется в "mpfs/core/lenta/logic/lenta_block_list.py"
        return self.val


class Resources(object):

    def __init__(self, **kwargs):
        for k,v in kwargs.iteritems():
            setattr(self, k, v)

    def __getitem__(self, k):
        try:
            result = getattr(self, k)
        except AttributeError:
            raise KeyError(k)
        else:
            return result


class MPFSStorageService(StorageService):

    control = disk

    def __init__(self, *args, **kwargs):
        super(MPFSStorageService, self).__init__(*args, **kwargs)
        """ Better to keep filters' description in constructor,
        cause Disk is a static field in resource class

        """
        self.available_filters = {
            'ctime': CtimeFilter,
            'visible': VisibleFilter,
            'name': NameFilter,
            'public': PublicFilter,
            'media_type': MediatypeFilter,

            'resource_type': ResourceType,
            'mtime_gte': MtimeGreaterThenOrEqual,
            'mtime_lte': MtimeLowerThenOrEqual,
            'mtime_interval': MtimeInterval,
            'modify_uid': ModifyUid,
        }
        self.empty_filters = ()

        self.available_bounds = {
                                'sort' : 'sort',
                                'order' : 'order',
                                'amount' : 'limit',
                                'offset' : 'skip',
                                }

        self.available_sort_fields = {
                                      'ctime' : 'data.mtime',
                                      'mtime' : 'data.mtime',
                                      'utime' : 'data.utime',
                                      'etime' : 'data.etime',
                                      'size' : 'data.size',
                                      'name_natural' : 'name',
                                      'name': 'key',
                                      'key': 'key',
                                      }

    def get_data_for_uid(self, uid, doc):
        data = self.control.unpack_single_element(doc)
        data['wh_version'] = str(self.control.get_cached_version(uid))
        return data

    def get_resource(self, uid, address, version=None, **kwargs):
        resp = self.value(address, version)

        if not resp.ok():
            raise errors.ResourceNotFound(address.id)

        resource_data = resp.value.data
        resource_version = resp.value.version
        resource_type = resp.value.type
        resource_data['meta'] = resource_data.get('meta', {})
        resource_data['version'] = resource_version

        # временно закрываю пока не сделают https://jira.yandex-team.ru/browse/CHEMODAN-6032
        #if address.is_folder and resp.value.type != 'dir':
        #    raise errors.NotFolder()

        cls = self.resources[resource_type]
        return cls(uid, address, data=resource_data, version=resp.version, **kwargs)

    def _replace_sort_by_append_time(self, resource, uid, path, range_args):
        if not resource.form.args:
            return

        form_bounds = resource.form.args['bounds']
        if 'sort' in form_bounds and form_bounds['sort'] == 'append_time':
            from mpfs.metastorage.mongo.util import id_for_key
            if is_new_fs_spec_required(self.control.name):
                parent = path
            else:
                parent = id_for_key(uid, path)
            total_items_count = self.control.count(uid=uid, parent=parent)

            if total_items_count > SYSTEM_SYSTEM_TRASH_SORT_BY_APPEND_TIME_LIMIT:
                # не сортируем по append_time, сортируем по key вместо этого и добавляем skip и limit в bounds
                bounds = {}
                unprocessed_bounds = {}
                for k, v in form_bounds.iteritems():
                    if k not in self.available_bounds:
                        unprocessed_bounds[k] = v
                        continue

                    if k == 'sort' or k == 'order':
                        continue
                    else:
                        bounds[self.available_bounds[k]] = v

                bounds[self.available_bounds['sort']] = self.available_sort_fields['key']
                bounds[self.available_bounds['order']] = 1

                range_args.update(bounds)
                resource.form.args['bounds'] = unprocessed_bounds

    def folder_content(self, resource, timeline=None):
        '''
        Загружает частичное содержимое папки из метастораджа
        '''
        super(MPFSStorageService, self).folder_content(resource)
        uid, path = resource.address.uid, resource.address.path
        filter_args, range_args = self.get_request_parameters()

        max_sample_size = None
        if resource.form.args:
            max_sample_size = resource.form.args['bounds'].pop(
                'max_sample_size', None
            )

        if resource.address.storage_name == 'trash':
            # https://st.yandex-team.ru/CHEMODAN-26546
            self._replace_sort_by_append_time(resource, uid, path, range_args)

        # https://jira.yandex-team.ru/browse/CHEMODAN-17852
        if resource.request and hasattr(resource.request, 'meta') and \
                resource.request.meta and 'numchildren' in resource.request.meta:
            resource.meta['numchildren'] = self.control.folder_content_count(uid, path)

        ycrid = mpfs.engine.process.get_cloud_req_id()
        if FEATURE_TOGGLES_CHECK_DAV_MAX_FILES_LISTING_LIMIT and 'limit' not in range_args and \
                        YcridParser.get_platform(ycrid) == 'dav':
            total_len = self.control.folder_content_count(uid, path)
            if total_len > DAV_MAX_FILES_LISTING_LIMIT:
               raise errors.TooManyFilesInListingError()

        response = self.control.folder_content(
            uid, path, None, filter_args, range_args,
            max_sample_size=max_sample_size
        )

        resource.meta['wh_version'] = response.version
        resource.meta['total_results_count'] = response.total_results_count

        if not response.value:
            return []
        else:
            def fill_res(args):
                item_name = args[0]
                item = args[1].data
                item['static_name'] = item.pop('name', '')
                item['name'] = filter(None, args[1].key.split('/'))[-1]
                item['key'] = args[1].key
                item['meta'] = item.get('meta') or {}
                item['meta'].pop('wh_version', '')
                item['version'] = args[1].version
                item.pop('id', '')
                return item

            resource.hasfolders = 1 if resource.child_folders else 0
            children_map = imap(fill_res, response.value.iteritems())
            return self.process_children(resource, children_map)

    def process_children(self, parent, resources, timeline=False):
        result = []

        for item in resources:
            if not item.get('type'):
                log.warn('Bad resource \n%s' % '\n'.join('%s = %s' %
                    (str(k.encode('utf-8')), str(v.encode('utf-8')))
                    for k, v in item.iteritems() if isinstance((k, v), unicode)))
                continue

            if not 'meta' in item:
                item['meta'] = {}

            try:
                del item['static_name']
            except KeyError:
                pass

            item_name = item['name']
            if item['type'] == 'dir':
                item_id = item['id'] = parent.address.get_child_folder(item_name).path
                parent.sorted_folders.append(item_id)
                parent.child_folders[item_id] = item
                parent.construct_child_folder(item_id, item, parent.version, use_key=timeline)
            elif item['type'] == 'file':
                item_id = item['id'] = parent.address.get_child_file(item_name).path
                item['size'] = int(item.get('size') or 0)
                if 'source' not in item:
                    item['source'] = parent.address.storage_name
                parent.sorted_files.append(item_id)
                parent.child_files[item_id] = item
                parent.construct_child_file(item_id, item, parent.version)

            result.append(item)

        self._update_views_counter(result)

        return result

    @staticmethod
    def _update_views_counter(resources):
        """Обновить счетчики показа для ресурсов из `resources`

        :type resources: list[:class:`~Resource` | dict]
        :rtype: None
        """
        if not resources:
            return

        get_meta = attrgetter('meta')
        if isinstance(resources[0], dict):
            get_meta = itemgetter('meta')

        hashes = [get_meta(r)['public_hash'] for r in resources if get_meta(r).get('public_hash')]
        if not hashes:
            return

        counters = logreader.get_counters(hashes)
        for r in resources:
            public_hash = get_meta(r).get('public_hash')
            if public_hash:
                get_meta(r)[VIEWS_COUNTER] = counters.get(public_hash, 0)

    def set_last_files(self, parent):
        """Установить последние файлы для ``parent`` в том числе из его
        общих подпапок.

        Форматтер сериализует ``parent`` и его доч. файлы в виде ответа.

        :type parent: :class:`~DiskFolder`
        :rtype: None
        """
        # цикл. импорт
        from mpfs.core.social.share import ShareProcessor
        from mpfs.core.filesystem.resources.disk import DiskFolder, DiskFile

        uid, path = parent.address.uid, parent.address.path
        if not isinstance(parent, DiskFolder):
            raise TypeError("`DiskFolder` is required, got `%s` instead" % type(parent))

        super(MPFSStorageService, self).folder_content(parent)

        _, range_args = self.get_request_parameters()
        committer = parent.committer  # по идее тоже, что и uid

        limit = range_args['limit']
        if not isinstance(limit, int):
            raise TypeError("`int` is required, got `%s` instead" % type(limit))

        if not limit > 0:
            raise ValueError('`limit` must be > 0')

        now = int(time.time())

        def prepare_merge(link_iterable):
            """Подготовить данные для упорядоченного слияния.

            :type link_iterable: tuple[None | :class:`~LinkToGroup`, collections.Iterable[tuple]]
            :rtype: collections.Iterable[tuple]
            """
            link, iterable = link_iterable
            for key, data in iterable:
                cmp_key = now - int(data['mtime'])  # инвертируем порядок для :meth:`~heapq.merge`
                yield (cmp_key, link, key, data)

        links = [None]  # disk/
        if mpfs.engine.process.use_shared_folders():
            owned_links = ShareProcessor().list_owned_links(committer)
            links.extend(l for l in owned_links if l.path.startswith(path + '/'))

        args = []
        collection = self.control.db[self.control.name]
        args.append((collection, uid, path))
        for link in links[1:]:
            args.append((collection, link.group.owner, link.group.path))

        cursors = self.control.get_last_files(limit, args)
        merged = merge(*map(prepare_merge, izip(links, cursors)))

        files = []
        for i, item in enumerate(merged):
            if i >= limit:
                break

            _, link, key, data = item
            if link:
                link_key = link.get_link_path(key)
                data['key'] = link_key  # хак какой-то, но без этого не работает
                address = GroupAddress('%s:%s' % (committer, link_key), '%s:%s' % (link.group.owner, key))
                file_ = DiskFile(committer, address, data=data, link=link)
            else:
                data['key'] = key  # хак какой-то, но без этого не работает
                address = Address('%s:%s' % (uid, key))
                file_ = DiskFile(uid, address, data=data)

            file_.set_request(parent.request)
            files.append(file_)

        self._update_views_counter(files)
        # Форматтер заберет результат именно отсюда
        parent.children_items['files'].extend(files)

    def timeline(self, resource, folders=False):
        '''
        Загружает таймлайн из метастораджа
        '''

        self.available_sort_fields.pop('name')
        bounds = copy.deepcopy(resource.form.args['bounds'])
        super(MPFSStorageService, self).folder_content(resource)

        if bounds.get('sort', None) == 'ctime':
            # костыль имени self.available_sort_fields, а именно 'ctime' : 'data.mtime',
            # сделано для консистентности, а то из базы выгребаем по mtime, а дальше форматтер сортирует по ctime,
            # в результате получаем такую проблему: https://st.yandex-team.ru/CHEMODAN-29937
            bounds['sort'] = 'mtime'

        resource_children = []
        filter_args, range_args = self.get_request_parameters()
        uid, path = resource.address.uid, resource.address.path
        version = resource.meta.get('wh_version')

        # получаем шаренные файлы
        from mpfs.core.filesystem.resources.disk import DiskResource
        if mpfs.engine.process.use_shared_folders() and isinstance(resource, DiskResource):
            from mpfs.core.filesystem.resources.disk import DiskFile, DiskFolder
            from mpfs.core.social.share import ShareProcessor
            for link in ShareProcessor().list_owned_links(resource.committer):
                if not link.path.startswith(path + '/'):
                    continue
                group_owner_timeline = self.control.timeline(link.group.owner, link.group.path, version, filter_args, range_args, folders=folders)
                for document in group_owner_timeline.value:
                    key = link.get_link_path(document.key)
                    document.data['version'] = document.version
                    address = GroupAddress('%s:%s' % (resource.committer, key), '%s:%s' % (link.group.owner, document.key))
                    if document.data['type'] == 'dir':
                        new_resource = DiskFolder(resource.committer, address, data=document.data, link=link, version=version)
                    elif document.data['type'] == 'file':
                        new_resource = DiskFile(resource.committer, address, data=document.data, link=link, version=version)
                    data = new_resource.dict()
                    data['key'] = key
                    resource_children.append(data)

        if range_args:
            # для полей из available_sort_fields делаем особую выборку и обрезку
            num_shared_files = len(resource_children)
            offset, limit = range_args.get('skip', 0), range_args.get('limit', -666)
            if offset < num_shared_files:
                range_args['skip'] = 0
                range_args['limit'] = offset + limit
            else:
                range_args['skip'] = offset - num_shared_files
                range_args['limit'] = num_shared_files + limit
            if limit == -666:
                range_args.pop('limit')

            # дополнительно обрезаем на стороне frontend
            if offset > num_shared_files:
                bounds['offset'] = num_shared_files
            resource.form.args['bounds'] = bounds
        #получаем свои файлы
        response_user = self.control.timeline(uid, path, version, filter_args, range_args, folders=folders)

        resource.meta['wh_version'] = response_user.version

        if not (response_user.value or resource_children):
            return []

        def fill_res(args):
            item = args.data
            item.get('meta', {}).pop('wh_version', '')
            item.pop('id', '')
            item.update({
                         'static_name' : item.pop('name', ''),
                         'name' : filter(None, args.key.split('/'))[-1],
                         'key' : args.key,
                         'meta' : item.get('meta') or {},
                         'version' : args.version,
                         })
            return item

        resource_children.extend(map(fill_res, response_user.value))

        resource.hasfolders = 1 if resource.child_folders else 0
        return self.process_children(resource, resource_children, timeline=True)

    def tree(self, resource, **params):
        level = params.get('level', 1)

        response = self.control.tree(resource.uid, resource.id, level)

        version = resource.meta['wh_version'] = response.version

        key_lists = defaultdict(list)
        from mpfs.core.filesystem.resources.disk import DiskResource
        if mpfs.engine.process.use_shared_folders() and isinstance(resource, DiskResource):
            resources = {resource.id: resource.dict()}
            from mpfs.core.filesystem.resources.disk import DiskFile, DiskFolder
            from mpfs.core.social.share import ShareProcessor
            for key, document in response.value.iteritems():

                address = Address('%s:%s' % (resource.committer, key))
                if document.version is not None:
                    document.data['version'] = document.version
                if document.data['type'] == 'dir':
                    new_resource = DiskFolder(resource.committer, address, data=document.data, version=version)
                    resource.hasfolders = 1
                    new_resource.hasfolders = int(self.control.has_subfolders(new_resource.address.uid, new_resource.address.path))
                elif document.data['type'] == 'file':
                    new_resource = DiskFile(resource.committer, address, data=document.data, version=version)
                parent_key = document.parent_key
                data = new_resource.dict()
                key_lists[parent_key].append(data)
                resources[key] = data

            def process_shared(response, link):
                for real_key, document in response.value.iteritems():
                    key = link.get_link_path(real_key)
                    parent_key = link.get_link_path(document.parent_key)
                    address = GroupAddress('%s:%s' % (resource.committer, key), '%s:%s' % (link.group.owner, real_key))
                    if document.data['type'] == 'dir':
                        new_resource = DiskFolder(resource.committer, address, data=document.data, link=link,
                                                  version=version)
                        resource.hasfolders = 1
                        new_resource.hasfolders = int(self.control.has_subfolders(new_resource.address.uid, new_resource.address.path))
                    elif document.data['type'] == 'file':
                        new_resource = DiskFile(resource.committer, address, data=document.data, link=link,
                                                version=version)
                    data = new_resource.dict()
                    key_lists[parent_key].append(data)
                    resources[key] = data

            from mpfs.core.filesystem.resources.share import SharedFolder, SharedRootFolder
            if isinstance(resource, SharedRootFolder):
                response = self.control.tree(resource.link.group.owner, resource.link.group.path, level)
                process_shared(response, resource.link)
            elif isinstance(resource, SharedFolder):
                response = self.control.tree(resource.address.uid, resource.address.path, level)
                process_shared(response, resource.link)
            else:
                for link in ShareProcessor().list_owned_links(resource.committer):
                    is_parent_resource = link.path.startswith(resource.id + '/')
                    if is_parent_resource:
                        deep_level = level - (len(link.path.split('/')) - len(resource.id.split('/')))
                        if deep_level:
                            response = self.control.tree(link.group.owner, link.group.path, deep_level)
                            process_shared(response, link)
        else:
            resources = {resource.id: resource.dict()}
            for key, document in response.value.iteritems():
                item = document.data
                item.get('meta', {}).pop('wh_version', '')
                item.update({
                    'name'       : filter(None, key.split('/'))[-1],
                    'id'         : key,
                    'key'        : key,
                    'uid'        : resource.uid,
                    'meta'       : item.get('meta') or {},
                    'version'    : version,
                })

                parent_key = document.parent_key
                key_lists[parent_key].append(item)
                resources[key] = item

        def recurse_resource(res):
            children = []
            for item in sorted(key_lists[res], key=itemgetter('name')):
                children.append(recurse_resource(item['id']))
            resources[res]['hasfolders'] = resources[res]['hasfolders'] or int(bool(len(children)))
            return { 'this': resources[res], 'list': children }

        return recurse_resource(resource.id)

    def get_request_parameters(self):
        range_args = {}
        for k, v in self.request_processing_fields['bounds'].iteritems():
            if k == 'order':
                v = (-1, 1)[v]
            range_args[k] = v
        filter_args = {}
        for k, v in self.request_processing_fields['filters'].iteritems():
            filter_args.update(self.available_filters[k](v).get())
        return filter_args, range_args

    def value(self, address, version=None):
        """
        Получение значения по ключу
        """
        resp = self.control.show(address.uid, address.path, version)
        if resp.value:
            resp.value.data.pop('id', '')
            resp.value.data.pop('name', '')
        return resp

    def values(self, uid, addresses):
        keys = map(lambda x: x.path, addresses)
        response = self.control.show(uid, keys)
        return response

    def show_single(self, address, version=None):
        if isinstance(address, tuple):
            uid, path = address
        else:
            uid = address.uid
            path = address.path
        return self.control.show_single(uid, path, version)

    def put(self, address, val, version=None, **kwargs):
        '''
        Внесение значения в хранилище
        '''
        return self.control.put(address.uid, address.path, val,
                                version, **kwargs)

    def remove(self, address, version=None, data=None):
        '''
        Удаление значение
        '''
        return self.control.remove(address.uid, address.path, version, data=data)

    def remove_single(self, address, version=None, changelog=True, data=None):
        '''
            address = (uid, path) or Address()
        '''
        if isinstance(address, tuple):
            uid, path = address
        else:
            uid  = address.uid
            path = address.path
        if changelog:
            return self.control.remove_single(uid, path, version, data=data)
        else:
            return self.control.remove_single_without_changelog(uid, path, version)

    def rename(self, src, dst):
        '''
        Переименование ресурса
        '''
        return self.control.move(src.uid, src, dst)

    def update_symlinks(self, src, dst):
        uid = src.uid
        source_id = src.id
        for element in link_data.get_all(uid=uid):
            data = element['data']
            if data and 'tgt' in data:
                tgt = data['tgt']
                if source_id == tgt:
                    new_tgt = dst
                elif tgt.startswith(source_id + '/'):
                    new_tgt = Address(tgt)
                    new_tgt.change_parent(dst)
                else:
                    continue
                symlink = Symlink(Symlink.address_from_key(uid, element['key']), data=data)
                symlink.set_target(new_tgt)

    def make_folder(self, address, parent, data={}):
        """
        Создание папки по родителю

        :param address:
        :param parent:
        :param data: Исходные данные создаваемой папки. InOut аргумент.
        """
        value = {
            'visible': 1,
            'meta': {},
        }
        value.update(copy.deepcopy(data))
        # override type even if it's presented in data
        value['type'] = 'dir'

        creation_time = int(time.time())
        for k in ('ctime', 'mtime', 'utime',):
            value[k] = data.get(k, creation_time)
            # FIXME: Костыль ради получения mtime и ctime в mpfs.core.filesystem.resources.base.Folder.Create()
            # Из опасений поломать что-то глобальное вынося установку этих параметров в
            # mpfs.core.filesystem.resources.base.Folder.Create() и воизбежание лишнего запроса
            # к metastorage в методе mpfs.core.filesystem.resources.base.Folder.Create()
            # параметр data заполняется данными здесь.
            data[k] = value[k]

        return self.control.make_folder(address.uid, address.path, value)

    def make_storage_folder(self, address):
        return self.control.make_folder(address.uid, address.path, {})

    def edit_folder(self, address, value, **kwargs):
        '''
        Изменение полей каталога
        '''
        return self.control.change_folder(address.uid, address.path, value, **kwargs)

    def change_folder_fields(self, address, data, version=None):
        return self.control.change_folder_fields(address.uid, address.path,
                                                 data, version=version)

    def edit_file(self, address, val, version, **kwargs):
        return self.put(address, val, version, **kwargs)

    def list(self, address, version=None):
        return self.control.list(address.uid, address.path, version)

    def diff(self, address, version):
        return self.control.diff(address.uid, address.path, version)

    def used(self, uid):
        resp = disk_info.value_for_quota(uid, 'total_size')
        try:
            size = int(resp.value.data)
        except (ValueError, AttributeError):
            size = 0

        ignore_shared_folder = ignores_shared_folders_space(uid)
        if mpfs.engine.process.use_shared_folders() and not ignore_shared_folder:
            self._preload_all_shared_data(uid)
            from mpfs.core.social.share import LinkToGroup
            for link in LinkToGroup._instances['uid_path'].itervalues():
                if link.uid == uid:
                    size += link.group.size

        if size < 0:
            size = 0
        return int(size)

    def limit(self, uid):
        # Для b2b пользователей надо учитывать место в организации
        # https://st.yandex-team.ru/CHE-122
        if B2B_SHARED_ORGANIZATION_SPACE_ENABLED:
            from mpfs.core.user.base import User
            organization_id = User(uid).b2b_key
            if organization_id:
                organization = OrganizationDAO().find_by_id(organization_id)
                if organization and organization.is_paid:
                    used = self.used(uid)
                    from mpfs.core.organizations.logic import organization_user_bought_space
                    bought = organization_user_bought_space(uid)
                    return max(organization.quota_free, 0) + max(bought - used, 0) + used

        resp = disk_info.value_for_quota(uid, 'limit')
        if not resp.value:
            size = self.default_limit
        else:
            size = int(resp.value.data)
        return int(size)

    def files_count(self, uid):
        resp = disk_info.value(uid, 'files_count')
        if resp.value:
            val = resp.value.data
        else:
            val = resp.value
        return int(val or 0)

    def count_resources(self, uid):
        """
        Count resources for UID.
        :param uid:
        :return:
        """
        return self.control.count(uid)

    def _increment_db_counter(self, uid, key, diff):
        try:
            diff = int(diff)
            if diff != 0:
                disk_info.cache_used_space(uid, key, diff)
                after = int(disk_info.value(uid, key).value.data)
                log.info('SPACE %s %s: diff %s, result %s' % (uid, key, diff, after))
        except Exception:
            error_log.error('SPACE fixation error for %s %s' % (uid, key))
            error_log.error(traceback.format_exc())
            recount.add(uid)

    def free(self, uid):
        return self.limit(uid) - self.used(uid)

    def set_limit(self, uid, val):
        disk_info.put(uid, 'limit', val, None)

    def change_limit_on_delta(self, uid, delta_limit):
        disk_info.increment(uid, 'limit', delta_limit, 'data')

    def update_where(self, address, new_data, old_data, version):
        return self.control.update_where(address.uid, address.path, new_data, old_data, version)

    def url(self, uid, mid, name, **kwargs):
        return zaberun.generate_file_url(uid, mid, name, **kwargs)

    def folder_url(self, uid, mid, name, **kwargs):
        return zaberun.generate_folder_url(uid, mid, name, **kwargs)

    def preview_url(self, uid, mid, filename, **kwargs):
        return zaberun.generate_preview_url(uid, mid, filename, **kwargs)

    def public_download_url(self, mid, filename, **kwargs):
        return zaberun.generate_public_url(mid, filename, **kwargs)

    def direct_url(self, mid):
        return mulca.get_local_url(mid)

    def get_version(self, uid):
        return self.control.get_version(uid)

    def flat_index(self, uid, path, mediatype=None):
        resp = self.control.flat_index(uid, path, mediatype=mediatype)
        return resp.value, resp.version

    def tree_index(self, uid, path):
        return self.control.tree_index(uid, path)

    def lock_set(self, uid, path, version=None):
        return self.control.resource_lock(uid, path, version=version)

    def lock_unset(self, uid, path, version=None):
        return self.control.resource_unlock(uid, path, version=version)

    def lock_check(self, uid, path, version=None):
        return self.control.is_resource_locked(uid, path, version=version)

    def lock_list(self, uid, version=None):
        return self.control.show_locks(uid, version=version)

    def remove_folder_content(self, uid, path):
        return self.control.remove_folder_content(uid, path)

    def get_all_files(self, uid):
        return self.control.find_by_field(uid, args={'type' : 'file'})

    def _preload_all_shared_data(self, uid):
        from mpfs.core.social.share import Group
        from mpfs.core.social.share import LinkToGroup
        Group.load_all(uid)
        LinkToGroup.load_all(uid)

    def _get_resources(self, uid, addresses):
        result = []
        response = self.values(uid, addresses)
        version = response.version
        elements = dict((v.key, v) for v in ifilter(lambda x: x is not None, response.value))
        for address in addresses:
            resource = None
            try:
                resource_document = elements[address.path]
            except Exception:
                if mpfs.engine.process.use_shared_folders():
                    try:
                        resource = self.find_shared(uid, address)
                    except Exception:
                        pass
            else:
                resource_version = resource_document.version
                resource_type = resource_document.type
                resource_data = resource_document.data
                resource_data['meta'] = resource_data.get('meta', {})
                resource_data['version'] = resource_version
                cls = self.resources[resource_type]
                if mpfs.engine.process.use_shared_folders():
                    self._preload_all_shared_data(uid)
                    from mpfs.core.social.share import Group
                    try:
                        group = Group.find(uid=uid, path=address.path, link=False, group=True)
                    except Exception:
                        try:
                            link = Group.find(uid=uid, path=address.path, link=True, group=False)
                        except Exception:
                            resource = cls(uid, address, data=resource_data, version=version)
                        else:
                            resource = cls(uid, address, data=resource_data, version=version, link=link)
                    else:
                        resource = cls(uid, address, data=resource_data, version=version, group=group)
            if resource:
                result.append(resource)
            else:
                result.append(None)
        return result

    def get_resources(self, uid, addresses):
        """
        Возвращает список ресурсов без ненайденных ресурсов

        :param uid: string c uid юзера
        :param addresses: list адресов
        :return: list ресурсов
        """
        resources = self._get_resources(uid, addresses)
        return filter(lambda x: x is not None, resources)

    def get_resources_with_dummies(self, uid, addresses, dummies):
        """
        Возвращает список ресурсов с заглушками вместо ненайденных
        Данные для заглушек берет "как есть" из dummies

        :param uid: string c uid юзера
        :param addresses: list адресов
        :param dummies: list заглушек
        :return: list ресурсов
        """
        resources = self._get_resources(uid, addresses)

        for i in xrange(len(resources)):
            resource = resources[i]
            if resource is None:
                resources[i] = dummies[i]

        return resources

    def increment_download_counter(self, resource, count):
        field = 'data.download_counter'
        spec = {
                field : {'$exists' : 1},
                }
        uid = resource.storage_address.uid
        path = resource.storage_address.path
        return self.control.increment(uid, path, count, field, spec=spec, verbose=True)

    def find_snapshot_chunk(self, uid, file_id, last_ids, last_file_type, limit):
        """
        :rtype: list[tuple[str, dict]]
        """
        chunk = []
        if not file_id:
            cursor = self.control.find_snapshot_initial_chunk(uid)
            chunk.extend(cursor)
            file_id = ''

        cursor = self.control.find_snapshot_chunk(uid, file_id, last_ids, last_file_type, limit)
        chunk.extend(cursor)

        return self.get_unpacked_documents_with_fixed_file_id(chunk)

    def find_snapshot_from_subtree(self, uid, parent, time_limit, items_limit=1000, last_folder_file_id=None):
        """
        Обходит папку итеративно по первому уровню.

        Сперва обходит папки, в последней итерации добавляет все файлы первого подуровня и корневую папку. Если
        встретятся папки без file_id, то их тоже гарантированно подцепим в поселней итерации вместе с файлами.
        Второе возвращаемое значения - последний обработанный file_id. Если обошли все, то значение равно None

        :param last_folder_file_id: последний file_id папки, который мы обошли. При первой итерации должен быть None
        :rtype: list[tuple[str, dict]], str
        """
        all_children = self.control.get_immediate_children(uid, parent)
        file_id_to_path_map = defaultdict(list)
        last_folder_file_id = last_folder_file_id or ''
        for i in all_children:
            if i['type'] != 'dir':
                continue
            file_id = i['data'].get('file_id', '')
            if file_id <= last_folder_file_id:
                continue
            file_id_to_path_map.setdefault(file_id, []).append(i['key'])
        child_folders_tuples_sorted = sorted(file_id_to_path_map.iteritems())

        chunk = []

        def extend_with_resources_from_subfolder(subfolder_path):
            for doc in self.control.iter_subtree(uid, subfolder_path):
                if time.time() - start > time_limit:
                    raise SnapshotSharedFoldersTimeOut()
                chunk.append(doc)
            chunk.append(self.control.find_one_by_field(uid, {'key': subfolder_path}))

        start = time.time()
        for file_id, folder_paths in child_folders_tuples_sorted:
            for folder_path in folder_paths:
                extend_with_resources_from_subfolder(folder_path)
            if len(chunk) >= items_limit:
                return self.get_unpacked_documents_with_fixed_file_id(chunk), file_id

        for folder_path in file_id_to_path_map.get('', []):
            extend_with_resources_from_subfolder(folder_path)

        chunk.extend([x for x in all_children if x['type'] == 'file'])
        return self.get_unpacked_documents_with_fixed_file_id(chunk), None

    def get_unpacked_documents_with_fixed_file_id(self, docs):
        """Распаковывает зазипованные данные из базы и добавляет `file_id`
        если он отсутсвует.

        Это костыль для тех записей базы, в которых нет `file_id`.
        Наличие `file_id` обязятаельно для записей базы, но не всегда
        это требование выполняется.

        :type docs: list[dict]
        :rtype: list[tuple[str, dict]]
        """

        unpacked_docs = []
        for doc in docs:
            doc_id = doc['_id']
            doc = self.control.unpack_single_element(doc)
            if 'file_id' not in doc['data'].get('meta', {}):
                doc['data']['meta'] = {
                    'file_id': Resource.generate_file_id(doc['uid'], doc['key'])
                }
            unpacked_docs.append((doc_id, doc))
        return unpacked_docs

    #===========================================================================
    # Теги CHEMODAN-14569
    def set_tags(self, uid, file_id, scope, data):
        return tag_data.put(uid, file_id, scope, data)

    def elements_for_tag(self, uid, path, tags):
        return tag_data.get_elements_by_tag(uid, path, tags)

    def tags_for_element(self, uid, file_id):
        return tag_data.get_tags_by_fid(uid, file_id)

    def tags_in_folder_list(self, uid, path):
        return tag_data.get_tags_in_folder_list(uid, path)

    def tags_in_folder_tree(self, uid, path):
        return tag_data.get_tags_in_folder_tree(uid, path)

    def elements_in_folder_list(self, uid, path, data):
        return tag_data.get_elements_in_folder_list(uid, path, data)

    def elements_in_folder_tree(self, uid, path, data):
        return tag_data.get_elements_in_folder_tree(uid, path, data)

    def tags_photo_timeline(self, uid, system_tags, user_tags):
        return tag_data.tags_photo_timeline(uid, system_tags, user_tags)

    def tags_photo_list(self, parent, filters, system_tags, user_tags):
        result = []
        for item in tag_data.tags_photo_list(parent.storage_address.uid, filters, system_tags, user_tags):
            if 'source' not in item:
                item['source'] = parent.address.storage_name
            result.append(parent.construct_child_file(item['key'], item['data'], item['version']).dict())
        return result

    def tags_set_photo_all(self, uid):
        tag_data.tags_set_photo_all(uid)

    def tags_set_photo_file(self, uid, file_id, etime, camera, geo):
        tag_data.tags_set_photo_file(uid, file_id, etime, camera, geo)

    def tags_remove_single(self, uid, file_id):
        """
            Удаление всех тегов для 1 файла.
        """
        tag_data.tags_remove_single(uid, file_id)
    #===========================================================================


class StorageSpaceLimited(MPFSStorageService):

    space_counter_key = None
    space_counter_groups = ()

    def cache_delta(self, uid, value):
        disk_info.cache_used_delta(uid, self.space_counter_key, value)

    def flush_delta(self, uid, value):
        disk_info.flush_used_delta(uid, self.space_counter_key, value)

    def update_counter(self, uid, value, version=None):
        from mpfs.core.filesystem.helpers.counter import Counter
        for g in self.space_counter_groups:
            Counter().add(g, uid, value)

        if B2B_SHARED_ORGANIZATION_SPACE_ENABLED:
            from mpfs.core.user.base import User
            organization_id = User(uid).b2b_key
            if organization_id:
                from mpfs.core.organizations.dao.organizations import OrganizationDAO
                organization = OrganizationDAO().find_by_id(organization_id)
                if organization and organization.is_paid:
                    from mpfs.core.organizations.logic import calculate_organization_used_space_async
                    calculate_organization_used_space_async(organization_id)


class StorageSpaceUnlimited(MPFSStorageService):
    pass


class Disk(StorageSpaceLimited):

    name = 'disk'
    control = disk
    space_counter_key = 'total_size'
    space_counter_groups = ('disk',)

    def __init__(self, *args, **kwargs):
        super(Disk, self).__init__(*args, **kwargs)
        from mpfs.core.filesystem.resources.disk import DiskFile, DiskFolder
        self.resources = Resources(dir=DiskFolder, file=DiskFile)

    def get_resource(self, uid, address, version=None, **kwargs):
        resp = self.value(address, version)
        if resp.ok():
            resource_data = resp.value.data
            resource_version = resp.value.version
            resource_type = resp.value.type
            resource_data['meta'] = resource_data.get('meta', {})
            resource_data['version'] = resource_version

            # временно закрываю пока не сделают https://jira.yandex-team.ru/browse/CHEMODAN-6032
            #if address.is_folder and resp.value.type != 'dir':
            #    raise errors.NotFolder()

            cls = self.resources[resource_type]
            if mpfs.engine.process.use_shared_folders():
                self._preload_all_shared_data(uid)
                from mpfs.core.social.share import Group
                try:
                    group = Group.find(uid=address.uid, path=address.path, link=False, group=True)
                except Exception:
                    try:
                        link = Group.find(uid=address.uid, path=address.path, link=True, group=False)
                    except Exception:
                        resource = cls(uid, address, data=resource_data, version=resp.version, **kwargs)
                    else:
                        resource = cls(uid, address, data=resource_data, version=resp.version, link=link, **kwargs)
                else:
                    resource = cls(uid, address, data=resource_data, version=resp.version, group=group, **kwargs)
            else:
                resource = cls(uid, address, data=resource_data, version=resp.version, **kwargs)
            return resource
        else:
            if mpfs.engine.process.use_shared_folders():
                return self.find_shared(uid, address, version)
            else:
                raise errors.ResourceNotFound(address.id)

    def find_shared(self, uid, address, version=None):
        link = None
        from mpfs.core.filesystem.resources.share import SharedFile, SharedFolder
        template = {
                    'dir'  : SharedFolder,
                    'file' : SharedFile,
                    }
        from mpfs.core.social.share import LinkToGroup
        LinkToGroup.load_all(uid)

        address_split = filter(None, address.path.split('/'))
        address_len = len(address_split)
        # We don't need this element, root and /disk
        for i in xrange(address_len - 1, 1, -1):
            path = '/' + '/'.join(address_split[:i])
            try:
                link = LinkToGroup.load(uid=address.uid, path=path)
            except ShareNotFound:
                pass
            else:
                owner_group_path = link.group.path
                self_group_path = len(filter(None, link.path.split('/')))
                real_path_split = filter(None, owner_group_path.split('/') + filter(None, address_split)[self_group_path:])
                real_path = '/' + '/'.join(real_path_split)
                address = GroupAddress(address.id, '%s:%s' % (link.group.owner, real_path))
                resp = self.value(address, version)
                if resp.ok():
                    resource_type = resp.value.type
                    cls = template[resource_type]
                    resource_data = resp.value.data
                    resource_version = resp.value.version
                    resource_type = resp.value.type
                    resource_data['meta'] = resource_data.get('meta', {})
                    resource_data['version'] = resource_version
                    if ENABLE_WH_VERSION_FIX:
                        return cls(uid, address, data=resource_data, version=resp.version, link=link)
                    return cls(uid, address, data=resource_data, link=link)
        raise errors.ResourceNotFound(address.id)

    def owners_folder_created(self, resource):
        return changelog.owners_folder_created(resource.uid, resource.group.path, resource.dict(safe=True), 660)

    def users_folder_created(self, resource):
        return changelog.users_folder_created(resource.uid, resource.link.path, resource.dict(safe=True), resource.link.rights)

    def users_folder_updated(self, resource):
        return changelog.users_folder_updated(resource.uid, resource.link.path, resource.dict(safe=True), resource.link.rights)

    def folder_unshared(self, resource):
        try:
            rights = resource._link.rights
        except AttributeError:
            rights = 660
        return changelog.folder_unshared(resource.uid, resource.id, resource.dict(safe=True), rights)
