# -*- coding: utf-8 -*-
from mpfs.common.errors import DjfsError, DjfsAlbumsProxyError, DjfsAlbumsProxyNotImplementedError
from mpfs.common.util import from_json
from mpfs.core.services.common_service import Service, RequestsPoweredServiceBase

MPFS_ERROR_CODE_HEADER_NAME = 'X-MPFS-Legacy-Error-Code'
DJFS_NOT_IMPLEMENTED_HTTP_CODE = 422  # special not implemented code


class DjfsAlbumsLegacyService(Service):
    name = 'DjfsAlbumsLegacyService'
    api_error = DjfsError

    def exclude_photo_from_geo_album(self, uid, path):
        params = {'uid': uid, 'path': path}
        url = self.base_url + '/api/v1/albums/exclude_photo_from_geo_album'
        return self.open_url(url, params, method='POST')

    def album_append_items(self, params, meta, body):
        if meta is not None:
            params['meta'] = ','.join(meta)
        url = self.base_url + '/api/v1/albums/album_append_items'
        return self.open_url(url, params, pure_data=body, method='POST',
                             headers={"Content-type": "application/json"})

    def album_item_remove(self, params):
        url = self.base_url + '/api/v1/albums/album_item_remove'
        return self.open_url(url, params, method='GET')


class DjfsAlbumsService(RequestsPoweredServiceBase):
    api_error = DjfsError

    def __init__(self):
        super(DjfsAlbumsService, self).__init__()
        self.send_cloud_request_id = True

    def _exception_from_response(self, response):
        if MPFS_ERROR_CODE_HEADER_NAME in response.headers:
            error_code = int(response.headers[MPFS_ERROR_CODE_HEADER_NAME])
            json = from_json(response.content)

            data = None
            if isinstance(json, dict):
                data = json.get('data')
            if data is None:
                data = {'code': response.status_code}

            return DjfsAlbumsProxyError(error_code, data)
        if response.status_code == DJFS_NOT_IMPLEMENTED_HTTP_CODE:
            return DjfsAlbumsProxyNotImplementedError()
        return super(DjfsAlbumsService, self)._exception_from_response(response)

    def album_append_items(self, params, meta, body):
        if meta is not None:
            params['meta'] = ','.join(meta)
        response = self.request('POST', '/api/v1/albums/album_append_items', params, data=body,
                                headers={'Content-Type': 'application/json'})
        return from_json(response.content)

    def album_item_remove(self, params):
        self.request('DELETE', '/api/v1/albums/album_item_remove', params)

    def album_remove(self, params):
        self.request('DELETE', '/api/v1/albums/album_remove', params)

    def album_set_attr(self, params, meta):
        if meta is not None:
            params['meta'] = ','.join(meta)
        response = self.request('PATCH', '/api/v1/albums/album_set_attr', params)
        return from_json(response.content)

    def album_publish(self, params, meta):
        if meta is not None:
            params['meta'] = ','.join(meta)
        response = self.request('PATCH', '/api/v1/albums/album_publish', params)
        return from_json(response.content)

    def album_unpublish(self, params, meta):
        if meta is not None:
            params['meta'] = ','.join(meta)
        response = self.request('PATCH', '/api/v1/albums/album_unpublish', params)
        return from_json(response.content)

    def reindex_faces(self, params):
        response = self.request('POST', '/api/v1/albums/faces/reindex', params)
        return from_json(response.content)

    def album_snapshot(self, params):
        response = self.request('GET', '/api/v1/albums/snapshot', params)
        return from_json(response.content)


djfs_albums_legacy = DjfsAlbumsLegacyService()
djfs_albums = DjfsAlbumsService()
