from mpfs.common.errors import DjfsApiProxyError, DjfsError, DjfsApiProxyNotImplementedError
from mpfs.common.util import from_json
from mpfs.common.util.urls import urlencode
from mpfs.core.services.common_service import RequestsPoweredServiceBase, Service

MPFS_ERROR_CODE_HEADER_NAME = 'X-MPFS-Legacy-Error-Code'
DJFS_NOT_IMPLEMENTED_HTTP_CODE = 422  # special not implemented code


class DjfsApiService(RequestsPoweredServiceBase):
    def __init__(self):
        super(DjfsApiService, self).__init__()
        self.send_cloud_request_id = True

    def _exception_from_response(self, response):
        if MPFS_ERROR_CODE_HEADER_NAME in response.headers:
            error_code = int(response.headers[MPFS_ERROR_CODE_HEADER_NAME])
            json = from_json(response.content)

            data = None
            if isinstance(json, dict):
                data = json.get('data')
            if data is None:
                data = {'code': response.status_code}

            return DjfsApiProxyError(error_code, data)
        if response.status_code == DJFS_NOT_IMPLEMENTED_HTTP_CODE:
            return DjfsApiProxyNotImplementedError()
        return super(DjfsApiService, self)._exception_from_response(response)

    def mkdir(self, uid, path, connection_id=None):
        params = {'uid': uid, 'path': path}
        if connection_id is not None:
            params['connection_id'] = connection_id
        response = self.request('GET', '/json/mkdir', params)
        return response.content

    def move(self, uid, src, dst, force, connection_id, callback, get_lenta_block_id, return_status,
             check_hids_blockings, force_djfs_albums_callback=False):
        params = {
            'uid': uid,
            'src': src,
            'dst': dst,
            'force': force,
            'callback': callback,
            'get_lenta_block_id': get_lenta_block_id,
            'return_status': return_status,
            'check_hids_blockings': check_hids_blockings,
        }
        if connection_id is not None:
            params['connection_id'] = connection_id
        response = self.request('GET', '/json/move', params)
        return from_json(response.content)

    def async_move(self, uid, src, dst, force, connection_id, callback, force_djfs_albums_callback=False):
        params = {
            'uid': uid,
            'src': src,
            'dst': dst,
            'force': force,
            'callback': callback,
        }
        if connection_id is not None:
            params['connection_id'] = connection_id
        response = self.request('GET', '/json/async_move', params)
        return from_json(response.content)

    def trash_append(self, uid, path, md5, version, connection_id):
        params = {
            'uid': uid,
            'path': path,
            'md5': md5,
            'version': version,
        }
        if connection_id is not None:
            params['connection_id'] = connection_id
        response = self.request('GET', '/json/trash_append', params)
        return from_json(response.content)

    def async_trash_append(self, uid, path, md5, version, callback, connection_id):
        params = {
            'uid': uid,
            'path': path,
            'md5': md5,
            'version': version,
            'callback': callback,
        }
        if connection_id is not None:
            params['connection_id'] = connection_id
        response = self.request('GET', '/json/async_trash_append', params)
        return from_json(response.content)

    def bulk_info(self, params, meta, body):
        if meta is not None:
            params['meta'] = ','.join(meta)
        response = self.request('POST', '/json/bulk_info', params, data=body,
                                headers={'Content-Type': 'application/json'})
        return from_json(response.content)

    def bulk_info_by_resource_ids(self, params, meta, body):
        if meta is not None:
            params['meta'] = ','.join(meta)
        response = self.request('POST', '/json/bulk_info_by_resource_ids', params, data=body,
                                headers={'Content-Type': 'application/json'})
        return from_json(response.content)

    def public_list(self, params, meta):
        if meta is not None:
            params['meta'] = ','.join(meta)
        response = self.request('GET', '/json/public_list', params)
        return from_json(response.content)

    def public_info(self, params, meta):
        if meta is not None:
            params['meta'] = ','.join(meta)
        response = self.request('GET', '/json/public_info', params)
        return from_json(response.content)

    def info(self, params, meta):
        if meta is not None:
            params['meta'] = ','.join(meta)
        response = self.request('GET', '/json/info', params)
        return from_json(response.content)

    def default_folders(self, params):
        response = self.request('GET', '/json/default_folders', params)
        return from_json(response.content)

    def album_append_items(self, params, meta, body):
        if meta is not None:
            params['meta'] = ','.join(meta)
        response = self.request('POST', '/json/album_append_items', params, data=body,
                                headers={'Content-Type': 'application/json'})
        return from_json(response.content)

    def album_item_remove(self, params):
        response = self.request('GET', '/json/album_item_remove', params)
        return from_json(response.content)


class DjfsApiLegacyService(Service):
    name = 'DjfsApiLegacyService'
    api_error = DjfsError

    def build_cv_callback(self, uid, resource_id):
        return self.base_url + '/api/v1/indexer/set_aesthetics?uid=%(uid)s&resource_id=%(resource_id)s' % {
            'uid': uid,
            'resource_id': resource_id
        }

    def build_dimensions_callback(self, uid, resource_id):
        return self.base_url + '/api/v1/indexer/set_dimensions?uid=%(uid)s&resource_id=%(resource_id)s' % {
            'uid': uid,
            'resource_id': resource_id
        }

    def reset_overdraft(self, uid):
        self.open_url(
            self.base_url + '/api/reset_overdraft_date?' + urlencode({'uid': uid}),
            post={},
            method='POST',
            headers={'Content-Type': 'application/json'}
        )


djfs_api_legacy = DjfsApiLegacyService()
djfs_api = DjfsApiService()
