# -*- coding: utf-8 -*-

"""

MPFS
CORE

Сервис общения с фотками

"""
import logging
import re
import os
import cjson
import sys
import traceback

from lxml import etree
from time import mktime
from datetime import datetime

import mpfs.engine.process

from mpfs.common import errors
from mpfs.core.services.common_service import StorageService, BaseFilter


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log("fotki")


class PrivilegeFilter(BaseFilter):
    req_field = 'privilege'

    def get(self):
        return {self.req_field : self.val[0]}


class Fotki(StorageService):

    name = 'fotki'
    api_error = errors.FotkiNoResponse
    log = service_log


    def __init__(self, *args, **kwargs):
        StorageService.__init__(self, *args, **kwargs)
        self.list_params = {
            'sort'        : 'order',
            'order'       : 'asc',
            'privilege'   : 'privilege',
            'offset'      : 'offset',
            'amt'         : 'amount'
        }

        self.available_bounds = {
            'order'  : 'asc',
            'amount' : 'amount',
            'offset' : 'offset',
            'sort'   : 'order',
        }

        self.available_filters = {
            'privilege' : PrivilegeFilter,
        }

        self.available_sort_fields = {
#            'name'  : 'name',
            'ctime' : 'created'
        }

        self.command_map = {
            'load_file'              : 'getImage',
            'count_files'            : 'countImages',
            'folder_content'         : 'listAlbum',
            'edit_folder'            : 'editAlbum',
            'edit_file'              : 'editImage',
            'remove_folder'          : 'deleteAlbum',
            'create_folder'          : 'addAlbum',
            'make_folder'            : 'addAlbum',
            'get_original_photo_url' : 'getEditedOriginal',
        }

        self.errors_map = {
            'image not found'         : errors.FotkiNotFound,
            'user not found'          : errors.FotkiUserNotFound,
            'album not found'         : errors.FotkiNotFound,
            'parent album not found'  : errors.FotkiNotFound,
        }

    def process_response(self, data):
        result = None
        try:
            action = self.command_map[sys._getframe(1).f_code.co_name]
            response = self.open_url(self.base_url + action, data)
            result = etree.fromstring(response)
        except self.api_error, ae:
            return self._parse_error(ae.data['text'])
        except etree.XMLSyntaxError:
            error_log.error("Bad response from Fotki: %s",traceback.format_exc())
            raise errors.MailBadResponse()

        error = result.tag == 'error'
        if error is True:
            self._parse_error(result)

        return result


    def _parse_error(self, err_element):
        exception_class = None
        try:
            error_map = dict(err_element.items())
            exception_message = error_map.get('name')
            exception_class = self.errors_map[exception_message]
        except KeyError:
            error_log.error("Unknown error code from Fotki: %s", exception_message)
            exception_class = errors.FotkiBadResponse
        except Exception:
            exception_class = self.api_error

        raise exception_class()


    def count_files(self, resource):
        if resource.address.root:
            album_id = '0'
        else:
            album_id = resource.address.name
        request_data = {'authorId' : resource.uid, 'albumId' : album_id}
        try:
            resp = self.process_response(request_data)
        except errors.FotkiUserNotFound:
            return '0'
        if resp.tag == 'images':
            return resp.find('count').text
        else:
            raise self.api_error()


    def load_file(self, resource):
        image_id = resource.address.name
        request_data = {'authorId' : resource.uid, 'imageId' : image_id}
        resp = self.process_response(request_data)
        image_data = resp.find('image')
        resource.update(self._image(image_data))
        resource.id = resource.address.path


    def folder_content(self, resource):
        super(Fotki, self).folder_content(resource)

        result = []
        request_data = {}

        for k, v in self.request_processing_fields['filters'].iteritems():
            request_data.update(self.available_filters[k](v).get())

        for k, v in self.request_processing_fields['bounds'].iteritems():
            request_data[k] = v
        if not 'amount' in request_data:
            request_data['amount'] = 1000000

        if resource.address.is_storage:
            album_id = '0'
        else:
            album_id = resource.address.name

        request_data['albumId'] = album_id
        request_data['authorId'] = resource.uid

        resp = self.process_response(request_data)

        parsed = self.parse_response(resp)
        resource.update(parsed['this'])
        resource.id = resource.address.path
        for child in parsed['list']:
            child = child.get('this', child)
            child['id'] = resource.address.path + os.path.sep + child['id']
            child['visible'] = 1
            if child['type'] == 'dir':
                resource.sorted_folders.append(child['id'])
                resource.child_folders[child['id']] = child
                resource.construct_child_folder(child['id'], child)
            else:
                resource.sorted_files.append(child['id'])
                resource.child_files[child['id']] = child
                resource.construct_child_file(child['id'], child)
            result.append(child)
        return result


    def make_folder(self, address, parent, data={}):
        if parent.name == 'fotki':
            parent.name = 0
            
        request_data = {
            'authorId' : address.uid,
            'title'    : address.name,
            'parentId' : parent.name,
        }
        resp = self.process_response(request_data)
        album_data = resp.find('album')
        resp = {'meta' : {}}
        resp['id'] = parent.address.get_child_folder(album_data.get('album-id')).path
        resp['meta']['url'] = album_data.find('url').text
        resp['name'] = album_data.find('title').text
        return resp


    def remove_folder(self, address):
        request_data = {
            'authorId' : address.uid,
            'albumId' : address.name,
        }
        resp = self.process_response(request_data)
        return resp


    def edit_folder(self, address, **params):
        '''
            Chenges folder/album fields
        '''
        editable_fields = ('title', 'description', 'password')
        data = {
            'authorId' : address.uid,
            'albumId'  : address.name,
        }
        data.update(
            dict(
                filter(
                    lambda (k, v): k in editable_fields and v,
                    params.iteritems()
                )
            )
        )
        resp = self.process_response(data)
        return resp


    def edit_file(self, address, **params):
        editable_fields = (
            'ownerId',
            'authId',
            'newAlbumId',
            'title',
            'description',
            'access',
            'tags',
            'hideOriginal',
            'disableComments',
            'porno',
            'latitude',
            'longitude',
            'mapType',
            'zoom'
        )
        bool_fields = (
            'porno',
            'hideOriginal',
            'disableComments'
        )
        data = {
            'ownerId' : address.uid,
            'authId'  : address.uid,
            'imageId' : address.name
        }
        for k, v in params.iteritems():
            if k in bool_fields:
                params[k] = cjson.encode(bool(int(v)))
        data.update(
            dict(
                filter(
                    lambda (k, v): k in editable_fields,
                    params.iteritems()
                )
            )
        )
        resp = self.process_response(data)
        return resp


    def parse_response(self, resp):
        if resp.tag == 'view_album':
            album = resp.find('album')
            if album.get('num_images') and int(album.get('num_images')):
                return self._photos_in_album(album)
            else:
                return self._albums_in_album(album)
        else:
            error_log.error("Bad response from fotki: %s", etree.tostring(resp))
            raise errors.FotkiBadResponse()


    def _albums_in_album(self, raw):
        this = {}
        this['id'] = raw.get('id')
        this['type'] = 'dir'
        this['meta'] = {}

        title = raw.find('title')
        if title is not None:
            this['name'] = title.text

        subalbums = []

        raw_subalbums = raw.find('albums')
        if raw_subalbums is not None:
            for album in raw_subalbums.findall('album'):
                subalbums.append(self._albums_in_album(album))

        for fkey, mkey in (('created', 'ctime'), ('updated', 'mtime')):
            this[mkey] = self.time_parse(raw.get(fkey))

        for attr in ('url', 'description'):
            tag_data = raw.find(attr)
            if tag_data is not None:
                this['meta'][attr] = tag_data.text

        preview = raw.find('preview')
        can_preview = preview.get('can_view')
        if not can_preview or can_preview != 'false':
            preview_data = {}

            preview_data['flags'] = {}
            for flag in preview.find('flags').getchildren():
                preview_data['flags'][flag.tag] = flag.text

            url = preview.find('url')
            if url is not None:
                preview_data['url'] = url.text

            raw_flags_all = preview.find('flags-all')
            if raw_flags_all is not None:
                preview_data['flags-all'] = {}
                for flag in raw_flags_all.getchildren():
                    preview_data['flags-all'][flag.tag] = flag.text

            preview_data['sizes'] = []
            for preview_size in preview.find('sizes').findall('size'):
                preview_data['sizes'].append(dict(preview_size.items()))
                if preview_size.get('name') == 'XXXS':
                    preview_data['thumbnail'] = preview_size.get('url')
                if preview_size.get('name') == 'S':
                    preview_data['preview'] = preview_size.get('url')

            this['meta'].update(preview_data)

        return {'this' : this, 'list' : subalbums}


    def _image(self, raw):
        this = {}
        this['meta'] = {}
        this['id'] = raw.get('id')
        file_name = raw.find('file_name')
        if file_name is not None:
            this['name'] = file_name.text
        if raw.find('title') is not None:
            this['meta']['file_name'] = this.get('name', '')
            this['name'] = raw.find('title').text
        this['type'] = 'file'
        this['flags'] = {}
        this['size'] = 0
        this['source'] = self.name
        this['visible'] = 1

        size_bytes = raw.find('size-bytes')
        if size_bytes is not None:
            this['size'] = size_bytes.get('ORIG')

        for attr in ('title', 'description'):
            tag_data = raw.find(attr)
            if tag_data is not None:
                this['meta'][attr] = tag_data.text

        coord = raw.find('coord')
        if coord is not None:
            for k, v in coord.items():
                this['meta'][k] = v
        tags_item = raw.find('tags')
        if tags_item is not None:
            this['meta']['tags'] = []
            for tag_item in tags_item.getchildren():
                this['meta']['tags'].append(tag_item.text)

        for fkey, mkey in (('added', 'ctime'), ('updated', 'mtime')):
            this[mkey] = self.time_parse(raw.get(fkey))

        this['meta']['exif_date'] = self.time_parse(raw.get('exif_date'))

        flags = raw.find('flags')
        if flags is not None:
            for flag in flags.getchildren():
                this['meta'][flag.tag] = 1

        all_flags = raw.find('flags-all')
        if all_flags is None:
            for flag in all_flags.getchildren():
                this['meta'][flag.tag] = 1

        this['meta']['sizes'] = []
        image_sizes = raw.find('sizes').findall('size')

        for image_size in image_sizes:
            this['meta']['sizes'].append(dict(image_size.items()))
            if image_size.get('name') == 'XXXS':
                this['meta']['thumbnail'] = image_size.get('url')
            if image_size.get('name') == 'S':
                this['meta']['preview'] = image_size.get('url')

        page_url = raw.find('page-url')
        if page_url is not None:
            this['meta']['url'] = page_url.text

        this['meta']['drweb'] = 1

        return this


    def _photos_in_album(self, raw):
        this = {}
        this['id'] = raw.get('id')
        this['type'] = 'dir'
        this['meta'] = {}


        title = raw.find('title')
        if title is not None:
            this['name'] = title.text

        for fkey, mkey in (('created', 'ctime'), ('updated', 'mtime')):
            this[mkey] = self.time_parse(raw.get(fkey))

        for attr in ('url', 'description'):
            tag_data = raw.find(attr)
            if tag_data is not None:
                this['meta'][attr] = tag_data.text

        preview = raw.find('preview')

        this['meta']['flags'] = {}
        if preview.find('flags') is not None:
            for flag in preview.find('flags').getchildren():
                this['meta']['flags'][flag.tag] = flag.text

        this['meta']['flags-all'] = {}
        if preview.find('flags-all') is not None:
            for flag in preview.find('flags-all').getchildren():
                this['meta']['flags-all'][flag.tag] = flag.text

        this['meta']['sizes'] = []
        if preview.find('sizes') is not None:
            for preview_size in preview.find('sizes').findall('size'):
                this['meta']['sizes'].append(dict(preview_size.items()))
                if preview_size.get('name') == 'XXXS':
                    this['meta']['thumbnail'] = preview_size.get('url')
                if preview_size.get('name') == 'S':
                    this['meta']['preview'] = preview_size.get('url')

        images = []
        for raw_image in raw.findall('image'):
            images.append(self._image(raw_image))

        this['meta']['chunk_numfiles'] = len(images)

        return {'this' : this, 'list' : images}


    def time_parse(self, val):
        if not val or val == '0000-00-00 00:00:00':
            val = '0'
        elif not str(val).isdigit():
            RE_TIME = '(\d+)-(\d+)-(\d+).(\d+):(\d+):(\d+)(.*)'
            RE_DIFF = '(.)(\d+):(\d+)'
            try:
                _y, _mnth, _d, _h, _min, _s, difference = \
                    re.search(RE_TIME, val).groups()
                val = int(
                    mktime(
                        datetime(
                            int(_y),
                            int(_mnth),
                            int(_d),
                            int(_h),
                            int(_min),
                            int(_s)
                        ).timetuple()
                    )
                )
            except Exception, e:
                log.warn(e)
        return int(val)


    def get_original_photo_url(self, resource):
        image_id = resource.path.name
        request_data = {'authorId' : resource.uid, 'imageId' : image_id}
        resp = self.process_response(request_data)
        if resp.tag != 'edited-orig':
            raise self.api_error()
        cookie = resp.get('FSessionId') or resp.get('fSessionId')
        return resp.get('url'), cookie
