# -*- coding: utf-8 -*-

from mpfs.config import settings
from mpfs.common.util import from_json
from mpfs.common.util.urls import quote, update_qs_params
from mpfs.core.services.common_service import Service
from mpfs.core.user.constants import SUPPORTED_LOCALES, DEFAULT_LOCALE
from mpfs.engine.process import get_global_tld

OFFICE_HANCOM_DISCOVERY = settings.office['hancom']['open_document_button']['discovery']


class HancomService(Service):
    name = 'hancom'
    _discovery = None

    _app_aliases = {
        'WRITE_EDITOR': 'weboffice',
        'CALC_EDITOR': 'webCalc',
        'SHOW_EDITOR': 'webShow',
    }

    def get_edit_url(self, uid, resource_id, resource_name, extension, locale, editor_name='editor'):
        extension = extension.lower()
        locale = locale if locale in SUPPORTED_LOCALES else DEFAULT_LOCALE
        qs_params = {
            'app': self._get_discovery()[extension],
            'lang': '%s_%s' % (locale, SUPPORTED_LOCALES[locale]),
            'user_id': editor_name,
            'uid': uid,
            'owner_uid': resource_id.uid,
            'fid': resource_id.file_id,
        }
        host = self.external_host % {'tld': get_global_tld()}
        url = host + '/cloud-office/api/yadisk2/%s/open' % quote(resource_name)
        return update_qs_params(url, qs_params)

    def get_document_status(self, extension, file_id):
        path = '/%s/api/v1/getDocumentStatus.do' % self.get_app_alias(extension)
        url = update_qs_params(self.internal_host + path, {'docId': file_id})
        response = from_json(self.open_url(url))
        last_modified = int(response['lastModified'])
        continue_polling = bool(response['pollingEnabled'])
        return last_modified, continue_polling

    def get_download_url(self, extension, file_id):
        path = '/%s/api/v1/exportDocument.do' % self.get_app_alias(extension)
        return update_qs_params(self.internal_host + path, {'docId': file_id})

    @classmethod
    def is_extension_supported(cls, extension):
        if not extension:
            return False
        extension = extension.lower()
        return extension in cls._get_discovery()

    @classmethod
    def get_app_alias(cls, extension):
        app_name = cls._get_discovery()[extension]
        return cls._app_aliases[app_name]

    @classmethod
    def _get_discovery(cls):
        if cls._discovery is None:
            cls._discovery = {}
            for app_name, ext_list in OFFICE_HANCOM_DISCOVERY.iteritems():
                cls._discovery.update({e: app_name for e in ext_list})
        return cls._discovery
