# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис для получения списка сетей по макросу из hbf
https://wiki.yandex-team.ru/NOC/newnetwork/hbf/server/#zaprossoderzhimogomakrosa

Пример запроса для получения списка хостов по сетевому макросу:
    curl -s "http://hbf.yandex.net/macros/_DYNAMICACCESSNETS_?format=json"


"""
import time
import os

import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.util.persistent_dict import PersistentDict
from mpfs.core.services.common_service import RequestsPoweredServiceBase
from mpfs.common.util import from_json, Singleton
from mpfs.common.util.iptools import IPRangeList
from mpfs.config import settings
from mpfs.core.signals import register_signal, add_timer


service_log = mpfs.engine.process.get_service_log('hbf')


class HbfService(RequestsPoweredServiceBase, Singleton):
    name = 'hbf'
    log = service_log
    api_error = errors.HbfError
    cache_file_format = 'json'
    service_check_interval = None
    cache_update_period = None
    cache_file_path = None
    macroses = None
    _cache_data = {
        'networks': {},
        'update_time': 0,
    }

    def _check_cache_actual(self):
        now = self._get_timestamp()
        if now > self._cache_data['update_time'] + self.cache_update_period:
            self._reload_cache_from_file()
            self._cache_data['update_time'] = now  # обновляем в любом случае

    @classmethod
    def is_cache_file_exists(cls):
        return os.path.exists(cls.cache_file_path)

    def _reload_cache_from_file(self):
        if not os.path.exists(self.cache_file_path):
            self.update_cache()

        cache_dict = PersistentDict(self.cache_file_path, flag='r', mode=None, format=self.cache_file_format)
        if not cache_dict:
            self.log.info('Reload cache from file failed')
            # продолжаем пользоваться старым кешом - новый (на диске) побитый, ждем следующего обновления
        else:
            self.networks = {macros: IPRangeList(*networks) for macros, networks in cache_dict.items()}

    @property
    def networks(self):
        self._check_cache_actual()
        return self._cache_data['networks']

    @networks.setter
    def networks(self, value):
        self._cache_data['networks'] = value
        self._cache_data['update_time'] = self._get_timestamp()

    def fetch_networks_by_macros(self, macros):
        """
        Получаем список хостов по имени макроса

        :param macros: имя сетевого макроса
        :type macros: str
        :return: список сетей
        """
        result = self.request('GET', 'macros/%s' % macros, params={'format': 'json', 'trypo_format': 'none'})
        service_result = from_json(result.content)
        return service_result

    @staticmethod
    def _get_timestamp():
        return int(time.time() * 1000)

    def _update_cache_for_item(self, macros, persistent_dict):
        try:
            networks = self.fetch_networks_by_macros(macros)
        except Exception:
            self.log.exception('Fetch networks from HBF server failed')
            return
        persistent_dict[macros] = networks

    def update_cache(self):
        cache_dir = os.path.dirname(self.cache_file_path)
        if not os.path.exists(cache_dir):
            os.makedirs(cache_dir)

        with PersistentDict(self.cache_file_path, flag='c', mode=None, format=self.cache_file_format) as persistent_dict:
            for network_macros in self.macroses:
                self._update_cache_for_item(network_macros, persistent_dict)
            for macros in persistent_dict.keys():
                if macros not in self.macroses:
                    del persistent_dict[macros]


def setup_hbf_cache_update():
    """Запустить периодическое обновление кеша.
    """
    hbf = HbfService()
    register_signal(
        mpfs.engine.process.Signal.HBF_CACHE,
        lambda signum: hbf.update_cache(),
    )
    add_timer(mpfs.engine.process.Signal.HBF_CACHE, hbf.service_check_interval)
    hbf.update_cache()
