# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис индексации всего

http://wiki.yandex-team.ru/pochta/ya.disk/MPFS/search

"""
import time
from itertools import imap

import mpfs.engine.process
import mpfs.common.errors as errors
from mpfs.core.services.common_service import Service
from mpfs.core.services.djfs_api_service import djfs_api_legacy
from mpfs.core.services.tikaite_service import Tikaite
from mpfs.common.util import get_file_extension, to_json
from mpfs.common.util.filetypes import getGroupNumber
from mpfs.config import settings
from mpfs.core.metastorage.control import user_index
from mpfs.core.services.smartcache_service import smartcache
from mpfs.common.util.urls import update_qs_params, quote


default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
search_log = mpfs.engine.process.get_service_log('indexer')

tikaite = Tikaite()


class Indexer(Service):
    log = search_log


class SearchIndexer(Indexer):
    name = 'search_indexer'
    log = search_log

    def get_file_body(self, file_data):
        extension = get_file_extension(file_data['id'])
        mimetype = file_data['mimetype']
        result = {}
        if (extension in settings.tikaite['excluded_extensions'] or
                mimetype in settings.tikaite['excluded_mimetypes']):
            return result
        elif (settings.indexer['limit_index_body'] and
              file_data['size'] >= settings.tikaite['file_size_limit']):
            return result
        else:
            result = tikaite.get_body(file_data['stid'], mimetype)
            return result

    def push_change(self, data, wait_search_response=False, service='disk_queue', append_smartcache_callback=False):
        if isinstance(data, dict):
            data = [data]

        changes = {
            'delete': [],
            'modify': [],
            'update': [],
            'reindex': [],
        }

        while len(data):
            each = data.pop()
            action = each.pop('action')
            operation = each.pop('operation', 'other')
            if action != 'delete':
                each['key'] = each.pop('id')
            each['id'] = each.pop('file_id')
            result = self.convert(each, action=action, append_smartcache_callback=append_smartcache_callback,
                                  operation=operation)
            result['operation'] = operation
            changes[action].append(result)

        for action_data in changes.itervalues():
            for each in action_data:
                if not each:
                    continue
                operation = each.pop('operation')
                each['docs'] = [x for x in each['docs'] if x['id']['value'] is not None]
                if not each['docs']:
                    default_log.info('Nothing to push, skipping... data=%s' % each)
                    continue
                id_ = ','.join(imap(lambda x: x['id']['value'], each['docs']))
                url = self.base_url % {
                    'id': id_,
                    'action': operation,
                    'prefix': each.get('prefix', ''),
                    'service': service,
                }
                # проставляем timestamp непосредственно перед отправкой
                url += '&timestamp=%s' % int(time.time())
                if wait_search_response:
                    url += '&wait=true'
                if each['docs'] and each['docs'][0].get('version'):
                    url += '&version=%s' % each['docs'][0]['version']['value']
                if each['docs'] and each['docs'][0].get('resource_id'):
                    url += '&resource_id=%s' % quote(each['docs'][0]['resource_id']['value'])
                if each['docs'] and each['docs'][0].get('metric'):
                    url += '&metric=%s' % quote(each['docs'][0]['metric']['value'])
                if each['docs'] and each['docs'][0].get('shared_folder_version'):
                    url += '&shared_folder_version=%s' % each['docs'][0]['shared_folder_version']['value']
                if each['docs'] and each['docs'][0].get('is_reindexed_for_quick_move'):
                    if each['docs'][0]['is_reindexed_for_quick_move']['value']:
                        url += '&fast-moved'
                if 'callbacks' in each:
                    ts = int(time.time())
                    each['callbacks'] = [update_qs_params(c, {'request_ts': [ts]}) for c in each['callbacks']]
                    for x in each['callbacks']:
                        url += '&callback=' + quote(x)
                if 'cv_callbacks' in each:
                    for c in each['cv_callbacks']:
                        url += '&cv_callback=' + quote(c)
                if each['docs'] and each['docs'][0].get('clusterize_face_enabled'):
                    url += '&clusterize_face=true'

                try:
                    self.open_url(url, log_data=False)
                except errors.MPFSError, e:
                    if e.data.get('code') != 400:
                        raise

    def convert(self, data, action=None, append_smartcache_callback=False, operation=None):
        uid = int(data.pop('uid'))

        if 'mediatype' in data:
            data['mediatype'] = getGroupNumber(data['mediatype'])

        docs = {}
        for k, v in data.iteritems():
            docs[k] = {'value': v}

        result = {
            'action': action,
            'prefix': uid,
            'docs': [docs],
        }
        result.update(
            self._format_callbacks(uid, data, operation, action, append_smartcache_callback)
        )

        return result

    def _format_callbacks(self, uid, data, operation, action, append_smartcache_callback):
        append_djfs_callbacks = data.get('append_djfs_callbacks')
        resource_id = data.get('resource_id')

        callbacks = []
        cv_callbacks = []

        if append_smartcache_callback:
            timestamp_data = self._get_timestamps_data_for_smarctache_worker(data)
            smartcache_callback = smartcache.build_worker_url(uid, timestamp_data, operation, action)
            callbacks.append(smartcache_callback)
            if append_djfs_callbacks:
                cv_callbacks.append(smartcache_callback)
        if append_djfs_callbacks and resource_id:
            callbacks.append(djfs_api_legacy.build_dimensions_callback(uid, resource_id))
            cv_callbacks.append(djfs_api_legacy.build_cv_callback(uid, resource_id))

        result = {}

        if callbacks:
            result['callbacks'] = callbacks
        if cv_callbacks:
            result['cv_callbacks'] = cv_callbacks

        return result

    @staticmethod
    def _get_timestamps_data_for_smarctache_worker(file_data):
        timestamps = {}
        for i in ('etime', 'mtime', 'ctime', 'utime'):
            timestamp_val = file_data.get(i)
            if timestamp_val is None:
                continue
            timestamps[i] = timestamp_val
        return timestamps

    def start_reindex_for_quick_move(self, uid):
        callback_host = self.reindex_callback_host
        if self.reindex_callback_host == 'localhost':
            callback_host = 'http://' + mpfs.engine.process.hostname()

        callback = callback_host + self.reindex_callback_path % {'uid': uid}
        url = self.reindex_url % {'uid': uid, 'callback': quote(callback)}

        self.open_url(url)
