# -*- coding: utf-8 -*-
"""
Интерфейс к Juggler-у

Инофрмацию можно почерпнуть с этой страницы:
https://wiki.yandex-team.ru/sm/juggler/batching/#schema
"""
from enum import Enum

from mpfs.common.util import from_json, to_json
from mpfs.core.services.common_service import Service


class JStatus(Enum):
    OK = 'OK'
    CRIT = 'CRIT'
    WARN = 'WARN'


class JEvent(object):
    """Базовый класс евента в juggler

    Копипаста отсюда: https://wiki.yandex-team.ru/sm/juggler/introduction/
    * Хост (host) - уникальное имя объекта мониторинга.
    * Сервис (service) - имя метрики объекта мониторинга. У одного хоста может быть много метрик.
    * Статус (status) - качественное именование состояния метрики объекта мониторинга.
        В каждый момент времени у метрики объекта мониторинга может быть какой-то один статус, Juggler'е их три: OK , WARNING и CRITICAL .
    * Дескрипшн (description) или Описание - дополнительная информация к статусу, его текстовое описание.
    """
    host = None
    service = None

    def __init__(self, status, description=None):
        if self.host is None:
            raise ValueError("Juggler `host` not specified")
        if self.service is None:
            raise ValueError("Juggler `service` not specified")
        self.status = status
        self.description = description

    def send(self):
        """Отправить событие (себя) в Juggler"""
        Juggler().send_event(self)

    def as_dict(self):
        return {
            "status": self.status.value,
            "description": self.description if self.description else self.status.value,
            "host": self.host,
            "service": self.service
        }

    def __repr__(self):
        return '%s(host=%r, service=%r, status=%r, description=%r)' % (
            self.__class__.__name__,
            self.host,
            self.service,
            self.status,
            self.description
        )


class FindDeletedStidJEvent(JEvent):
    host = 'disk_pworker_devops'
    service = 'find_deleted_stid_in_db'


class FrozenOperationsCommonJEvent(JEvent):
    host = 'disk_mworker_devops'
    service = 'frozen_operations_common'


class FrozenOperationsImportantJEvent(JEvent):
    host = 'disk_mworker_devops'
    service = 'frozen_operations_important'


class Juggler(Service):
    name = 'juggler'
    source = 'mpfs'

    def __init__(self, *args, **kwargs):
        super(Juggler, self).__init__(*args, **kwargs)

    def send_event(self, event):
        """Отправить событие в juggler

        :param JEvent event: отправляемое событие
        """
        body = {
            'source': self.source,
            'events': [event.as_dict()],
        }
        body = to_json(body)
        url = "%s%s" % (self.base_url, '/events')
        reply = self.open_url(url, pure_data=body, method='POST', headers={"Content-type": "application/json"})
        event_status = from_json(reply)['events'][0]
        if event_status['code'] != 200:
            raise self.api_error(event_status['error'])
