# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис Кладун

"""
import sys

from lxml import etree
from urllib import quote
from urlparse import urljoin

import mpfs.engine.process

from mpfs.common import errors
from mpfs.engine.http import client
from mpfs.engine.queue2.utils import get_callback_host
from mpfs.core.services.common_service import Service
from mpfs.common.util import from_json
from mpfs.common.util.urls import urlencode
from mpfs.core.filesystem.hardlinks.common import FileChecksums


service_log = mpfs.engine.process.get_service_log('kladun')


class Kladun(Service):
    name = 'kladun'
    action = ''
    api_error = errors.KladunNoResponse
    log = service_log
    # оверайдится из конфигов
    recalc_digests_timeout = None

    def __init__(self, *args, **kwargs):
        Service.__init__(self, *args, **kwargs)
        if not self.callback_url:
            self.callback_url = 'http://' + mpfs.engine.process.hostname()
        if self.action:
            self.base_url = self.base_url + self.action + '/'

    def _get_callback_url(self, uid, oid):
        callback_url = 'http://' + get_callback_host() + '/service/kladun_callback'
        return callback_url + '?uid={}&oid={}'.format(uid, oid)

    def open_url(self, url, data=None, json=False, timeout=None, retry=True):
        post = self._preprocess_body_arguments(data)
        try:
            response = super(Kladun, self).open_url(url, post=post, timeout=timeout, retry=retry)
        except errors.MPFSError, e:
            if e.data.get('code') == 410:
                raise errors.OperationNotFoundError(data=e.data)
            raise e

        try:
            if json:
                return from_json(response)
            return etree.fromstring(response)
        except Exception as e:
            raise errors.KladunBadResponse, e, sys.exc_info()[2]

    def check_service_is_alive(self, url):
        try:
            super(Kladun, self).open_url(url, method='HEAD')
        except errors.KladunNoResponse, e:
            if e.data.get('code') == 404:
                # Если кладун дал внятный ответ, то считаем, что он жив.
                return True
            return False
        return True

    def status(self, url, response=''):
        if not response:
            element = self.open_url(url)
        else:
            service_log.debug(response)
            element = etree.fromstring(response)
        return element

    def _preprocess_body_arguments(self, body_args):
        if not body_args:
            return {}
        body_args.update({
            'api': self.api_ver,
            'callback': self._get_callback_url(body_args['uid'], body_args['oid'])
        })
        str_uid = body_args.get('uid', None)
        if str_uid:
            try:
                int(str_uid)
            except ValueError:
                del body_args['uid']
                body_args['special-uid'] = str_uid
        free_space = body_args.get('free_space', None)
        if free_space:
            del body_args['free_space']
            body_args['max-file-size'] = free_space
        return body_args

    def extract_exif(self, file_mid):
        """
            Получить EXIF изображения.
        """
        url = self.base_url + 'regenerate-exif?mulca-id=%s' % file_mid
        try:
            response = client.open_url(url, service_log, timeout=self.timeout,
                                       api_err=self.api_error)
        except Exception:
            return None, {}
        else:
            if response:
                result = from_json(response)
                return result['mid'], result['exif']
            else:
                return None, {}

    def get_file_checksums(self, file_mid):
        """Получить хеши и размер файла

        :param str file_mid: stid файла
        :rtype: FileChecksums
        """
        resp = self._recalc_digests(file_mid, generate_digest=False)
        return FileChecksums(resp['md5'], resp['sha256'], int(resp['content-length']))

    def regenerate_digest(self, file_mid):
        """Сделать запрос на генерацию нового дайджеста

        :param str file_mid: stid файла
        :return: digest_stid
        """
        return self._recalc_digests(file_mid, generate_digest=True)['webdav-digest-mulca-id']

    @staticmethod
    def get_image_info_from_response(dom_element):
        image_info = {'original_width': None, 'original_height': None, 'rotate_angle': None}
        try:
            original_width = int(dom_element.get('original-width', ''))
            original_height = int(dom_element.get('original-height', ''))
        except ValueError:
            pass
        else:
            if original_width > 0 and original_height > 0:
                image_info['original_width'] = original_width
                image_info['original_height'] = original_height

        try:
            rotate_angle = int(dom_element.get('rotate-angle', ''))
        except ValueError:
            pass
        else:
            if rotate_angle in (0, 90, 180, 270):
                image_info['rotate_angle'] = rotate_angle
        return image_info

    def _recalc_digests(self, file_mid, generate_digest=False):
        params = {
            'mulca-id': file_mid,
            'with-webdav': 'false',
        }
        if generate_digest:
            params['with-webdav'] = 'true'

        url = urljoin(self.base_url, 'recalc-digests')
        url += '?%s' % urlencode(params)
        return self.open_url(url, timeout=self.recalc_digests_timeout, json=True)


class Upload(Kladun):
    def post_request(self, post_data):
        response = self.open_url(self.base_url + post_data['service'],
                                 post_data)
        return response.get('post-target'), response.get('poll-result')


class OfficeConvertAndUpload(Kladun):
    def post_request(self, post_data):
        response = self.open_url(self.base_url, post_data)
        return response.get('poll-result')


class ExtractFileFromArchive(Kladun):
    """
    Распаковать файл из архива

    https://wiki.yandex-team.ru/Pochta/chemodan/uploader/api#raspakovatfajjlizarxivainternalmpfsuploader
    """
    action = 'extract-file-from-archive'

    def post_request(self, post_data):
        required_keys = {'uid', 'oid', 'file-id', 'path', 'source-service',
                         'service-file-id', 'file-to-extract', 'max-file-size'}
        # должны быть обязательные параметры
        missed_required_keys = required_keys - post_data.viewkeys()
        if missed_required_keys:
            raise KeyError("Missed required params: %s" % list(missed_required_keys))
        response = self.open_url(self.base_url, post_data)
        return response.get('post-target'), response.get('poll-result')


class UploadToService(Upload):
    '''
        Загрузить файл в сервис
    '''

    action = 'upload-to-service-url'


class UploadToDisk(Upload):
    '''
        Загрузить файл в диск
    '''

    action = 'upload-url'


class OfficeConvertAndUploadToDisk(OfficeConvertAndUpload):
    """
        Сконвертировать старый офисный файл и положить в диск
    """

    action = 'convert-to-ms-ooxml-format'


class Patch(Kladun):
    '''
        Наложить дельта обновление на файл
    '''

    action = 'patch-url'

    def post_request(self, post_data):
        response = self.open_url(self.base_url + post_data.pop('service'),
                                 post_data)
        return response.get('delta-target'), response.get('poll-result')


class Capture(Kladun):
    '''
        Скопировать файл с сервиса в диск
    '''

    action = 'upload-from-service'

    def post_request(self, post_data):
        response = self.open_url(self.base_url, post_data)
        return response.get('poll-result')


class Publish(Kladun):
    '''
        Скопировать файл из диска в сервис
    '''

    action = 'publish'

    def post_request(self, post_data):
        response = self.open_url(self.base_url + post_data.pop('service'),
                                 post_data)
        return response.get('poll-result')


class ExtractRotation(Kladun):
    '''
        Извлечь поворот изображения.
    '''

    action = 'extract-rotation'

    def post_request(self, post_data):
        response = self.open_url(self.base_url, post_data)
        return {
            'rotation': int(response.find('rotation').text)
        }


class ExportPhotos(Kladun):
    '''
        Экспортировать список фото в социальную сеть
    '''

    action = 'export-photos'

    def post_request(self, post_data):
        response = self.open_url(self.base_url, post_data)
        return response.get('poll-result')
