# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис Почта

"""
import os
import operator
import traceback
import urllib2
import itertools
from lxml import etree
from time import time as now

import mpfs.engine.process
import mpfs.common.errors as errors

from mpfs.core.services.common_service import StorageService, BaseFilter
from mpfs.engine.http.client import open_url
from mpfs.core.services.common_service import Service
from mpfs.core.services import mulca_service
from mpfs.common.util import filetypes, from_json
from mpfs.common.util.urls import update_qs_params

error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log("mail")

mulca = mulca_service.Mulca()


class SimpleFilter(BaseFilter):

    req_field = ''
    def get(self):
        if not isinstance(self.val, (list, tuple)):
            self.val = [str(self.val), ]
        else:
            self.val = map(str, self.val)
        return self.req_field + ':' + '%s' % (' OR %s' % self.req_field + ':').join(self.val)


class GapFilter(BaseFilter):

    def get(self):
        def group(all):
            '''
                type(all) = list
            '''
            result = ''
            if all and not len(all) % 2:
                result = '%s:[%s TO %s]' % (self.req_field, all.pop(0), all.pop(0))
                if all:
                    result = '(' + result + ' OR ' + group(all) + ')'
            return result

        return group(self.val)


class FromFilter(SimpleFilter):

    req_field = 'hdr_from'

    def get(self):
        if not isinstance(self.val, (list, tuple)):
            self.val = [str(self.val), ]
        else:
            self.val = map(str, self.val)
        query = ' OR '.join(map(lambda x: '*%s*' % x, filter(lambda x: '@' not in urllib2.unquote(x), self.val))
            + filter(lambda x: '@' in urllib2.unquote(x), self.val))
        return '%s:(%s)' % (self.req_field, query)


class ToFilter(SimpleFilter):

    req_field = 'hdr_to'


class SubjectFilter(SimpleFilter):

    req_field = 'hdr_subject'

class FileNameFilter(SimpleFilter):

    req_field = 'attachname'

class FolderNameFilter(SimpleFilter):

    req_field = 'foldername'

class TypeFilter(SimpleFilter):

    req_field = 'attachname'

    def __init__(self, val):
        if isinstance(val, (str, unicode)):
            val = val.split(',')
            res = {}
            res['exts']=[]
            res['mimes']=[]
            for v in val:
                res['mimes'].extend(filetypes.getMimeTypesFromGroup(v))
                exts = filetypes.getExtensionsFromGroup(v)
                res['exts'].extend(map(lambda x: '*.'+x, exts))
            self.val = res

    def get(self):
        query_exts = ' OR '.join(self.val.get('exts'))
        query_mimes = ' OR '.join(self.val.get('mimes'))
        return '%s:(%s) OR mimetype:(%s)' % (self.req_field, query_exts, query_mimes)


class TimeFilter(GapFilter):

    req_field = 'received_date'


class VisibleFilter(SimpleFilter):

    req_field = 'visible'

    def get(self):
        return ''


class MailStidService(Service):
    name = 'mail_stid'
    api_error = errors.MailNoResponse
    log = service_log

    def get_mail_stid(self, uid, mail_id):
        """Возвращает stid письма (сторадж айди, идентификатор бинарника).

        :param uid: идентификатор пользователя
        :param mail_id: идентификатор письма
        :return: stid, если найдено, в противном случае None
        """
        url = self.filter_search_url % {'uid': uid, 'mid': mail_id}
        response = self.open_url(url)
        response = from_json(response)

        envelopes = response['envelopes']
        if not envelopes:
            return None

        return envelopes[0].get('stid')

    def get_service_file_id(self, uid, mail_mid, mail_hid):
        """Возвращает service-file-id пригодный для использования в Кладуне
        по переданным `uid`, `mail_mid` и `mail_hid`.

        :param uid: индентификатор пользователя
        :param mail_mid: идентификатор письма (внешний, передают почтовики)
        :param mail_hid: часть письма
        :raises: :class:`~mpfs.common.errors.ResourceNotFound`
        :return: service-file-id
        :rtype: str
        """
        return '%s:%s/%s' % (uid, mail_mid, mail_hid)


class Mail(StorageService):
    name = 'mail'
    api_error = errors.MailNoResponse
    log = service_log

    def __init__(self, *args, **kwargs):
        StorageService.__init__(self, *args, **kwargs)
        self.available_filters = {
            'ctime'   : TimeFilter,
            'from'    : FromFilter,
            'to'      : ToFilter,
            'subject' : SubjectFilter,
            'name'    : FileNameFilter,
            'visible' : VisibleFilter,
            'media_type' : TypeFilter,
        }

        self.empty_filters = ('visible', )

        self.available_bounds = {
            'amount' : 'amt',
            'sort'   : 'sort',
            'offset' : 'offset',
            'order'  : 'order',
        }
        self.available_sort_fields = {
            'name'  : 'name',
            'ctime' : 'ctime',
            'mtime' : 'ctime',
        }

    def process_response(self, url):
        result = None
        try:
            response = self.open_url(url, cookie={self.cookie_name: self.cookie_data})
            result = etree.fromstring(response)
        except etree.XMLSyntaxError:
            error_log.error("Bad response from Mail: %s", traceback.format_exc())
            raise errors.MailBadResponse()
        return result

    def timeline(self, resource, *args, **kwargs):
        '''
            newstyle: allows "tree" option
            tree: 1 - first level content, 0 - content from all levels
        '''
        final_result = []
        parsed = self.folder_content(resource, timeline=True)
        if parsed.tag == 'resources':
            if int(parsed.get('count')):
                cur_elem = parsed

                def _parse_element(element):
                    element_dict = self.xml2dict(element)
                    element_dict['visible'] = 1
                    element_dict['name_tree'] = (['mail', ] + filter(None, element_dict['meta']['fullname'].split('/')))[::-1]
                    element_dict['path_named'] = '/mail' + element_dict['meta']['fullname']
                    if element.tag == 'folder':
                        element_dict['id'] = str(resource.uid) + ':/mail/' + element_dict['meta']['fullpath'] + '/'
                        resource.sorted_folders.append(element_dict['id'])
                        resource.child_folders[element_dict['id']] = element_dict
                        resource.construct_child_folder(element_dict['id'], element_dict)
                    else:
                        full_id_split = filter(None, element_dict['meta']['fullpath'].split('/'))
                        element_dict['id'] = str(resource.uid) + ':/mail/' + '/'.join(full_id_split[:-1]) + ':' + full_id_split[-1]
                        element_dict['meta']['drweb'] = 1
                        if element_dict['meta'].get('preview', False):
                            safe_preview_url = self._give_safe_url(element_dict, thumb=True)
                            element_dict['meta']['thumbnail'] = safe_preview_url
                            element_dict['meta']['preview']   = safe_preview_url
                            element_dict['meta']['sizes'] = [{'url': safe_preview_url, 'name': 'S'}]
                        element_dict['meta']['url'] = self._give_safe_url(element_dict)
                        resource.sorted_files.append(element_dict['id'])
                        resource.child_files[element_dict['id']] = element_dict
                        resource.construct_child_file(element_dict['id'], element_dict)
                    return element_dict

                final_result = map(_parse_element, filter(operator.methodcaller('get', 'id'), cur_elem.iterchildren()))
                resource.num_res_filtered = parsed.get('count')
        else:
            error_log.error("Bad response from Mail: %s", etree.tostring(parsed))
            raise errors.MailBadResponse()
        return final_result

    def tree(self, resource):

        def _parse_element(main_element, res=resource):
            main_result = []
            for element in main_element.iterchildren():
                meta_data = {}
                metainfo = element.findall('metainfo')
                if metainfo:
                    for meta_element in metainfo[0].findall('meta'):
                        meta_data[meta_element.get('name')] = meta_element.get('value')
                    fullpath = meta_data.pop('fullpath')
                    result = {
                              'id' : '/mail' + fullpath + '/',
                              'name' : element.get('name'),
                              'visible' : 1,
                              'fullname' : '/mail' + meta_data.get('fullname', ''),
                              'path_named': '/mail' + meta_data.get('fullname', ''),
                              'name_tree' : (['mail', ] + filter(None, fullpath.split('/')))[::-1],
                              'hasfolders' : meta_data.pop('has_folders'),
                              'meta' : meta_data,
                              }
                    if element.get('id') == res.address.name:
                        res.update(result)
                    pId = result['meta'].get('parentId')
                    if pId:
                        result['meta']['parentId'] = 'folder:' + pId
                    else:
                        result['meta']['parentId'] = 'mail'

                    main_result.append(result)

            groupped = dict((k, list(v)) for k,v in itertools.groupby(sorted(main_result, key=lambda x: x['meta']['parentId']), lambda x: x['meta']['parentId']))

            def make_tree(root):
                subfolder_result = []
                for each in groupped.get(filter(None, root.id.split('/'))[-1], ()):
                    each['id'] = root.uid + ':' + each['id']
                    root.child_folders[result['id']] = each
                    subfolder_result.append(make_tree(root.construct_child_folder(each['id'], each)))
                return {'this': root.dict(), 'list': subfolder_result}

            resssult = make_tree(res)
            return resssult

        final_result = {}
        parsed = self.folder_content(resource, tree=True)
        if parsed.tag == 'resources':
            if int(parsed.get('count')):

                final_result = _parse_element(parsed, resource)

                resource.num_res_filtered = parsed.get('count')
            else:
                final_result = {'this' : resource, 'list' : []}
            return final_result
        else:
            error_log.error("Bad response from Mail: %s", etree.tostring(parsed))
            raise errors.MailBadResponse()

    def listing(self, resource):
        final_result = []
        parsed = self.folder_content(resource)
        if parsed.tag == 'resources':
            if resource.address.is_storage:
                cur_elem = parsed
            else:
                cur_elem = parsed.find('folder')
                if cur_elem is None or not len(cur_elem):
                    self.load_folder(resource)
                    cur_elem = parsed
                else:
                    self._parse_folder_tag(resource, parsed)
            if int(parsed.get('count')):
                def _parse_element(element):
                    element_dict = self.xml2dict(element)
                    element_dict['visible'] = 1
                    element_dict['id'] = resource.address.get_clean_id() + os.path.sep + element.get('id')
                    element_dict['path_named'] = '/mail' + element_dict['meta']['fullname']
                    if element.tag == 'folder':
                        element_dict['id'] += os.path.sep
                        element_dict['hasfolders'] = element_dict['meta']['has_folders']
                        resource.sorted_folders.append(element_dict['id'])
                        resource.child_folders[element_dict['id']] = element_dict
                        resource.construct_child_folder(element_dict['id'], element_dict)
                    else:
                        id_split = element_dict['id'].split('/')
                        element_dict['id'] = '/'.join(id_split[:-1]) + ':' + id_split[-1]
                        element_dict['meta']['drweb'] = 1
                        if element_dict['meta'].get('preview', False):
                            safe_preview_url = self._give_safe_url(element_dict, thumb=True)
                            element_dict['meta']['thumbnail'] = safe_preview_url
                            element_dict['meta']['preview']   = safe_preview_url
                            element_dict['meta']['sizes'] = [{'url': safe_preview_url, 'name': 'S'}]

                        element_dict['meta']['url'] = self._give_safe_url(element_dict)
                        resource.sorted_files.append(element_dict['id'])
                        resource.child_files[element_dict['id']] = element_dict
                        resource.construct_child_file(element_dict['id'], element_dict)
                    return element_dict

                final_result = map(_parse_element, filter(operator.methodcaller('get', 'id'), cur_elem.iterchildren()))
                resource.num_res_filtered = parsed.get('count')
        else:
            error_log.error("Bad response from Mail: %s", etree.tostring(parsed))
            raise errors.MailBadResponse()


        return final_result

    def folder_content(self, resource, tree=False, timeline=False):
        '''
        Частичная загрузка файлов и подпапок по параметрам
        '''
        super(Mail, self).folder_content(resource)

        _url = self.base_url % (resource.uid)

        fid = filter(None, resource.address.path.split(os.path.sep))[-1]

        if not resource.address.is_storage:
            _url += '&path=%s' % fid

        for k, v in self.request_processing_fields['bounds'].iteritems():
            if k == 'order':
                v = ('desc', 'asc')[v]
            _url += '&%s=%s' % (k, v)

        if tree:
            self.available_filters['name'] = FolderNameFilter
        else:
            self.available_filters['name'] = FileNameFilter

        _url += '&newstyle=1'

        for i in self.empty_filters:
            self.request_processing_fields['filters'].pop(i, '')

        if timeline or tree or self.request_processing_fields['filters']:
            _url += '&tree=0'
            name_field = self.request_processing_fields['filters'].get('name')
            if tree:
                name_field = '*%s*' % name_field if name_field else '*'
            else:
                name_field = '*%s*' % name_field if name_field else None
            if name_field:
                self.request_processing_fields['filters']['name'] = name_field
        else:
            _url += '&tree=1'

        filter_values = filter(None,
                               map(lambda (k,v): self.available_filters[k](v).get(),
                                   self.request_processing_fields['filters'].iteritems()
                            ))


        if filter_values:
            _url += ('&request=%s' % ' AND '.join(filter_values))

        return self.process_response(_url)

    def load_folder(self, resource):
        _url = self.info_url % resource.uid + '&id=%s' % resource.address.name
        try:
            _src = self.process_response(_url)
        except Exception:
            raise self.api_error()
        self._parse_folder_tag(resource, _src)

    def _parse_folder_tag(self, resource, _src):
        try:
            folder_tag = _src.find('folder')
            resource.name = folder_tag.get('name')
            resource.ctime = folder_tag.get('ctime')
            meta_info = folder_tag.find('metainfo')
            if meta_info is not None:
                for _m in meta_info.findall('meta'):
                    if _m is not None:
                        resource.meta[_m.get('name')] = _m.get('value')
        except Exception:
            raise self.api_error()

    def load_file(self, resource):
        if '/' in resource.address.name:
            _name = resource.address.name
        else:
            name_split = resource.address.name.split(':')
            _name = ':'.join(name_split[:-1]) + '/' + name_split[-1]
        _url = self.info_url % resource.address.uid + '&id=%s' % _name
        try:
            _src = self.process_response(_url)
            file_tag = _src.find('file')
            resource.name = file_tag.get('name')
            resource.size = file_tag.get('size')
            resource.source = file_tag.get('source')
            resource.mimetype = file_tag.get('type')
            meta_data = file_tag.find('metainfo')
            if meta_data is not None:
                for _m in meta_data.findall('meta'):
                    if _m.get('name') in ('to', 'from', 'cc', 'bcc'):
                        resource.meta[_m.get('name')] = self.parse_addr(_m.get('value'))
                    else:
                        resource.meta[_m.get('name')] = _m.get('value')
            resource.meta['drweb'] = 1
            resource.ctime = resource.meta.pop('date', '')
            resource.mtime = resource.meta.pop('received_date', '')
            if resource.meta.get('preview', False):
                resource.meta['thumbnail'] = resource.meta['preview']
                resource.meta['sizes'] = [{'url': resource.meta['preview'], 'name': 'S'}]
            resource.meta['url'] = self._give_safe_url({
                'name': resource.name,
                'meta': {'part': resource.meta['part'], 'mid': resource.meta['mid']}
            })
        except AttributeError:
            error_log.error(traceback.format_exc())
            raise errors.ResourceNotFound
        except Exception:
            error_log.error(traceback.format_exc())
            raise self.api_error()

    def webattach_url(self, resource):
        if '/' in resource.address.name:
            _name = resource.address.name
        else:
            name_split = resource.address.name.split(':')
            _name = ':'.join(name_split[:-1]) + '/' + name_split[-1]
        _url = self.info_url % resource.address.uid + '&id=%s' % _name
        try:
            _src = self.process_response(_url)
            meta_tag = _src.find('file').find('metainfo')
            meta = {}
            for el in meta_tag:
                meta[el.get('name')] = el.get('value')
            return meta['url']
        except Exception, e:
            error_log.error(e)
            error_log.error(traceback.format_exc())
            raise self.api_error()

    def direct_url(self, stid, part):
        mid = '%s/%s' % (stid, part)
        return mulca.get_local_url(mid)

    def _give_safe_url(self, element, thumb=False):
        url = self.safe_web_url % (
            urllib2.quote(element['name'].encode('utf-8')),
            urllib2.quote(element['meta']['part']),
            urllib2.quote(element['meta']['mid']),
            urllib2.quote(element['name'].encode('utf-8'))
        )
        if thumb:
            url += '&thumb=y&no_disposition=y'
        return url


class MailObject(object):

    def __init__(self):
        self.meta = None
        self.name = None


class MailTVM(StorageService):
    name = 'mail_tvm'
    api_error = errors.MailNoResponse
    log = service_log

    def mimes(self, uid, mids=tuple()):
        """Возвращает метаинформацию о частях письма."""
        assert mids
        url = update_qs_params(self.base_url + '/mimes', {
            'uid': uid,
            'mid': mids,
            'with_mulca': 'yes',
            'with_inline': 'yes'
        })
        response = self.open_url(url)
        response = from_json(response)

        result = {}
        for mid in response['mimes']:
            root_data = response['mimes'][mid]['root']
            other_data = response['mimes'][mid]['other']

            if root_data:
                for _, mime_part_data in root_data['mimeParts'].items():
                    hid = mime_part_data['hid']
                    result.setdefault(mid, {})[hid] = mime_part_data

            if other_data:
                for stid, stid_data in other_data.items():
                    for _, hid_data in stid_data.items():
                        mime_part_data = hid_data['mimePart']
                        result.setdefault(mid, {})[mime_part_data['hid']] = hid_data['mimePart']

        return result

    def get_mime_part_name(self, uid, mid, hid):
        """Получить имя MIME-парта.

        В случае если парт не найден или не найдено письмо, то будет возвращен None.
        """
        result = self.mimes(uid=uid, mids=(mid,))
        if mid not in result:
            return

        if hid not in result[mid]:
            return

        return result[mid][hid]['name']
