# -*- coding: utf-8 -*-
import datetime
import time

import mpfs.common.errors.billing as errors
import mpfs.engine.process
from mpfs.common.static.tags.billing import *
from mpfs.common.util import from_json, to_json
from mpfs.core.services.common_service import RequestsPoweredServiceBase

default_log = mpfs.engine.process.get_default_log()


def mediabilling_datetime_to_ts(value):
    tz_shift = 3 * 60 * 60
    return time.mktime(datetime.datetime.strptime(value, "%Y-%m-%dT%H:%M:%SZ").timetuple()) + tz_shift


class MediabillingStatOrder(object):
    def __init__(self, number, price, currency):
        self.price = int(float(price))
        self.currency = currency
        self.number = number
        self.auto = True


class MediabillingOrderInfo(dict):
    """
    Вспомогательный класс-обертка над ответом биллинга на запрос check_order
    """
    def __init__(self, data):
        super(MediabillingOrderInfo, self).__init__(data)
        self._btime = None

    @property
    def has_btime(self):
        return bool(self._btime)

    @property
    def has_lbtime(self):
        return 'lastPaymentTime' in self

    @property
    def btime(self):
        return self._btime

    @btime.setter
    def btime(self, value):
        """
        сеттер
        """
        self._btime = mediabilling_datetime_to_ts(value)

    @property
    def lbtime(self):
        if self.has_lbtime:
            return time.mktime(datetime.datetime.strptime(self['lastPaymentTime'][:19], "%Y-%m-%dT%H:%M:%S").timetuple())

    @property
    def billing_is_active(self):
        return self['status'] == OK
        # TODO: медиабиллинг не умеет в заказе отдавать статус услуги. Тут всегда ОК когда услуга была оплаченв


class MediabillingPaymentsService(RequestsPoweredServiceBase):
    """
    Сервис для общения с биллингом медиасервисов.

    https://wiki.yandex-team.ru/mb/music-billing-ruchkas/
    """

    valid_statuses = [200, 201, 451]

    def __init__(self):
        self.enabled_for_all = None
        self.enabled_for_uids = []
        super(MediabillingPaymentsService, self).__init__()

    def is_enabled(self, uid=None):
        if self.enabled_for_all:
            return False
        return uid in self.enabled_for_uids

    def _build_headers(self, headers=None, uid=None, region=None, user_ip=None):
        headers = headers or {}
        if uid is not None:
            headers['X-Uid'] = uid
        if region is not None:
            headers['X-Region-Id'] = region
        if user_ip is not None:
            headers['X-User-Ip'] = user_ip
        headers.update({
            'Content-Type': 'application/json',
        })
        return headers

    def request(self, method, relative_url, params=None, data=None, headers=None, cookies=None, timeout=None,
                send_tvm_ticket=None, check_status=True, *args, **kwargs):
        body = None
        if data is not None:
            body = to_json(data, true_utf8=False)

        response = super(MediabillingPaymentsService, self).request(
            method, relative_url, params=params, data=body, headers=headers, cookies=cookies, timeout=timeout,
            send_tvm_ticket=send_tvm_ticket, *args, **kwargs)

        status = response.status_code
        parsed_response = from_json(response.content)
        if check_status:
            if not status or status not in self.valid_statuses:  # А надо ли valid_statuses
                raise errors.MediaBillingError(parsed_response)
        return parsed_response

    def order_info(self, uid, order_id):
        """
        Получить информацию о заказе.

        https://wiki.yandex-team.ru/mb/music-billing-ruchkas/order-info/

        # {u'created': u'2019-09-11T13:32:30+03:00',
        #   u'currency': u'RUB',
        #   u'debitAmount': 750,
        #   u'lastPaymentAmount': 750.0,
        #   u'lastPaymentCurrency': u'RUB',
        #   u'lastPaymentTime': u'2019-09-11T13:36:57+03:00',
        #   u'orderId': 5035907,
        #   u'paidAmount': 750.0,
        #   u'paidDays': 0,
        #   u'paidQuantity': 1.0,
        #   u'paymentMethod': u'card-x48a5c229a07ac5d67dba6a82',
        #   u'paymentMethodType': u'card',
        #   u'rrn': u'489012',
        #   u'status': u'ok',
        #   u'trialPayment': False,
        #   u'type': u'native-auto-subscription',
        #   u'userAccount': u'510000****3459'}
        """
        relative_url = '/internal-api/account/billing/order-info'
        params = {
            '__uid': uid,
            'orderId': order_id,
        }
        headers = self._build_headers(uid=uid)

        response = self.request('GET', relative_url, params=params, headers=headers)
        return response['result']

    def native_products(self, ip_address, uid=None, language=None, campaign=None, target='disk'):
        """
        Получить список нативных продуктов, доступных для оформления.

        https://wiki.yandex-team.ru/mb/music-billing-ruchkas/native-products/
        """
        relative_url = '/internal-api/account/native-products'
        params = {
            'target': target,
            'ipAddress': ip_address,
        }
        if uid:
            params['__uid'] = uid
        if language:
            params['language'] = language
        if campaign:
            params['campaign'] = campaign
        headers = self._build_headers(uid=uid, user_ip=ip_address)

        response = self.request('GET', relative_url, params=params, headers=headers)
        if 'result' in response and 'nativeProducts' in response['result']:
            result = response['result']['nativeProducts']
        else:
            result = []
        return result

    def submit_native_order(self, uid, ip_address, product_id, return_path='', template_tag='desktop/form'):
        """
        Покупка нативного продукта.

        https://wiki.yandex-team.ru/mb/music-billing-ruchkas/submit-native-order/
        """

        relative_url = '/internal-api/account/submit-native-order'
        params = {
            '__uid': uid,
            'ipAddress': ip_address,
            'productId': product_id,
            'source': 'disk',
            'returnPath': return_path,
            'templateTag': template_tag,
        }
        headers = self._build_headers(uid=uid, user_ip=ip_address)
        response = self.request('POST', relative_url, params=params, headers=headers)

        if response['result']['status'] not in (SUCCESS, ALREADY_PURCHASED):
            raise errors.MediaBillingStatusError

        if response['result']['status'] != ALREADY_PURCHASED:
            order_id = response['result']['orderId']
            payment_id = response['result']['trustPaymentId']

            url = None
            params = {}
            form = response['result'].get('form')
            if form:
                params = {}
                for field in form['fields']:
                    params[field['name']] = field['value']
                if 'purchase_token' not in params:
                    raise AttributeError
                url = '%s?purchase_token=%s' % (form['target'], params['purchase_token'])
            return {'url': url, 'params': params, PAYMENT_ID: payment_id, 'order_id': order_id}
        return {}

    def stop_subscription(self, uid, product_id):
        """
        Остановить продление подписки.

        https://wiki.yandex-team.ru/mb/music-billing-ruchkas/stop-subscription/
        """

        relative_url = '/internal-api/account/stop-subscription'

        params = {
            '__uid': uid,
            'productId': product_id,
        }
        headers = self._build_headers(uid=uid)
        response = self.request('POST', relative_url, params=params, headers=headers)
        return response.get('result')

    def intervals(self, uid):
        """
        Проверка колбеков

        https://wiki.yandex-team.ru/mb/intervals/
        """

        relative_url = '/internal-api/billing/intervals'

        params = {
            '__uid': uid,
        }
        headers = self._build_headers(uid=uid)
        result = self.request('GET', relative_url, params=params, headers=headers)
        return result

    def get_last_active_interval_for_pids(self, uid, pids):
        response = self.intervals(uid)
        data = None
        start = None
        now = time.time()
        for interval in response['result']['intervals']:
            pid = interval['orderInfo']['productInfo']['id']

            if interval['orderInfo'].get('stopped') or pid not in pids:
                continue

            if 'end' in interval and interval['end']:
                expires_ts = int(mediabilling_datetime_to_ts(interval['end']))
                if expires_ts <= now:
                    continue

            if not start or interval['start'] > start:
                data = {UID: uid,
                        NUMBER: str(interval['orderInfo']['id']),
                        PRICE: str(interval['orderInfo']['productInfo']['price']['amount']),
                        CURRENCY: str(interval['orderInfo']['productInfo']['price']['currency']),
                        PID: pid,
                        STATUS: SUCCESS,
                        STATUS_CODE: '',
                        EXPIRES: interval['end']}
                start = interval['start']

        return data


mediabilling_payment_service = MediabillingPaymentsService()
