# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис MPFS (сам себя)

http://wiki.yandex-team.ru/disk/mpfs/api/json

"""
from collections import defaultdict
from types import NoneType

import mpfs.engine.process
import mpfs.common.errors.platform as errors

from mpfs.common.util import from_json, to_json
from mpfs.core.services.common_service import Service
from mpfs.core.services.mulca_service import Mulca
from mpfs.common.util import iso8601, urls

service_log = mpfs.engine.process.get_service_log('mpfsproxy')

_DEFAULT_BULK_INFO_META = ['size', 'mimetype', 'md5', 'custom_preview', 'public_hash', 'short_url', 'original_id',
                           'media_type', 'append_time', 'custom_properties', 'resource_id', 'group']


class MpfsProxy(Service):

    name = 'mpfsproxy'
    api_error = errors.MpfsProxyBadResponse
    log = service_log
    base_url = None  # should be set in config
    timeout = None  # should be set in config

    errors_map = {
        404: errors.MpfsProxyNotFound,
        409: errors.MpfsProxyConflict,
        497: errors.MpfsProxyNoFreeSpace,
    }

    def __init__(self, *args, **kwargs):
        super(MpfsProxy, self).__init__(*args, **kwargs)
        self._cache = defaultdict()

    def clear_cache(self):
        self._cache = defaultdict()

    def process(self, command_base, params, **kwargs):
        url_base = self.base_url + '/json' + command_base
    
        try:
            response = self.open_url(url_base, params, retry=False, **kwargs)
            result = from_json(response)
        except errors.MpfsProxyBadResponse, e:
            try:
                code = int(e.data.get('code'))
                message = from_json(e.data.get('text')).get('title') or ''
                err_class = self.errors_map.get(code, errors.MpfsProxyBadResponse)
            except Exception:
                message = ''
                err_class = errors.MpfsProxyBadResponse            
            raise err_class(extra_msg=message)
            
        return result

    def bulk_info(self, uid, paths, meta=None, preview_size=None, preview_crop=None, preview_quality=None,
                  preview_allow_big_size=None):
        return self.bulk_info_base('bulk_info', uid, to_json(paths), meta, preview_size, preview_crop, preview_quality,
                                   preview_allow_big_size)

    def bulk_info_by_resource_ids(self, uid, resource_ids, meta=None, preview_size=None, preview_crop=None,
                                  preview_quality=None, preview_allow_big_size=None):
        return self.bulk_info_base('bulk_info_by_resource_ids', uid, to_json(resource_ids), meta,
                                   preview_size, preview_crop, preview_quality, preview_allow_big_size)

    def bulk_info_base(self, action, uid, data, meta, preview_size, preview_crop, preview_quality,
                       preview_allow_big_size):
        if meta is None:
            meta = _DEFAULT_BULK_INFO_META

        command_base = '/%s?uid=%s' % (action, uid)
        if meta:
            command_base += '&meta=%s' % ','.join(meta)
        if preview_size:
            command_base += '&preview_size=%s' % preview_size
        if preview_crop:
            command_base += '&preview_crop=1'
        if preview_quality is not None:
            command_base += '&preview_quality=%s' % preview_quality
        if preview_allow_big_size:
            command_base += '&preview_allow_big_size=1'
        return self.process(command_base, {}, method='POST', headers={'Content-Type': 'application/json'},
                            pure_data=data)

    def content(self, uid, resource, options):
        def format(item):
            resource = {
                'path': item['id'].replace('/disk/', '/', 1),
                'type': item['type'],
                'name': item['name'],
            }
            
            for inner, outer in ({'ctime':'created', 'mtime':'modified'}).iteritems():
                resource[outer] = iso8601(item[inner])
            
            if item['type'] == 'dir':
                if item['name'] == 'disk':
                    resource['name'] = '/'
            elif item['type'] == 'file':
                meta = item['meta']
                resource['size'] = meta['size']
                resource['mime-type'] = meta['mimetype']
                resource['md5'] = meta['md5']
                resource['content_location'] = meta['file_url']
                
                if 'sizes' in meta:
                    resource['preview'] = meta['sizes'][0]['url']
                    
            return resource
        
        params = {
            'meta': 'sizes,mimetype,size,md5,file_url',
        }
        params.update(options)
        
        command_base = '/list?uid=%s&path=%s' % (uid, resource)

        raw_result = self.process(command_base, params)
        if isinstance(raw_result, list):
            # Return requested dir and directory children
            dir = format(raw_result[0])
            children = [format(item) for item in raw_result[1:]]
            return (dir, children)
        else:
            # Return requested file and None as children
            file = format(raw_result)
            return (file, None)

    def download_url(self, uid, resource):
        command_base = '/info?uid=%s&path=%s' % (uid, resource)
        result = self.process(command_base, {'meta': 'file_url'})
        return result['meta']['file_url']

    def direct_url(self, uid, resource):
        command_base = '/info?uid=%s&path=%s' % (uid, resource)
        result = self.process(command_base, {'meta': 'file_mid'})
        mulca_id = result['meta']['file_mid']
        return Mulca().get_file_url(mulca_id)

    def store(self, uid, resource):
        command_base = '/store?uid=%s&path=%s' % (uid, resource)
        return self.process(command_base, {})

    def default_folders(self, uid):
        url = '/json/default_folders?uid=%s' % uid
        ret = self._cache.get('default_folders', {}).get(uid, None)
        if not ret:
            ret = self.open_url('%s%s' % (self.base_url, url), retry=False)
            if not isinstance(ret, (dict, list, NoneType)):
                ret = from_json(ret)
            if 'default_folders' not in self._cache:
                self._cache['default_folders'] = {}
            self._cache['default_folders'][uid] = ret
        return ret

    def get_album(self, uid, album_id, amount=None):
        command_base = '/album_get'
        params = {'uid': uid, 'album_id': album_id}
        if amount is not None:
            params['amount'] = amount
        url = urls.update_qs_params(command_base, params)
        try:
            return self.process(url, {})
        except errors.MpfsProxyNotFound:
            return None


mpfsproxy = MpfsProxy()
