# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис Мулька

"""
import re
import sys
import time

import mpfs.engine.process
import mpfs.common.errors as errors

from mpfs.engine.http import client as http_client
from mpfs.core.services.common_service import Service
from mpfs.common.util.urls import update_qs_params

log = mpfs.engine.process.get_default_log()
service_log = mpfs.engine.process.get_service_log('mulca')


class StidNotFound(errors.MulcaNoResponse):
    pass


class Mulca(Service):

    name = 'mulca'
    api_error = errors.MulcaNoResponse
    log = service_log
    retries_on_head_req = 2
    avatars_stid_pattern = re.compile(r'^ava:disk:(?P<group_id>[^:]+):(?P<image_name>[^:]+)$')

    command_map = {
        'remove': 'del',
        'check': 'get',
    }

    def build_url(self, action, mid, params=''):
        if mid.startswith('ava:'):
            return self._build_url_for_avatars(action, mid)
        return self.base_url % (action, mid, params)

    def _action(self, mid, params=''):
        command_name = sys._getframe(1).f_code.co_name
        req_url = self.build_url(self.command_map[command_name], mid, params=params)

        return self.open_url(req_url)

    def remove(self, mid):
        """
        Удаление файла из хранилища

        Возвращает HTTP code ответа мульки
        """
        url = self.build_url('del', mid)
        url = update_qs_params(url, {'service': 'disk_clean',
                                     'ns': 'disk'})

        log.debug('removing mid %s' % mid)
        http_code, _, _ = self.open_url(url, return_status=True, api_err=None)
        return int(http_code)

    def get_file_size(self, stid):
        """
        Получить размер файла от стораджа
        """
        http_code, _, headers = self._make_head_request(stid)
        if http_code == 404:
            raise StidNotFound(stid)
        if not 200 <= http_code < 300:
            raise self.api_error("Unexpected http status code: %s", http_code)

        try:
            return int(headers['content-length'])
        except Exception:
            raise self.api_error("Can't get `content-length`")

    def is_file_exist(self, stid):
        """
        Проверка наличия файла в хранилище

        Мулька возвращает 200 - файл есть - return True
        Мулька возвращает 4** - файла нет - return False
        Остальные статусы ретраются `retry_num` раз, а затем raise api_error
        """
        http_code, _, _ = self._make_head_request(stid)
        if http_code == 200:
            return True
        elif 400 <= http_code <= 499:
            return False
        raise self.api_error()

    def _make_head_request(self, stid):
        url = self.build_url('get', stid)
        url = update_qs_params(url, {'service': 'disk',
                                     'ns': 'disk'})

        for _ in xrange(self.retries_on_head_req):
            resp_tuple = self.open_url(url, method='HEAD', retry=False, return_status=True, api_err=None)
            if resp_tuple[0] < 500:
                return resp_tuple
        raise self.api_error()

    def get_local_url(self, mid):
        """
        Получить локальный редирект на файл
        """
        return self.local_url % mid

    def get_local_mulcagate_url(self, mid):
        """
        Получить ссылку на файл на локальный по отношению к кладуну мулькагейт
        """
        return self.local_mulcagate_url % ('get', mid, '')

    def get_file_url(self, mid):
        """
        Получить ссылку на файл
        """
        return self.base_url % ('get', mid, '')

    def get_signed_local_url(self, mid):
        """
        Получить подписанный локальный редирект на файл
        """
        timestamp = int(time.time())
        token = self.signed_token('md5', timestamp)
        return self.local_signed_url % (mid, timestamp, token)

    def _build_url_for_avatars(self, action, mid):
        match_result = self.avatars_stid_pattern.match(mid)
        if not match_result:
            raise errors.MalformedAvatarsStidError(mid)
        if action == 'get':
            url = self.avatars_get_url % match_result.groupdict()
        elif action == 'del':
            url = self.avatars_delete_url % match_result.groupdict()
        else:
            raise NotImplementedError('Unexpected action: %s' % action)
        return url
