# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис Народ

"""
import time
import traceback
import urllib

from lxml import etree
from os.path import sep

import mpfs.engine.process
import mpfs.common.errors as errors

from mpfs.core.metastorage.control import lnarod
from mpfs.core.services.common_service import Service, StorageService
from mpfs.core.services.disk_service import StorageSpaceUnlimited, Resources
from mpfs.common.util import time_parse

error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log("narod")


class Narod(Service):
    def lock_check(self, uid, path, version=None):
        pass


class LegacyNarod(StorageSpaceUnlimited):
    name = 'narod'
    control = lnarod

    def __init__(self, *args, **kwargs):
        super(LegacyNarod, self).__init__(*args, **kwargs)
        from mpfs.core.filesystem.resources.narod import LegacyNarodFile, LegacyNarodFolder
        self.resources = Resources(dir=LegacyNarodFolder, file=LegacyNarodFile)

    def free(self, uid):
        return 107374182400


class OriginalNarod(StorageService):

    name = 'narod'
    api_error = errors.NarodNoResponse
    log = service_log

    def process_response(self, url):
        result = None
        try:
            response = self.open_url(url)
            data = []
            for line in response.split("\n"):
                try:
                    r = etree.fromstring(line)
                except etree.XMLSyntaxError, e:
                    line= unicode(line, errors='ignore').encode('utf-8')
                data.append(line)
            
            result = etree.fromstring("\n".join(data))
        except etree.XMLSyntaxError:
            error_log.error("Bad response from Narod: %s",traceback.format_exc())
            raise errors.NarodBadResponse()
        return result

    def folder_content(self, resource):
        super(OriginalNarod, self).folder_content(resource)

        parsed = self.process_response(self.list_url % resource.uid) 
        preloaded_list = []
        
        if parsed.tag == 'files' and 'count' in parsed.attrib and int(parsed.attrib['count']):
            elements = parsed.findall('file')

            for element in elements:
                if not element.get('fid'):
                    continue
                element_dict = dict(element.items())
                for _d in ('edate', 'dldate'):
                    if _d in element_dict:
                        element_dict[_d] = self.time_parse(element_dict[_d])
                file_hash = element_dict.pop('hash')
                res = {
                    'type'    : 'file',
                    'name'    : element_dict.pop('filename'),
                    'ctime'   : self.time_parse(element_dict.pop('cdate', '')),
                    'id'      : resource.address.path + sep + file_hash,
                    'mimetype': element_dict.pop('filetype'),
                    'size'    : int(element_dict.pop('size')),
                    'meta'    : element_dict,
                    'source'  : self.name,
                    'visible' : 1,
                }
                res['meta']['url'] = self.pub_url % (file_hash, res['name'])
                preloaded_list.append(res)
                resource.sorted_files.append(res['id'])
                resource.child_files[res['id']] = res
                resource.construct_child_file(res['id'], res)
        return preloaded_list


    def load_file(self, resource):
        hash = filter(None, resource.address.path.split(sep)).pop()

        element = self.process_response(self.info_url % (hash, resource.uid))

        if element.tag == 'file' and 'success' in element.attrib and int(element.attrib['success']):
            element_dict = dict(element.items())
            for _d in ('edate', 'dldate'):
                if _d in element_dict:
                    element_dict[_d] = self.time_parse(element_dict[_d])
            res = {
                'type'    : 'file',
                'name'    : element_dict.pop('filename'),
                'ctime'   : self.time_parse(element_dict.pop('cdate')),
                'mimetype': element_dict.pop('filetype'),
                'size'    : element_dict.pop('size'),
                'meta'    : element_dict,
                'source'  : self.name,
                'visible' : 1,
            }
            for k, v in res.iteritems():
                setattr(resource, k, v)
            resource.id = resource.address.path


    def count_files(self, resource):
        result = 0
        
        response = self.open_url(self.count_url % resource.uid)
        element = etree.fromstring(response)

        if element.tag == 'files' and int(element.get('success', 0)):
            result = element.get('count')

        return result

    def _call_with_retry(self, methodname, *args):
        result = None
        for i in (1, 2, 3):
            error = False
            try:
                method = getattr(self, methodname)
                result = method(*args)
                error = False
            except Exception:
                error_log.error('failed to call %s %s, try %s' % (methodname, args, i))
                error_log.error(traceback.format_exc())
                error = True
            if not error:
                break
            time.sleep(1)
        if result is None:
            raise errors.NarodNoResponse()
        else:
            return result

    def set_user_moved(self, uid):
        response = self._call_with_retry('open_url', self.user_moved_url % uid)
        element = etree.fromstring(response)
        if element.tag == 'user' and int(element.get('success', 0)):
            return True
        
    def set_file_moved(self, resource, url):
        quoted_url = urllib.quote(url)
        response = self._call_with_retry('open_url', self.file_moved_url % (resource.name, resource.uid, quoted_url))
        element = etree.fromstring(response)
        if element.tag == 'file' and int(element.get('success', 0)):
            return True
        
    def list_moved_files(self, uid):
        parsed = self._call_with_retry('process_response', self.list_moved_files_url % uid)
        result = []
        
        if parsed.tag == 'files':
            elements = parsed.findall('file')
            for element in elements:
                element_dict = dict(element.items())
                result.append(element_dict)
                
        return result

    def time_parse(self, val):
        return time_parse(val)
