# -*- coding: utf-8 -*-

"""
Сервис Паспорт

http://http://doc.yandex-team.ru/Passport
https://doc.yandex-team.ru/blackbox/concepts/about.xml
https://wiki.yandex-team.ru/passport/dbmoving/
https://wiki.yandex-team.ru/passport/sids/

"""
import collections
import re
import sys

from lxml import etree
import mpfs.engine.process
from mpfs.common.util import decode_email
import mpfs.common.errors as errors

from mpfs.common.util import from_json, to_json
from mpfs.common.util import Cached
from mpfs.common.util.credential_sanitizer import CredentialSanitizer
from mpfs.config import settings
from mpfs.core.services.common_service import Service, RequestsPoweredServiceBase
from mpfs.core.services.tvm_service import TVMSignatureParametersBuilder

default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log('passport')

SERVICES_TVM_CONSUMER = settings.services['tvm']['consumer']
SERVICES_TVM_ENABLED = settings.services['tvm']['enabled']
SERVICES_TVM_2_0_ENABLED = settings.services['tvm_2_0']['enabled']

# https://doc.yandex-team.ru/Passport/AuthDevGuide/concepts/DB_About.html
PASSPORT_ATTRIBUTES = {
    # number: (our_name, is_private, type, default_value)
    '1': ('reg_date', False, str, None),
    '6': ('pdd_accepted_agreement', False, str, None),
    # '27': ('firstname', True, str, None),
    # '28': ('lastname', True, str, None),
    # '29': ('sex', True, str, None),
    # '30': ('birth_date', True, str, None),
    '31': ('country', False, str, None),
    # '32': ('city', True, str, None),
    '34': ('language', False, str, None),
    # '35': ('phone', True, str, None),
    '78': ('has_disk', False, bool, False),
    '80': ('undeletable', False, bool, False),
    '82': ('has_mobile_disk', False, bool, False),
    '83': ('has_music', False, bool, False),
    '84': ('has_desktop_disk', False, bool, False),
    '96': ('personalization', False, bool, False),
    '107': ('is_app_password_enabled', False, bool, False),
    '190': ('has_telemost', False, bool, False),
    '197': ('has_mail360', False, bool, False),
    '1003': ('is_2fa_enabled', False, bool, False),
    '1005': ('has_password', False, bool, False),
    '1009': ('is_enabled', False, bool, False),
    '1015': ('has_plus', False, bool, False),
    '1017': ('connect_organization_ids', False, list, []),
    '1031': ('pdd_org_id', False, str, None)
}
PASSPORT_NAMES_TO_ATTRIBUTES = {v[0]: k for k, v in PASSPORT_ATTRIBUTES.iteritems()}


class PlatformUserDetailsHelper(object):
    """Работа с деталями по пользователю, которые можно запрашивать при авторизации в Платформе.

    Без доп. похода в паспорта с method=userinfo.
    """
    DB_FIELDS = {'lang': 'userinfo.lang.uid',
                 'firstname': 'userinfo.firstname.uid',
                 'lastname': 'userinfo.lastname.uid'}
    ADDITIONAL_PARAMS = {'dbfields': ','.join(DB_FIELDS.values())}

    @classmethod
    def add_details_params(cls, params):
        params.update(cls.ADDITIONAL_PARAMS)

    @classmethod
    def construct_details_from(cls, data):
        db_fields = data.get('dbfields', {})
        if not db_fields:
            return {}

        return {'firstname': db_fields.get(cls.DB_FIELDS['firstname']),
                'lastname': db_fields.get(cls.DB_FIELDS['lastname']),
                'lang': db_fields.get(cls.DB_FIELDS['lang'])}


class BlackboxServiceBase(RequestsPoweredServiceBase):
    FAILED_AUTH_STATUSES = {'INVALID', 'DISABLED', 'EXPIRED', 'NOAUTH'}

    @staticmethod
    def _raise_if_error_in_result(result):
        if 'error' in result and result['error'] != 'OK':
            error_log.info(to_json(result))
            error = result['error']
            message = result.get('message', '')
            if error == 'accountwithloginrequired' or error == 'accountwithpasswordrequired':
                raise errors.PassportPasswordNeeded(message)
            if error == 'expired_token':
                raise errors.PassportTokenExpired(message)
            if ('status' in result and type(result['status']) is dict and
                        'value' in result['status'] and
                        result['status']['value'] in BlackboxServiceBase.FAILED_AUTH_STATUSES):
                raise errors.PassportCookieInvalid(result['error'])
            raise errors.PassportBadResult(message)

    def check_oauth_token(self, oauth_token, user_ip, with_details=False):
        """
        Проверка OAuth токена

        Формат ответа:
        https://docs.yandex-team.ru/blackbox/methods/oauth
        """
        params = {}
        if SERVICES_TVM_ENABLED:
            params.update(self._get_tvm_needed_params())

        if SERVICES_TVM_2_0_ENABLED:
            params['get_user_ticket'] = 'yes'

        params['method'] = 'oauth'
        params['oauth_token'] = oauth_token
        params['userip'] = user_ip
        params['attributes'] = PASSPORT_NAMES_TO_ATTRIBUTES['pdd_accepted_agreement']
        params['format'] = 'json'
        if with_details:
            PlatformUserDetailsHelper.add_details_params(params)

        try:
            response = self.request('GET', 'blackbox', params=params)
            result = from_json(response.content)
        except Exception as e:
            error_log.warn(e)
            raise errors.PassportNoResponse(), None, sys.exc_info()[2]

        self._raise_if_error_in_result(result)
        return result

    def check_session_id(self, session_id, ssl_session_id, user_ip, host):
        result = self._check_session_id(session_id, ssl_session_id, user_ip, host, False)
        self._raise_if_error_in_result(result)
        return {
            'uid': result['uid']['value'],
            'login': result['login'],
            'status': result['status']['value'],
            'ticket': result.get('ticket'),
            'user_ticket': result.get('user_ticket')
        }

    def check_multi_session_id(self, session_id, ssl_session_id, user_ip, host):
        result = self._check_session_id(session_id, ssl_session_id, user_ip, host, True)
        self._raise_if_error_in_result(result)
        user_infos = []
        for user in result['users']:
            if user['status']['value'] in BlackboxServiceBase.FAILED_AUTH_STATUSES:
                continue
            user_infos.append({
                'uid': user['uid']['value'],
                'login': user['login'],
                'status': user['status']['value'],
                'ticket': result.get('ticket'),
                'user_ticket': result.get('user_ticket')
            })
        return user_infos

    def _check_session_id(self, session_id, ssl_session_id, user_ip, host, multi_session):
        """
        Проверка сессионной куки, получение по ней информации

        TODO
        Если метод будет использоваться не только в платформе, то перенести сюда проверку authparser-ом
        http://doc.yandex-team.ru/blackbox/reference/MethodSessionID.xml

        Формат ответа:
        https://docs.yandex-team.ru/blackbox/methods/sessionid#response_format
        """
        params = {}
        if SERVICES_TVM_ENABLED:
            params.update(self._get_tvm_needed_params())

        if SERVICES_TVM_2_0_ENABLED:
            params['get_user_ticket'] = 'yes'

        params['method'] = 'sessionid'
        params['sessionid'] = session_id
        if ssl_session_id is not None:
            params['sslsessionid'] = ssl_session_id
        params['userip'] = user_ip
        params['host'] = host
        params['format'] = 'json'
        if multi_session:
            params['multisession'] = 1

        try:
            response = self.request('GET', 'blackbox', params=params)
            return from_json(response.content)
        except Exception as e:
            error_log.warn(e)
            raise errors.PassportNoResponse(), None, sys.exc_info()[2]

    def _get_tvm_needed_params(self):
        params = TVMSignatureParametersBuilder.build()
        params['getticket'] = 'yes'
        params['consumer'] = SERVICES_TVM_CONSUMER
        return params

    @staticmethod
    def _transform_url_for_logging(url):
        """Хэширует для логирования креденшиалы, содержащиеся в qs"""
        for qs_param_name in ('oauth_token', 'sessionid', 'sslsessionid'):
            url = CredentialSanitizer.sanitize_qs_parameter_in_url(qs_param_name, url)
        return url


class BlackboxService(BlackboxServiceBase):
    pass


class YateamBlackboxService(BlackboxServiceBase):
    def _get_tvm_needed_params(self):
        if self.get_tvm_ticket:
            return super(YateamBlackboxService, self)._get_tvm_needed_params()
        return {}


class Passport(RequestsPoweredServiceBase, Cached):
    api_error = errors.PassportNoResponse
    log = service_log
    _users = {'uid': {}, 'login': {}}
    avatars_url = settings.services['Passport']['avatars_url']
    blackbox_url = settings.services['Passport']['blackbox_url']
    passport_url = settings.services['Passport']['passport_url']

    attributes = PASSPORT_ATTRIBUTES

    aliases = {
        # number: (our_name, is_private, type, default_value)
        '1': ('has_portal_login', False, bool, False),
        '13': ('has_staff', False, bool, False),
        '12': ('is_mailish', False, bool, False),
        '24': ('sso_user', False, str, ''),
    }

    names_to_aliases = {v[0]: k for k, v in aliases.iteritems()}

    db_fields = {
        'userinfo.firstname.uid': ('firstname', True, str, None),
        'userinfo.lastname.uid': ('lastname', True, str, None),
        'userinfo.sex.uid': ('sex', True, str, None),
        'userinfo.birth_date.uid': ('birth_date', True, str, None),
        'userinfo.city.uid': ('city', True, str, None),
    }

    command_map = {
        'subscribe': 'admsubscribe',
        'unsubscribe': 'admsubscribe',
        'userinfo': 'userinfo',
        'is_from_pdd': 'userinfo',
        'get_portal_login': 'userinfo',
        'get_suid_dbid': 'userinfo',
        'check_oauth_token': 'oauth',
        'check_session_id': 'sessionid',
    }

    dbfields_map = {
        'userinfo.country.uid': 'country',
        'userinfo.firstname.uid': 'firstname',
        'userinfo.lastname.uid': 'lastname',
        'userinfo.lang.uid': 'language',
    }

    errors_map = {
        'accountwithloginrequired': errors.PassportPasswordNeeded,
        'accountwithpasswordrequired': errors.PassportPasswordNeeded,
        'expired_token': errors.PassportTokenExpired,
        'unknownuid': errors.PassportUnknownUIDError,
    }

    def process_passport(self, data):
        action = self.command_map[sys._getframe(1).f_code.co_name]
        return self._process_response(self.passport_url, action, data)

    def process_blackbox(self, data, **kw):
        action = self.command_map[sys._getframe(1).f_code.co_name]
        return self._process_response(self.blackbox_url, action, data, **kw)

    def _process_response(self, url, action, data, **kw):
        qformat = kw.get('format')
        error = None
        message = ''

        if qformat:
            data['format'] = qformat

        if 'userip' not in data:
            data['userip'] = mpfs.engine.process.hostip()

        try:
            response = self.request('GET', url % action, base_url='', params=data)
            if qformat == 'json':
                result = from_json(response.content)
                error = result.get('error')
                message = result.get('message', '')
            else:
                result = etree.fromstring(response.content)
                try:
                    error = result.find('error').text
                    message = result.find('message').text
                except Exception, e:
                    pass
        except Exception, e:
            error_log.warn(e)
            raise self.api_error(), None, sys.exc_info()[2]
        else:
            if error is not None and error != 'OK':
                message = message.encode('utf-8').strip() or 'Unknown error'
                if error in self.errors_map:
                    err_class = self.errors_map[error]
                else:
                    err_class = errors.PassportBadResult
                    error_log.error('Bad result from Passport. %s. Passport response: %s' % (message, response))
                raise err_class(message)
            else:
                return result

    def subscribe(self, uid, sid='cloud'):
        request_data = {'uid': uid, 'from': sid}
        resp = self.process_passport(request_data)
        return True

    def unsubscribe(self, uid, sid='cloud'):
        request_data = {'uid': uid, 'from': sid, 'unsubscribe': 'true'}
        resp = self.process_passport(request_data)
        return True

    def username(self, uid):
        request_data = {
            'uid'      : uid,
            'dbfields' : 'userinfo.firstname.uid,userinfo.lastname.uid,userinfo.lang.uid',
        }
        resp = self.process_blackbox(request_data)
        result = {
                  'login'    : resp.find('login').text,
                  }
        for field in resp.iterfind('dbfield'):
            result[self.dbfields_map[field.get('id')]] = field.text
        return result

    def is_from_pdd(self, address=None, uid=None):
        if not uid and not address:
            raise Exception("Either `uid` or `address` must be set.")

        request_data = {'sid': 'passport'}
        if uid:
            request_data['uid'] = uid
        elif address:
            request_data['login'] = address

        resp = self.process_blackbox(request_data)
        reg_info = resp.find('uid')
        if reg_info.text and reg_info.get('mx') and reg_info.get('hosted') == '1':
            return True
        else:
            return False

    def is_yandex_internal_qa_user(self, email=None, uid=None):
        """
        Проверяет наличие портального алиаса для пользователя и его соответствие маске yndx-* или yandex-team-*.
        Такие пользователи могут быть созданы только из внутренней сети Яндекса.
        Требуется использовать именно алиас пользователя на портале, так как для некоторых типов аккаунтов ограничение
        на создание аккаунтов по данной маске не работает.
        Справочная информация о типах алиасов: https://wiki.yandex-team.ru/passport/dbmoving/#tipyaliasov
        """
        if email is None and uid is None:
            raise Exception("Either `guest_uid` or `email` must be set.")

        if not uid:
            # для
            if any(email.endswith(x) for x in ['@yandex.ru', '@yandex.com']):
                uid = self.userinfo(login=email.split('@')[0])['uid']
            else:
                uid = self.userinfo(login=email)['uid']

        portal_login = self.get_portal_login(uid=uid)
        return portal_login and portal_login.startswith("yndx-") | portal_login.startswith("yandex-team-")

    def get_portal_login(self, uid=None):
        """
        Возвращает алиас пользователя на портале.
        """
        if not uid:
            raise Exception("`uid` must be set.")

        request_data = {'sid': 'passport', 'uid': uid, 'aliases': '1'}

        resp = self.process_blackbox(request_data)
        aliases = resp.find('aliases')
        if not aliases:
            return None
        for alias in aliases:
            if alias.get('type') == '1':
                return alias.text
        return None

    def get_suid_dbid(self, uid):
        result = {}
        fields = {
                  'subscription.suid.2' : 'suid',
                  'hosts.db_id.2' : 'db',
                  }
        request_data = {
                        'uid'      : uid,
                        'dbfields' : ','.join(fields),
                        }
        resp = self.process_blackbox(request_data)
        for field in resp.findall('dbfield'):
            result[fields[field.get('id')]] = field.text
        return result

    def userinfo_summary(self, uid=None, login=None, fields=()):
        fields = fields or ('login', 'username', 'email', 'decoded_email', 'public_name')
        user_info = self.userinfo(uid, login)
        result = dict((k, user_info[k]) for k in fields)
        try:
            result['uid'] = user_info['uid']
        except KeyError:
            result['uid'] = uid
        return result

    @classmethod
    def reset(cls):
        cls._users = {'uid' : {}, 'login' : {}}

    def is_user_exists(self, uid):
        user_info = self.userinfo(uid=uid)
        if user_info.get('uid'):
            return True
        return False

    def public_userinfo(self, uid, raise_if_does_not_exist=False):
        """
        Возвращает публичную часть userinfo.

        Возвращает только паспортную часть публичной информации.
        Полную публичную информацию бери тут:
            `mpfs.core.user.standart.StandartUser -> public_info`
        """
        result = {}
        try:
            user_info = self.userinfo(uid=uid)
        except errors.PassportError:
            return result

        if raise_if_does_not_exist and not user_info.get('uid'):
            raise errors.PassportUserDoesNotExistError()

        result.update({
            'uid': uid,
            'login': user_info['login'],
            'username': user_info['username'],
            'locale': user_info['language'],
            'display_name': user_info['display_name'],
            'public_name': user_info['public_name'],
        })
        return result

    def _prepare_userinfo_json_request_params(self, load_all_emails=False, add_private_data=False):
        args = {
            'format': 'json',
            'userip': mpfs.engine.process.hostip(),
            'regname': 'yes',
            'get_public_name': 'yes',
            'aliases': ','.join(self.aliases.keys())
        }

        attributes = []
        db_fields = []
        for (identifier, (name, is_private, _type, default_value)), lst in (
            zip(self.attributes.items(), [attributes] * len(self.attributes))
            + zip(self.db_fields.items(), [db_fields] * len(self.db_fields))
        ):
            if not add_private_data:
                # поля, которые добавляем только при передаче приватных данных
                if name == 'personalization':
                    continue
                if is_private and name not in ('firstname', 'lastname'):
                    # имя и фамилию запрашиваем всегда
                    continue
            lst.append(identifier)
        args['attributes'] = ','.join(attributes)
        args['dbfields'] = ','.join(db_fields)

        if load_all_emails:
            args['emails'] = 'getall'
        else:
            args['emails'] = 'getyandex'

        return args

    def _process_userinfo_json_data(self, data):
        """
        Распарсить JSON ответ ручки `userinfo` и сформировать из него необходимую структуру.

        В случае если при запросе пользовательской информации в паспорте произошла ошибка или
        же мы передали некорректный идентификатор, то в результат попадет структура пользовательской информации,
        но у нее поле `uid` будет иметь значение None.
        """
        result = []
        for user_data in data.get('users', []):
            obj = {
                'id': None,
                'uid': None,
                'login': '',
                'username': None,
                'display_name': '',
                'public_name': '',
                'avatar': '',
                'email': '',
                'decoded_email': '',
                'karma': 0,
                'locations': {},
                'pdd': False,
                'pdd_domain': None,
                'address-list': None
            }
            for number, (name, is_private, _type, default_value) in self.attributes.iteritems():
                obj[name] = default_value
            for number, (name, is_private, _type, default_value) in self.aliases.iteritems():
                obj[name] = default_value
            for db_name, (name, is_private, _type, default_value) in self.db_fields.iteritems():
                obj[name] = default_value

            result.append(obj)
            obj['id'] = user_data['id']  # паспорт всегда его возвращает, чтоб можно было идентифицировать юзера
            if 'exception' in user_data or 'error' in user_data:
                continue

            if not user_data['uid']:
                # неверный uid
                continue

            obj['uid'] = user_data['uid']['value']
            if user_data['login'] is not None:
                obj['login'] = user_data['login']

            # итерируемся по всем атрибутам и алиасам и заносим их в структуру
            _attributes = user_data.get('attributes', {}).items()
            _aliases = user_data.get('aliases', {}).items()
            _db_fields = user_data.get('dbfields', {}).items()
            _map = {
                'attribute': self.attributes,
                'alias': self.aliases,
                'db_field': self.db_fields
            }
            for ((identifier, value), key_type) in (
                zip(_attributes, ['attribute'] * len(_attributes))
                + zip(_aliases, ['alias'] * len(_aliases))
                + zip(_db_fields, ['db_field'] * len(_db_fields))
            ):
                if str(identifier) in _map[key_type]:
                    field_name, is_private, field_type, default_value = _map[key_type][str(identifier)]
                    if field_name not in obj:
                        continue

                    if field_type == str and value is not None:
                        obj[field_name] = value
                    elif field_type == bool and value is not None:
                        obj[field_name] = True
                    elif field_type == list and value is not None:
                        obj[field_name] = value.split(',')

            # заносим в структуру остальные необходимые поля
            obj['address-list'] = user_data.get('address-list', [])
            for idx, item_ in enumerate(obj['address-list']):
                try:
                    address = item_['address']
                    if idx == 0:
                        obj['email'] = address
                    if item_['default']:
                        obj['email'] = address
                except KeyError:
                    pass

            display_name = user_data.get('display_name', {})
            obj['display_name'] = display_name.get('name', '')
            obj['public_name'] = display_name.get('public_name', '')

            avatar = display_name.get('avatar', {}).get('default')
            if avatar:
                obj['avatar'] = self.avatars_url % avatar

            if user_data['karma'].get('value') is not None:
                obj['karma'] = user_data['karma']['value']

            try:
                obj['decoded_email'] = decode_email(obj['email'])
            except ValueError:
                pass

            for location_type, locations in user_data.get('locations', {}).iteritems():
                obj['locations'] = locations

            if obj['sex']:
                obj['sex'] = {'1': 'M', '2': 'F'}.get(str(obj['sex']), 'U')

            if user_data['uid'].get('hosted'):
                obj['pdd'] = user_data['uid']['hosted']

            if user_data['uid'].get('domid'):
                obj['domain_id'] = user_data['uid']['domid']

            if user_data['uid'].get('domid'):
                # ключ не включается в ответ, если домен не был найден в базе ПДД
                obj['pdd_domain'] = user_data['uid']['domain']

            if obj['firstname'] and obj['lastname']:
                obj['username'] = '%s %s' % (obj['firstname'], obj['lastname'])
            else:
                obj['username'] = obj['display_name']

            if obj['reg_date']:
                obj['reg_date'] = int(obj['reg_date'])

            if obj['connect_organization_ids']:
                obj['external_organization_ids'] = [int(org_id.strip()) for org_id in obj['connect_organization_ids']]

            if user_data.get('aliases', {}).get(self.names_to_aliases['has_staff']):
                obj['yateam_login'] = user_data['aliases'][self.names_to_aliases['has_staff']]

        return result

    def bulk_userinfo(self, uids, load_all_emails=False, add_private_data=False):
        """
        Возвратить пользовательскую информацию из паспорта для списка пользователей.

        Если для пользователя произошла ошибка в результате запроса или передан некорректный пользователь, то информация
        по нему будет включена в ответ в виде стандартной структуры, но параметр `uid` будет иметь значение None.

        Ссылки на документацию приведены в шапке модуля.
        """
        if isinstance(uids, basestring) or not isinstance(uids, collections.Iterable):
            raise TypeError('Unsupported `uids` type (%s)' % uids.__class__.__name__)

        if not uids:
            return []

        args = self._prepare_userinfo_json_request_params(load_all_emails=load_all_emails,
                                                          add_private_data=add_private_data)
        args['uid'] = ','.join(map(str, uids))

        data = self._process_userinfo_json_data(
            self._process_response(self.blackbox_url, 'userinfo', args, format='json')
        )
        return data

    def userinfo(self, uid=None, login=None, all_emails=False, personal=False):
        if not uid and not login:
            raise Exception()

        try:
            if uid is not None:
                result = Passport._users['uid'][uid]
            else:
                result = Passport._users['login'][login]
        except KeyError:
            pass
        else:
            return result

        args = self._prepare_userinfo_json_request_params(load_all_emails=all_emails, add_private_data=personal)
        if uid is not None:
            args['uid'] = uid
        else:
            args['login'] = login

        passport_response = self._process_response(self.blackbox_url, 'userinfo', args, format='json')
        err_msg = None
        if 'users' not in passport_response or type(passport_response['users']) is not list:
            err_msg = 'No user list returned by Passport'
        elif not passport_response['users']:
            err_msg = 'Empty list returned by Passport'
        elif 'error' in passport_response['users'][0]:
            err_msg = passport_response['users'][0]['error']
        if err_msg is not None:
            error_log.error('Bad result from Passport. %s. Passport response: %s' % (err_msg, passport_response))
            raise errors.PassportBadResult(err_msg)

        data = self._process_userinfo_json_data(passport_response)
        r = data[0]
        Passport._users['uid'][r['uid']] = r
        Passport._users['login'][r['login']] = r

        return r

    def get_hosted_domains(self, domain):
        """Вернуть список информации о доменах организации,
        за которой зарегистрирован домен ``domain``.

        ..notes:: Домены бывают двух типов: основной, алиас.
            Один домен может переходить в другой, а так же удаляться.
            Тогда есть три типа ответов:
                1. Если ``domain`` - основной, то в ответ придут и основной и алиасы.
                2. Если ``domain`` - алиас, то в ответ придет алиас.
                3. Если домен не найден или был удален, то ответ пустой.

        :type: str
        :rtype: list[dict]
        """
        query_data = {'domain': domain, 'aliases': 'true'}
        resp = self._process_response(self.blackbox_url, 'hosted_domains',
                                      query_data, format='json')
        return resp['hosted_domains']

    def get_all_domain_list(self, domain):
        """Вернуть список доменов организации,
        за которой зарегистрирован домен ``domain``.

        :type domain: str
        :rtype: list[str]
        """
        hosted_domains = self.get_hosted_domains(domain)

        domains = set()
        for domain_info in hosted_domains:
            domains.add(domain_info['domain'])
            domains.add(domain_info['master_domain'])
            for slave_domain in domain_info['slaves'].split(','):
                domains.add(slave_domain)
        return [item for item in domains if item]


class PassportDedicated(Passport):
    pass


class PassportSID(object):
    TELEMOST = 'telemost'


class PassportAPI(RequestsPoweredServiceBase):
    static_params = {'consumer': 'cloud-api'}

    def subscribe(self, uid, sid):
        response = self.request('POST', '/1/account/%(uid)s/subscription/%(sid)s/' % {'uid': uid,
                                                                                      'sid': sid},
                                self.static_params)
        return from_json(response.content)

    def unsubscribe(self, uid, sid):
        response = self.request('POST', '/1/account/%(uid)s/subscription/%(sid)s/' % {'uid': uid,
                                                                                      'sid': sid},
                                self.static_params)
        return from_json(response.content)


passport_api = PassportAPI()
passport = Passport()
passport_dedicated = PassportDedicated()
blackbox = BlackboxService()
yateam_blackbox = YateamBlackboxService()
