# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис Превьювер (ex. Кладун)

"""
import sys

from lxml import etree
from urllib import quote
from urlparse import urljoin

import mpfs.engine.process

from mpfs.common import errors
from mpfs.engine.http import client
from mpfs.engine.queue2.utils import get_callback_host
from mpfs.core.services.common_service import Service
from mpfs.common.util import from_json
from mpfs.common.util.urls import urlencode
from mpfs.core.filesystem.hardlinks.common import FileChecksums


service_log = mpfs.engine.process.get_service_log('previewer')


class RegeneratePreviewResult(object):
    def __init__(self, pmid, video_info=None, original_width=None, original_height=None, rotate_angle=None):
        self.pmid = pmid
        self.video_info = video_info
        self.original_width = original_width
        self.original_height = original_height
        self.rotate_angle = rotate_angle

    def has_video_info(self):
        return self.video_info is not None


class Previewer(Service):
    name = 'previewer'
    action = ''
    api_error = errors.KladunNoResponse
    log = service_log
    # оверайдится из конфигов
    regenerate_preview_timeout = None
    recalc_digests_timeout = None

    def __init__(self, *args, **kwargs):
        super(Previewer, self).__init__(*args, **kwargs)
        if not self.callback_url:
            self.callback_url = 'http://' + mpfs.engine.process.hostname()
        if self.action:
            self.base_url = self.base_url + self.action + '/'

    def _get_callback_url(self, uid, oid):
        callback_url = 'http://' + get_callback_host() + '/service/kladun_callback'
        return callback_url + '?uid={}&oid={}'.format(uid, oid)

    def open_url(self, url, data=None, json=False, timeout=None, retry=True):
        post = self._preprocess_body_arguments(data)
        try:
            response = super(Previewer, self).open_url(url, post=post, timeout=timeout, retry=retry)
        except errors.MPFSError, e:
            if e.data.get('code') == 410:
                raise errors.OperationNotFoundError(data=e.data)
            raise e

        try:
            if json:
                return from_json(response)
            return etree.fromstring(response)
        except Exception as e:
            raise errors.KladunBadResponse, e, sys.exc_info()[2]

    def check_service_is_alive(self, url):
        try:
            super(Previewer, self).open_url(url, method='HEAD')
        except errors.KladunNoResponse, e:
            if e.data.get('code') == 404:
                # Если превьювер дал внятный ответ, то считаем, что он жив.
                return True
            return False
        return True

    def status(self, url, response=''):
        if not response:
            element = self.open_url(url)
        else:
            service_log.debug(response)
            element = etree.fromstring(response)
        return element

    def _preprocess_body_arguments(self, body_args):
        if not body_args:
            return {}
        body_args.update({
            'api': self.api_ver,
            'callback': self._get_callback_url(body_args['uid'], body_args['oid'])
        })
        str_uid = body_args.get('uid', None)
        if str_uid:
            try:
                int(str_uid)
            except ValueError:
                del body_args['uid']
                body_args['special-uid'] = str_uid
        free_space = body_args.get('free_space', None)
        if free_space:
            del body_args['free_space']
            body_args['max-file-size'] = free_space
        return body_args

    def get_file_checksums(self, file_mid):
        """Получить хеши и размер файла

        :param str file_mid: stid файла
        :rtype: FileChecksums
        """
        resp = self._recalc_digests(file_mid, generate_digest=False)
        return FileChecksums(resp['md5'], resp['sha256'], int(resp['content-length']))

    def regenerate_digest(self, file_mid):
        """Сделать запрос на генерацию нового дайджеста

        :param str file_mid: stid файла
        :return: digest_stid
        """
        return self._recalc_digests(file_mid, generate_digest=True)['webdav-digest-mulca-id']

    @staticmethod
    def get_image_info_from_response(dom_element):
        image_info = {'original_width': None, 'original_height': None, 'rotate_angle': None}
        try:
            original_width = int(dom_element.get('original-width', ''))
            original_height = int(dom_element.get('original-height', ''))
        except ValueError:
            pass
        else:
            if original_width > 0 and original_height > 0:
                image_info['original_width'] = original_width
                image_info['original_height'] = original_height

        try:
            rotate_angle = int(dom_element.get('rotate-angle', ''))
        except ValueError:
            pass
        else:
            if rotate_angle in (0, 90, 180, 270):
                image_info['rotate_angle'] = rotate_angle
        return image_info

    def _recalc_digests(self, file_mid, generate_digest=False):
        params = {
            'mulca-id': file_mid,
            'with-webdav': 'false',
        }
        if generate_digest:
            params['with-webdav'] = 'true'

        url = urljoin(self.base_url, 'recalc-digests')
        url += '?%s' % urlencode(params)
        return self.open_url(url, timeout=self.recalc_digests_timeout, json=True)

    def regenerate_preview(self, name, size, mimetype, file_mid, timeout=None):
        """Сделать запрос на генерацию нового ``pmid``

        Для видео файлов в результате дополнительно есть video_info

        :param str name: имя файла
        :param str size: размер файла
        :param str mimetype: mimetype файла
        :param str file_mid: stid файла
        :rtype: RegeneratePreviewResult
        """
        if timeout is None:
            timeout = self.regenerate_preview_timeout

        query = {
            'file-name': name,
            'file-size': size,
            'content-type': mimetype,
            'mulca-id': file_mid
        }

        url = urljoin(self.base_url, 'regenerate-preview')
        url += '?%s' % urlencode(query)
        dom = self.open_url(url, timeout=timeout)

        pmid = dom.find('mulca-id').text
        data = {}
        video_info = None
        video_info_el = dom.find('video-info')
        if video_info_el:
            video_info = from_json(video_info_el.text)
        image_info_el = dom.find('image-info')
        data['video_info'] = video_info

        data['original_width'] = None
        data['original_height'] = None
        data['rotate_angle'] = None

        if image_info_el is not None:
            image_info = self.get_image_info_from_response(image_info_el)
            data.update(image_info)
        return RegeneratePreviewResult(pmid, **data)

    def generate_album_preview(self, pmid, user_name, album_title, timeout=None, retry=True):
        """
        Сгенерировать обложку альбома для поделения в соц. сетях.

        :param str pmid: pmid обложки альбома
        :param str user_name: Отображаемое на обложке имя пользователя
        :param str album_title: Отображаемое на обложе название альбома
        :return: stid соц. обложки
        :rtype: str
        """
        query = {
            'mulca-id': pmid,
            'user-name': quote(user_name.encode('utf-8')),
            'album-name': quote(album_title.encode('utf-8')),
        }

        url = urljoin(self.base_url, 'generate-album-preview')
        url += '?%s' % '&'.join(['%s=%s' % i for i in query.iteritems()])
        if timeout is None:
            timeout = self.regenerate_preview_timeout
        resp = self.open_url(url, json=True, timeout=timeout, retry=retry)
        return resp.get('album-preview-mulca-id')


class UpdateCheckSum(Previewer):
    """
        Обновление всех контрольных сумм файла.
    """
    action = 'recalc-digests'

    def post_request(self, file_mid):
        """

        :param file_mid: mulca id
        :return: md5, sha256, size, digest_mid
        """
        url = self.base_url + '?mulca-id=%s' % file_mid
        response = self.open_url(url, json=True, timeout=60)
        result = {
            'md5': response['md5'],
            'sha256': response['sha256'],
            'size': response['content-length'],
            'digest_mid': response['webdav-digest-mulca-id'],
        }
        return result
