# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис отсылания пушей

"""
import sys
import types
import urllib
import urlparse

import mpfs.engine.process
import mpfs.common.errors as errors

from urllib2 import HTTPError

from collections import namedtuple
from mpfs.common.util import to_json, from_json
from mpfs.core.services.common_service import RequestsPoweredServiceBase, Service
from mpfs.engine.http import client as http_client

service_log = mpfs.engine.process.get_service_log('xiva')
error_log = mpfs.engine.process.get_error_log()

XivaSubscription = namedtuple('XivaSubscription', [
    'uid',
    'uuid',
    'xiva_id',
])


class XivaNotificationStatus(object):

    def __init__(self, code, body):
        self.code = code
        self.body = body

    def is_success(self):
        if self.code in [200, 202, 204]:
            return True
        else:
            return False


class XivaBaseService(RequestsPoweredServiceBase):
    """
    Класс общий для XivaSubscribeCommonService и XivaSendService
    """
    def get_service_headers(self, headers=None, timeout=None, *args, **kwargs):
        headers = super(XivaBaseService, self).get_service_headers(headers)

        # timeout = timeout or self.timeout
        # if timeout:
        #     headers['X-Request-Timeout'] = str(int(timeout * 1000))  # время в мс а не в секундах

        ycrid = mpfs.engine.process.get_cloud_req_id()
        if ycrid:
            headers['X-Request-Id'] = ycrid
        return headers


class XivaSubscribeCommonService(XivaBaseService):
    """
    Документация: https://wiki.yandex-team.ru/yxiva/api/v2/.

    API v2 Xiva: https://wiki.yandex-team.ru/yxiva/api/v2/
    Про фильтры в Xiva: https://wiki.yandex-team.ru/yxiva/filters/
    Подробней о Xiva: https://wiki.yandex-team.ru/nordsturm/yxiva-general/
    """
    subscribe_url_path = '/v2/subscribe/url'
    subscribe_app_path = '/v2/subscribe/app'
    unsubscribe_app_path = '/v2/unsubscribe/app'
    subscribtions_list_path = '/v2/list'
    timeout = 5.0
    base_url = None
    token = None
    """
    Токен подписки, полученный от транспорта нотификаций(xiva). Берем из конфигов.
    """

    def __init__(self):
        super(XivaSubscribeCommonService, self).__init__()
        self.static_headers = {'Authorization': 'Xiva %s' % self.token}

    def subscribe_app(self, service, uuid, push_token, app_name, platform, uid=None, filter_=None, device_id=None):
        """
        Подписать мобильное приложение

        Мапинг на: https://wiki.yandex-team.ru/yxiva/api/v2/#podpisatmobilnoeprilozhenie

        :param str service: Название сервиса, на который подписываем
        :param str uuid: Уникальный идентификатор экземпляра приложения в формате UUID
        :param str push_token: Выдается Apple, Google
        :param str app_name: Название приложения. Пример: "ru.yandex.mail.new"
        :param str platform: Идентификатор платформы: 'ios', 'gcm', 'wns', 'courier'
        :param str uid: Паспортный uid или идентификатор в формате [a-zA-Z0-9][a-zA-Z0-9:@.-]*>
        :param dict filter_: Фильтр версии 2
        :return: None
        """
        assert isinstance(filter_, (dict, types.NoneType))
        params = {
            'service': service,
            'uuid': uuid,
            'app_name': app_name,
            'platform': platform,
        }
        if uid:
            params['uid'] = uid
        if device_id:
            params['device'] = device_id

        data = {'push_token': push_token}
        if filter_:
            data['filter'] = to_json(filter_)

        try:
            self.request('POST', relative_url=self.subscribe_app_path, params=params, data=data)
        except HTTPError as e:
            if e.code == 400:
                if e.msg == 'filter size limit exceeded':
                    raise errors.XivaLimitExceeded(e.msg)
                if e.msg == 'application %s for platform fcm is not registered' % app_name:
                    raise errors.XivaAppNotRegistered(e.msg)
                if 'invalid characters' in e.msg:
                    raise errors.XivaBadToken(e.msg)
            raise

    def unsubscribe_app(self, service, uuid, push_token=None, uid=None):
        """
        Отпписать мобильное приложение

        Мапинг на: https://wiki.yandex-team.ru/yxiva/api/v2/#otpisatmobilnoeprilozhenie

        :param str service: Название сервиса.
        :param str uuid: Уникальный идентификатор экземпляра приложения в формате UUID
        :param str push_token: Выдается Apple, Google
        :param str uid: Паспортный uid или идентификатор в формате [a-zA-Z0-9][a-zA-Z0-9:@.-]*>
        :return: None
        """
        params = {
            'service': service,
            'uuid': uuid,
        }
        if push_token:
            params['push_token'] = push_token
        if uid:
            params['uid'] = uid

        self.request('POST', relative_url=self.unsubscribe_app_path, params=params)

    def subscribe_url(self, callback, service, uid, session='dummy', filter_=None):
        """
        Подписаться по callback из бэкенда

        Мапинг на: https://wiki.yandex-team.ru/yxiva/api/v2/#podpisatsjapocallbackizbjekenda

        :param str callback: URL хендлера обработки нотификаций на клиенте
        :param str service: Название сервиса, на который подписываем
        :param str uid: Паспортный uid
        :param str session: Уникальный идентификатор пользовательской сессии. Если callback уникальный, то можно оставить `dummy`
        :param dict filter_: Фильтр версии 2
        :return: None
        """
        assert isinstance(filter_, (dict, types.NoneType))
        params = {
            'service': service,
            'uid': uid,
            'callback': callback,
            'session': session
        }

        if filter_:
            data = {'filter': to_json(filter_)}
        else:
            data = None

        self.request('POST', relative_url=self.subscribe_url_path, params=params, data=data)

    def subscriptions_list(self, uid, service='disk-json'):
        """
        Получить список подписок

        :param str service: Название сервиса
        :param str uid: Паспортный uid
        :return: list
        """

        params = {
            'service': service,
            'uid': uid,
        }

        response = self.request('GET', relative_url=self.subscribtions_list_path, params=params)
        response_content = from_json(response.content)
        subscriptions = []
        for each in response_content:
            subscriptions.append(XivaSubscription(uid, each.get('uuid'), each['id']))
        return subscriptions


class XivaSubscribeDataSyncService(XivaSubscribeCommonService):
    pass


class XivaSubscribeNotifierService(XivaSubscribeCommonService):
    pass


class XivaSendService(XivaBaseService):
    """
    Класс для отправки событий в Xiva

    https://wiki.yandex-team.ru/yxiva/api/v2/#otpravitsoobshhenie
    """
    batch_send_path = '/v2/batch_send'
    send_path = '/v2/send'
    host = None
    token = None

    def __init__(self):
        super(XivaSendService, self).__init__()
        self.static_headers = {'Authorization': 'Xiva %s' % self.token}

    def send(self, uid, event_name, payload, keys=None, connection_id=None, repack=None):
        """
        Отправить сообщение в Xiva

        :param uid: uid пользователя, у кого произошло событие
        :param event_name: имя события. Пример: diff
        :param payload: тело события, которое получет клиент "как есть". Это поле попадает под `json.dumps`!
        :param keys: доп. поля, по которым ксива может фильтровать событие. Читай доку Xiva
        :param connection_id: идентификатор вкладки для веб клиентов
        :return: None
        """
        params = {
            'uid': uid,
            'event': event_name,
        }
        if connection_id:
            params['session'] = connection_id
        body = {
            'payload': payload
        }

        if keys:
            body['keys'] = keys

        if repack:
            body['repack'] = repack

        self.request('POST', relative_url=self.send_path, params=params, data=to_json(body))

    def batch_send(self, recipients, event, payload, keys=None):
        """
        Отправить сообщение в Xiva
        http://push.yandex.ru/doc/guide.html#api-reference-batch-send

        :param recipients: идентификаторы подписок типа [{uid: xiva_id_1}, {uid: xiva_id_2}]
        :param event: имя события. Пример: diff
        :param payload: тело события, которое получет клиент "как есть". Это поле попадает под `json.dumps`!
        :param keys: доп. поля, по которым ксива может фильтровать событие. Читай доку Xiva
        :return: list из статуса подписок XivaNotificationStatus в таком же порядке, как в subscriptions на входе
        """
        if not recipients:
            return []

        params = {
            'event': event,
        }
        recipients_list = [recipient for recipient in recipients]
        body = {
            'recipients': recipients_list,
            'payload': payload,
        }
        if keys:
            body['keys'] = keys
        response = self.request('POST', relative_url=self.batch_send_path, params=params, data=to_json(body))
        push_status = from_json(response.content)

        result = []
        for status in push_status.get('results', []):
            result.append(XivaNotificationStatus(status['code'], status.get('body', None)))
        return result


class XivaSubscribeService(object):
    """
    Класс для работы c подписками в Xiva для MPFS.

    Используется только для нужд MPFS.
    Является адаптером к `XivaSubscribeCommonService`.
    Умеет подписывать только на один сервис - "disk-json"

    Про webdav мобильные токены: https://wiki.yandex-team.ru/YandexMobile/pushkin/API2/#register
    """
    service_name = 'disk-json'

    token_xiva_params_map = {
        'c': 'app_name',
        'p': 'platform',
        't': 'push_token',
        'd': 'uuid',
    }
    token_xiva_platform_map = {
        'i': 'ios',
        'a': 'gcm',
        'w': 'wns',
        'c': 'courier',
        'hms': 'hms',
    }

    @classmethod
    def parse_app_token(cls, app_token):
        """
        Распарсить токен, прилетающий от моб. клиента и превести его к параметрам ручек xiva
        """
        if isinstance(app_token, basestring):
            try:
                app_token = from_json(app_token)
            except ValueError:
                try:
                    app_token = from_json(urllib.unquote(app_token))
                except ValueError:
                    raise errors.XivaBadToken("Cant parse app token: %s" % app_token)

        if not isinstance(app_token, dict) or cls.token_xiva_params_map.viewkeys() != app_token.viewkeys():
            raise errors.XivaBadToken("Bad mobile app token: %s" % app_token)

        result = {}
        for token_name, token_value in app_token.iteritems():
            param_name = cls.token_xiva_params_map[token_name]
            if not token_value:
                raise errors.XivaBadToken("Bad mobile app token: %s -> `%s` is empty" % (app_token, param_name))
            if param_name == 'platform':
                token_value = cls.token_xiva_platform_map[token_value]
            if param_name == 'push_token' and token_value == 'BLACKLISTED':
                raise errors.XivaBadToken("Bad mobile app token: %s" % app_token)
            result[param_name] = token_value
        return result

    def subscribe_app(self, uid, app_token, xiva_filter=None):
        """
        Подписать мобильное приложение из бэкенда

        :param uid: uid пользователя, у кого произошло событие
        :param app_token: webdav-токен, получаемый от моб. устройств.
        :param xiva_filter: правила, по которым xiva фильтрует события. Читай доку
        :return: None
        """
        token_params = self.parse_app_token(app_token)
        XivaSubscribeCommonService().subscribe_app(
            self.service_name,
            token_params['uuid'],
            token_params['push_token'],
            token_params['app_name'],
            token_params['platform'],
            uid=uid,
            filter_=xiva_filter
        )

    def unsubscribe_app(self, uid, app_token):
        """
        Отписать мобильное приложение из бэкенда

        :param uid: uid пользователя, у кого произошло событие
        :param app_token: webdav-токен, получаемый от моб. устройств.
        :return: None
        """
        token_params = self.parse_app_token(app_token)
        XivaSubscribeCommonService().unsubscribe_app(
            self.service_name,
            token_params['uuid'],
            push_token=token_params['push_token'],
            uid=uid,
        )

    def subscribe_url(self, uid, callback, session='dummy', xiva_filter=None):
        """
        Подписаться по callback из бэкенда

        :param uid: uid пользователя, у кого произошло событие
        :param callback: урл, который дернется в случае получения подходящего события.
        :param session: уникальный идентификатор пользовательской сессии. Если callback уникальный, то можно оставить `dummy`
        :param xiva_filter: правила, по которым xiva фильтрует события. Читай доку
        :return: None

        При одинаковых параметрах callback, service, uid, session, xiva_filter будет создана одна подписка
        """
        XivaSubscribeCommonService().subscribe_url(
            callback,
            self.service_name,
            uid,
            session=session,
            filter_=xiva_filter
        )


class PushSender(Service):
    """
    DEPRECATED!

    Отпилить вместе с браузером!
    """

    name = 'push'
    api_error = errors.PushNoResponse
    log = service_log

    def send(self, data, listener_url, **kwargs):
        '''
        push на адрес listener
        '''
        default_url_params = {
            'operation': 'action',
            'connection_id': '',
            'new_version': 1
        }
        for k, v in default_url_params.iteritems():
            if k not in kwargs:
                kwargs[k] = v
        url_params = '?uid=%(uid)s&lcn=%(new_version)s&operation=%(operation)s' % kwargs
        cid = kwargs.get('connection_id')

        if cid:
            url_params = url_params + '&connection_id=%s' % cid

        return http_client.open_and_return_code(
            listener_url + url_params,
            timeout=self.timeout,
            log=service_log,
            pure_data=data,
            retry=False,
        )


xiva_send = XivaSendService()
xiva_subscribe = XivaSubscribeCommonService()
