# -*- coding: utf-8 -*-
import traceback

from abc import abstractmethod
from enum import Enum

from kombu.utils import uuid

import mpfs.common.errors as errors
from mpfs.common.static.tags import START_EXECUTION_IMMEDIATELY

from mpfs.engine.process import get_error_log
from mpfs.common.util import from_json, to_json
from mpfs.config import settings
from mpfs.core.services.common_service import RequestsPoweredServiceBase
from mpfs.engine.queue2.celery import BaseTask, set_command_context


SYSTEM_HTTP_RETRIES = settings.system['http_retries']

error_log = get_error_log()


# Код для работы с onetime task на queller (если нужно получать статус таски из queller)
#
# Как создать новый "onetime-task"
# 1) Создать класс задачи отнаследовавшись от QuellerTask
#    a) Реализовать put для постановки таска в очередь
#    b) Переопределить метод run(self, *args, **kwargs) - собственно, выполнение задачи.
# 2) Создать к этому классу handler и прописать его в mpfs/engine/queue2/celery.py и  mpfs/engine/queue2/job_types.py
# 3) Зарегистрировать таск в queller (только через PG - лимит локальной очереди = 0)
#
# Как использовать его из кода:
# Если нужно разово(не пачками) ставить задачи, то используем QuellerTask.put
#
# Если статус получать не нужно, можно просто сделать функцию + декоратор в job_handlers
# (также зарегистрировав в job_types и queue2/celery.py)


class TaskStatus(Enum):
    READY = 'ready'
    STARTING = 'starting'
    RUNNING = 'running'
    COMPLETED = 'completed'
    FAILED = 'failed'

    @classmethod
    def not_finished(cls):
        return [cls.READY, cls.RUNNING, cls.STARTING]


class QuellerService(RequestsPoweredServiceBase):
    """Работа с queller через HTTP API.

    https://wiki.yandex-team.ru/disk/worker/#ruchkipgtasksactioncontainer
    """
    def __init__(self):
        self.http_retries_per_dc = SYSTEM_HTTP_RETRIES
        self.base_url_per_dc = None
        super(QuellerService, self).__init__()

    def get_tasks_count(self, task_cls_or_str, statuses=None):
        """Получение количества тасков в очереди

        https://wiki.yandex-team.ru/disk/worker/#poluchitkolichestvozapuskovodnorazovyxzadachpostatusu

        :param type task_cls_or_str: наследник QuellerTask или строка имя таска
        :param (list|TaskStatus) statuses: статус или массив статусов таска для фильтрации по ним. None - любой статус
        """
        if isinstance(task_cls_or_str, basestring):
            task_full_name = task_cls_or_str
        elif issubclass(task_cls_or_str, QuellerTask):
            task_full_name = task_cls_or_str.get_queue_task_name()
        else:
            raise TypeError('"QuellerTask" subclass or str required. Got %r' % task_cls_or_str)
        statuses = self._validate_statuses(statuses)
        params = {
            'task': task_full_name,
            'status': [s.value for s in statuses],
        }
        response = self.request('GET', '/tasks/get-tasks-count', params=params)
        resp_json = from_json(response.content)

        if 'result' in resp_json:
            return int(resp_json['result']['count'])
        else:
            raise errors.APIError(response)

    def _validate_statuses(self, statuses):
        if statuses is None:
            return list(TaskStatus)
        elif isinstance(statuses, TaskStatus):
            statuses = [statuses]
        elif isinstance(statuses, (list, tuple)):
            for status in statuses:
                if not isinstance(status, TaskStatus):
                    raise TypeError('Expected `TaskStatus`. Got: "%s"' % status)
        else:
            raise TypeError('"statuses" should be `list` or `TaskStatus` type')
        return statuses

    @staticmethod
    def _celery_task_to_dict(celery_task, kwargs, task_args=None, chord=None, retries=0, group_id=None, taskset_id=None,
                            task_id=None):
        # стараемся воспроизвести логику из celery (celery/app/amqp.TaskProducer.publish_task и вызывающих его методов)
        # не обрабатываем опции
        # utc -- всегда True
        # expires -- всегда None
        # time_limit, soft_time_limit
        # now, eta
        # link, link_error

        utc = True
        task_args = task_args or []
        if not isinstance(task_args, (list, tuple)):
            raise ValueError('task args must be a list or tuple')
        return {
            'chord': chord,
            'retries': retries or 0,
            'args': task_args,
            'callbacks': None,
            'errbacks': None,
            'taskset': group_id or taskset_id,
            'id': task_id or uuid(),
            'utc': utc,
            'task': celery_task.name,
            'expires': None,
            'timelimit': (None, None),
            'eta': None,
            'kwargs': kwargs
        }

    def send_celery_task(self, celery_task, kwargs, master_dc=None, **extra_options):
        """
        отправка celery-такса в виде json через http-интерфейс в queller
        аргументы аналогичны celery_task.apply_async
        """
        kwargs = kwargs.copy()
        set_command_context(kwargs)
        task_dict = self._celery_task_to_dict(celery_task, kwargs, **extra_options)
        payload = to_json(task_dict)
        if (master_dc and
                master_dc in self.base_url_per_dc and
                self.base_url_per_dc[master_dc]):
            try:
                self.request(
                    'POST', '/api/submit-job',
                    base_url=self.base_url_per_dc[master_dc],
                    params={START_EXECUTION_IMMEDIATELY: True},
                    data=payload,
                    headers={'Content-Type': 'application/json'},
                    http_retries=self.http_retries_per_dc,
                )
            except Exception:
                error_log.error("failed to send task via per DC balancer: %s" % traceback.format_exc())
                self.request('POST', '/api/submit-job', data=payload, headers={'Content-Type': 'application/json'})
        else:
            self.request('POST', '/api/submit-job', data=payload, headers={'Content-Type': 'application/json'})
        return task_dict


class QuellerTask(BaseTask):
    """
    Абстрактный класс задач в queller (инструкцию смотри в начале файла)
    """
    """ID задачи. Обязательно переопределяем!"""
    _queller_interface = QuellerService()

    def __init__(self, *args, **kwargs):
        BaseTask.__init__(self, *args, **kwargs)

    @classmethod
    @abstractmethod
    def put(cls):
        """Поставить задачу

        Для каждого таска пишем свой понятный интерфейс метода `put`.
        Надо сформировать словарь данных и вызвать cls._put(data)

        @classmethod
        def put(cls, field_1, field_2):
            data = {'field_1': field_1, 'field_2': field_2}
            return cls._put(data)
        """
        pass

    @classmethod
    def _put(cls, data):
        from mpfs.core.queue import mpfs_queue
        return mpfs_queue.put(data, cls.get_queue_task_name())

    @classmethod
    def get_queue_task_name(cls):
        """Имя задачи в очереди

        :rtype: str
        """
        return cls.__module__ + "." + cls.__name__

    @classmethod
    def queue_len(cls):
        """Получение длины очереди задачи"""
        return cls._queller_interface.get_tasks_count(cls, [TaskStatus.READY])

    @classmethod
    def not_finished_len(cls):
        """Число задач НЕ в терминальном состоянии"""
        return cls._queller_interface.get_tasks_count(cls, TaskStatus.not_finished())


http_queller = QuellerService()
