# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.config import settings
from mpfs.common import errors
from mpfs.core.services.common_service import RequestsPoweredServiceBase

service_log = mpfs.engine.process.get_service_log('rate-limiter')


class RateLimiterService(RequestsPoweredServiceBase):
    """
    Сервис для общения с ОЧЗУ-1

    Документация по ОЧЗУ-1: https://wiki.yandex-team.ru/users/stolen/ratelimiter
    """
    name = 'rate_limiter'
    api_error = errors.RateLimiterBadResponse
    log = service_log
    base_url = None  # should be set in config
    timeout = None  # should be set in config

    @property
    def enabled(self):
        return settings.feature_toggles['rate_limiter']

    @property
    def dry_run(self):
        return settings.feature_toggles['rate_limiter_dry_run']

    def raw_is_limit_exceeded(self, group, counter_key, value):
        """
        Выполняет запрос на проверку превышения лимита по счётчику.

        :param group: Ключ счётчиков.
        :param counter_key: Ключ счётчика в группе.
        :param value: На сколько нужно увеличить счетчик
        :return: Ответ ОЧЗУ-1 в виде tuple([<HTTP status code>, <content>, <headers>]).
        :rtype: tuple
        """
        return self.request('GET', '/'.join((group, counter_key)))

    def is_limit_exceeded(self, group, counter_key, value=1):
        """
        Проверяет превышен ли лимит запросов по счётчику.

        :param group: Ключ счётчиков.
        :param counter_key: Ключ счётчика в группе.
        :param value: На сколько нужно увеличить счетчик
        :return: Превышен ли лимит запросов.
        :rtype: bool
        """
        if not self.enabled:
            return False
        try:
            self.raw_is_limit_exceeded(group, counter_key, value)
        except Exception, e:
            if not self.dry_run and getattr(e, 'code', None) == 429:
                return True
        # в любой непонятной ситуации считаем, что лимит не превышен
        return False

    def check_limit_exceeded(self, group, counter_key, error_class=errors.RequestsLimitExceeded):
        """
        Бросает исключение, если превышен лимит запросов по счётчику.
        error_class сейчас можно выбрать из 2-х вариантов:
            * RequestsLimitExceeded - породит 503
            * RequestsLimitExceeded429 - породит 429

        :param group: Ключ счётчиков.
        :param counter_key: Ключ счётчика в группе.
        :param error_class: Класс бросаемого исключения
        :rtype: NoneType
        :raises error_class:
        """
        if self.is_limit_exceeded(group, counter_key):
            raise error_class()


rate_limiter = RateLimiterService()
