# -*- coding: utf-8 -*-
import time

import mpfs.engine.process

from mpfs.config import settings
from mpfs.core.services.bazinga_service import BazingaInterface
from mpfs.core.services.reindex.link_data_update_worker import BazingaLinkDataUpdateWorker
from mpfs.core.services.reindex.models import QuickMoveReindexTask, QuickMoveReindexTaskStatus, QuickMoveReindexSettings

log = mpfs.engine.process.get_default_log()

BAZINGA_REINDEX_LINK_DATA_UPDATE_MANAGER_ENABLED = settings.bazinga_reindex['link_data_update_manager']['enabled']
BAZINGA_REINDEX_LINK_DATA_UPDATE_MANAGER_WORKER_NUM_LIMIT = \
    settings.bazinga_reindex['link_data_update_manager']['worker_num_limit']


def create_task(uid, status):
    QuickMoveReindexTask(uid=uid, status=QuickMoveReindexTaskStatus(status)).save(force_insert=True)


def set_migration_task_status(uid, status):
    task = QuickMoveReindexTask.controller.get(uid=uid)
    task.status = QuickMoveReindexTaskStatus(status)
    task.save()


class BazingaLinkDataUpdateManager(object):
    """Менеджер запуска тасков для проставления link_data
    """
    ENABLED = BAZINGA_REINDEX_LINK_DATA_UPDATE_MANAGER_ENABLED
    SLEEP_DURATION = 2
    WORK_DURATION = 10 * 60

    def run(self):
        if not self.ENABLED:
            log.info('Disabled, check config')
            return

        end_time = time.time() + self.WORK_DURATION
        while end_time > time.time():
            manager_settings = QuickMoveReindexSettings.controller.get(_id='0')
            if manager_settings is None:
                tasks_limit = BAZINGA_REINDEX_LINK_DATA_UPDATE_MANAGER_WORKER_NUM_LIMIT
            else:
                tasks_limit = manager_settings.link_data_update_tasks_limit

            running_update_tasks_num = QuickMoveReindexTask.controller.filter(
                status={
                    '$in': [QuickMoveReindexTaskStatus.LINK_DATA_UPDATE_STARTED, QuickMoveReindexTaskStatus.ASSIGNED]
                }
            ).count()

            if running_update_tasks_num >= tasks_limit:
                log.info('Too many active tasks %i (limit: %i).', running_update_tasks_num, tasks_limit)
                time.sleep(self.SLEEP_DURATION)
                continue

            ready_tasks = self._get_ready_tasks(tasks_limit - running_update_tasks_num)
            if not ready_tasks:
                log.info('No suitable tasks for link data update')
                time.sleep(self.SLEEP_DURATION)
                continue

            self._run_workers(ready_tasks)
            time.sleep(self.SLEEP_DURATION)

    @staticmethod
    def _get_ready_tasks(max_tasks_count):
        return list(QuickMoveReindexTask.controller
                    .filter(status=QuickMoveReindexTaskStatus.READY)
                    .order_by('created_time')[:max_tasks_count])

    @staticmethod
    def _run_workers(ready_tasks):
        QuickMoveReindexTask.controller.filter(
            uid={'$in': [task.uid for task in ready_tasks]}
        ).update(
            status=QuickMoveReindexTaskStatus.ASSIGNED,
            upsert=False, multi=True
        )

        workers = (BazingaLinkDataUpdateWorker(t.uid) for t in ready_tasks)
        spawned_workers_num = len(BazingaInterface().bulk_create_tasks(workers))
        log.info('Spawned %i workers', spawned_workers_num)
