# -*- coding: utf-8 -*-
import datetime

import mpfs.engine.process
from mpfs.common.errors import NoShardForUidRoutingError, StorageInitUser, ResourceNotFound

from mpfs.config import settings
from mpfs.core import factory
from mpfs.core.address import Address
from mpfs.core.metastorage.control import link_data
from mpfs.core.services.bazinga_service import OnetimeTask
from mpfs.core.services.reindex.models import QuickMoveReindexTask, QuickMoveReindexTaskStatus

BAZINGA_REINDEX_LINK_DATA_UPDATE_WORKER_ENABLED = \
    settings.bazinga_reindex['link_data_update_worker']['enabled']


class BazingaLinkDataUpdateWorker(OnetimeTask):
    log = mpfs.engine.process.get_default_log()
    error_log = mpfs.engine.process.get_error_log()

    ENABLED = BAZINGA_REINDEX_LINK_DATA_UPDATE_WORKER_ENABLED
    BAZINGA_TASK_NAME = 'QuickMoveLinkDataUpdateWorker'

    def __init__(self, uid):
        super(BazingaLinkDataUpdateWorker, self).__init__()
        if not isinstance(uid, basestring):
            raise TypeError('BazingaLinkDataUpdateWorker parameter type is incorrect: %s' % str(uid))
        self.uid = uid

    def build_command_parameters(self):
        return [self.uid]

    def run(self):
        if not self.ENABLED:
            self.log.info('Disabled. Check config')
            return

        if not self.uid:
            self.log.info('No user id')
            return

        task = QuickMoveReindexTask.controller.get(uid=self.uid)
        if task.status != QuickMoveReindexTaskStatus.ASSIGNED:
            self.log.info('Task `%s` status is not ASSIGNED (`%s`)' % (self.uid, task.status))
            return

        try:
            mpfs.engine.process.usrctl().assert_user_init(self.uid)
        except (NoShardForUidRoutingError, StorageInitUser):
            self.log.info('Uid `%s` is not initialized' % self.uid)
            task.status = QuickMoveReindexTaskStatus.NOT_INITIALIZED
            task.link_data_started_time = datetime.datetime.now()
            task.link_data_finished_time = datetime.datetime.now()
            task.save()
            return

        task.status = QuickMoveReindexTaskStatus.LINK_DATA_UPDATE_STARTED
        task.link_data_started_time = datetime.datetime.now()
        task.ycrid = mpfs.engine.process.get_cloud_req_id()
        task.save()

        try:
            total_links, resource_not_found, updated_count = self._update_link_data()
            task.link_data_total_count = total_links
            task.link_data_not_found_count = resource_not_found
            task.link_data_updated_count = updated_count
            task.status = QuickMoveReindexTaskStatus.LINK_DATA_UPDATE_FINISHED
        except Exception as e:
            self.error_log.info('Failed with exception %s' % e)
            task.status = QuickMoveReindexTaskStatus.LINK_DATA_UPDATE_FAILED
        finally:
            task.link_data_finished_time = datetime.datetime.now()
            task.save()

    def _update_link_data(self):
        updated_count = 0
        resource_not_found = 0
        total_links = 0

        for link in list(link_data.find_all({'uid': self.uid})):
            data = link['data']
            uid = link['uid']
            if not data:
                continue

            total_links += 1

            if 'dtime' in data:
                continue  # skip deleted links

            address = Address(data['tgt'])

            try:
                resource = factory.get_resource(uid, address)
            except ResourceNotFound:
                resource_not_found += 1
                continue  # skip not found resources - maybe set dtime for these links?

            if address.uid != resource.resource_id.uid:
                raise ValueError('Uid is different in data: %s' % str(link))

            file_id = resource.meta['file_id']

            link['data']['file_id'] = file_id
            link['data']['resource_id_uid'] = resource.resource_id.uid

            link_data.update({'uid': uid, '_id': link['_id']}, link)
            updated_count += 1

        return total_links, resource_not_found, updated_count
