# -*- coding: utf-8 -*-
import datetime
import enum

from mpfs.core.models import Model, ModelField, EnumModelField


class QuickMoveReindexTaskStatus(enum.Enum):
    CREATED = 'CREATED'  # "безопасный" статус, означающий, что таск создан, но не готов к тому, чтобы быть обработанным
    READY = 'READY'  # менеджер простановки link_data ищет таски с таким статусом и распределяет их по воркерам
    ASSIGNED = 'ASSIGNED'  # статус, означающий, что менеджер обработал этот таск и отдал воркеру

    LINK_DATA_UPDATE_STARTED = 'LINK_DATA_UPDATE_STARTED'  # воркер обновления link_data начал обрабатывать таск
    LINK_DATA_UPDATE_FINISHED = 'LINK_DATA_UPDATE_FINISHED'  # воркер обновления link_data закончил обрабатывать таск
    LINK_DATA_UPDATE_FAILED = 'LINK_DATA_UPDATE_FAILED'  # воркер обновления link_data упал с ошибкой

    REINDEX_ASSIGNED = 'REINDEX_ASSIGNED'  # статус, означающий, что менеджер обработал этот таск и отдал воркеру
    REINDEX_STARTED = 'REINDEX_STARTED'  # переиндексация в поиске запущена
    REINDEX_FINISHED = 'REINDEX_FINISHED'  # переиндексация в поиске завершена, у пользователя включен быстрый мув
    REINDEX_FAILED = 'REINDEX_FAILED'  # переиндексация упала с ошибкой

    NOT_INITIALIZED = 'NOT_INITIALIZED'  # начали обрабатывать таск, но оказалось, что пользователь не инициализирован
    MONGO_USER = 'MONGO_USER'  # начали обрабатывать таск, но оказалось, что пользователь живет в монге
    DEPTH_LIMIT_EXCEEDED = 'DEPTH_LIMIT_EXCEEDED'  # вложенность папок у пользователя больше лимита (может сломать DJFS)


class QuickMoveReindexSettings(Model):
    enabled = ModelField(required=True)
    link_data_update_tasks_limit = ModelField(required=True)  # максимальное число одновременно бегущих тасок на
    # обновление link_data
    running_reindex_tasks_limit = ModelField(required=True)  # максимальное число одновременно запущенных переиндексаций
    # в поиске


class QuickMoveReindexTask(Model):
    primary_key_field = 'uid'

    uid = ModelField(required=True, source='_id')
    created_time = ModelField(required=True)  # время создания таска
    status = EnumModelField(QuickMoveReindexTaskStatus, required=True)  # статус таска

    link_data_started_time = ModelField()  # время начала заполнения link_data
    link_data_finished_time = ModelField()  # время конца заполнения link_data (или падения)
    link_data_total_count = ModelField()  # количество записей в link_data (всего)
    link_data_not_found_count = ModelField()  # количество записей в link_data, для которых не нашли файл по пути
    link_data_updated_count = ModelField()  # количество записей в link_data, для которых проставили uid/file_id

    reindex_started_time = ModelField()  # время отправки запроса в поиск для переиндексации (сразу после запроса)
    reindex_finished_time = ModelField()  # время конца переиндексации (когда пришел колбек)

    ycrid = ModelField()  # ycrid таска миграции, чтобы можно было грепать логи по нему

    def __init__(self, **kwargs):
        super(QuickMoveReindexTask, self).__init__(**kwargs)
        if not isinstance(kwargs.get('created_time'), datetime.datetime):
            self.created_time = datetime.datetime.now()
        if not isinstance(kwargs.get('status'), QuickMoveReindexTaskStatus):
            self.status = QuickMoveReindexTaskStatus.CREATED
