# -*- coding: utf-8 -*-
import datetime
import time

import mpfs.engine.process

from mpfs.config import settings
from mpfs.core.services.bazinga_service import BazingaInterface
from mpfs.core.services.reindex.models import QuickMoveReindexSettings, QuickMoveReindexTask, QuickMoveReindexTaskStatus
from mpfs.core.services.reindex.reindex_worker import BazingaReindexLaunchWorker
from mpfs.metastorage.mongo.collections.filesystem import is_reindexed_for_quick_move

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

BAZINGA_REINDEX_REINDEX_MANAGER_MANAGER_ENABLED = settings.bazinga_reindex['reindex_manager']['enabled']
BAZINGA_REINDEX_REINDEX_MANAGER_USERS_RUNNING_LIMIT = \
    settings.bazinga_reindex['reindex_manager']['users_running_limit']


class BazingaQuickMoveReindexManager(object):
    """Менеджер запуска тасков для проставления link_data
    """
    ENABLED = BAZINGA_REINDEX_REINDEX_MANAGER_MANAGER_ENABLED
    SLEEP_DURATION = 2
    WORK_DURATION = 10 * 60

    def run(self):
        if not self.ENABLED:
            log.info('Disabled, check config')
            return

        end_time = time.time() + self.WORK_DURATION
        while end_time > time.time():
            manager_settings = QuickMoveReindexSettings.controller.get(_id='0')
            if manager_settings is None:
                tasks_limit = BAZINGA_REINDEX_REINDEX_MANAGER_USERS_RUNNING_LIMIT
            else:
                tasks_limit = manager_settings.running_reindex_tasks_limit

            running_reindex_tasks_num = QuickMoveReindexTask.controller.filter(
                status={
                    '$in': [QuickMoveReindexTaskStatus.REINDEX_STARTED, QuickMoveReindexTaskStatus.REINDEX_ASSIGNED]
                }
            ).count()

            if running_reindex_tasks_num >= tasks_limit:
                log.info('Too many active tasks %i (limit: %i).', running_reindex_tasks_num, tasks_limit)
                time.sleep(self.SLEEP_DURATION)
                continue

            ready_tasks = self._get_ready_tasks(tasks_limit - running_reindex_tasks_num)
            if not ready_tasks:
                log.info('No suitable tasks for reindex')
                time.sleep(self.SLEEP_DURATION)
                continue

            self._run_workers(ready_tasks)
            time.sleep(self.SLEEP_DURATION)

    @staticmethod
    def _get_ready_tasks(max_tasks_count):
        return list(QuickMoveReindexTask.controller
                    .filter(status=QuickMoveReindexTaskStatus.LINK_DATA_UPDATE_FINISHED)
                    .order_by('created_time')[:max_tasks_count])

    @staticmethod
    def _run_workers(ready_tasks):
        already_reindexed_uids = []
        mongo_uids = []
        uids_to_reindex = []

        for task in ready_tasks:
            if is_reindexed_for_quick_move(task.uid):
                already_reindexed_uids.append(task.uid)
            elif not mpfs.engine.process.usrctl().is_user_in_postgres(task.uid):
                mongo_uids.append(task.uid)
            else:
                uids_to_reindex.append(task.uid)

        uid_statuses_to_set = (
            (already_reindexed_uids, QuickMoveReindexTaskStatus.REINDEX_FINISHED),
            (mongo_uids, QuickMoveReindexTaskStatus.MONGO_USER),
            (uids_to_reindex, QuickMoveReindexTaskStatus.REINDEX_ASSIGNED),
        )
        for uids, status in uid_statuses_to_set:
            if not uids:
                continue

            QuickMoveReindexTask.controller.filter(
                uid={'$in': [uid for uid in uids]}
            ).update(
                status=status,
                upsert=False, multi=True
            )

        workers = (BazingaReindexLaunchWorker(uid) for uid in uids_to_reindex)
        spawned_workers_num = len(BazingaInterface().bulk_create_tasks(workers))
        log.info('Reindex started for %i users', spawned_workers_num)
