# -*- coding: utf-8 -*-
import datetime

import mpfs.engine.process

from mpfs.config import settings
from mpfs.core.services.bazinga_service import OnetimeTask
from mpfs.core.services.index_service import SearchIndexer
from mpfs.core.services.reindex.models import QuickMoveReindexTask, QuickMoveReindexTaskStatus
from mpfs.core.services.reindex.utils import calculate_max_folder_depth
from mpfs.metastorage.mongo.collections.filesystem import is_reindexed_for_quick_move

BAZINGA_REINDEX_REINDEX_WORKER_ENABLED = settings.bazinga_reindex['reindex_worker']['enabled']
BAZINGA_REINDEX_REINDEX_WORKER_MAX_FOLDER_DEPTH_LIMIT = settings.bazinga_reindex['reindex_worker']['max_folder_depth_limit']


class BazingaReindexLaunchWorker(OnetimeTask):
    log = mpfs.engine.process.get_default_log()
    error_log = mpfs.engine.process.get_error_log()

    ENABLED = BAZINGA_REINDEX_REINDEX_WORKER_ENABLED
    BAZINGA_TASK_NAME = 'QuickMoveReindexWorker'

    def __init__(self, uid):
        super(BazingaReindexLaunchWorker, self).__init__()
        if not isinstance(uid, basestring):
            raise TypeError('BazingaReindexLaunchWorker parameter type is incorrect: %s' % str(uid))
        self.uid = uid

    def build_command_parameters(self):
        return [self.uid]

    def run(self):
        if not self.ENABLED:
            self.log.info('Disabled. Check config')
            return

        if not self.uid:
            self.log.info('No user id')
            return

        task = QuickMoveReindexTask.controller.get(uid=self.uid)
        if task.status != QuickMoveReindexTaskStatus.REINDEX_ASSIGNED:
            self.log.info('Task `%s` status is not REINDEX_ASSIGNED (`%s`)' % (self.uid, task.status))
            return

        if is_reindexed_for_quick_move(task.uid):
            task.status = QuickMoveReindexTaskStatus.REINDEX_FINISHED
            task.save()
            return
        if not mpfs.engine.process.usrctl().is_user_in_postgres(task.uid):
            task.status = QuickMoveReindexTaskStatus.MONGO_USER
            task.save()
            return

        task.status = QuickMoveReindexTaskStatus.REINDEX_STARTED
        task.reindex_started_time = datetime.datetime.now()
        task.ycrid = mpfs.engine.process.get_cloud_req_id()
        task.save()

        try:
            max_folder_depth = calculate_max_folder_depth(task.uid)
            if max_folder_depth > BAZINGA_REINDEX_REINDEX_WORKER_MAX_FOLDER_DEPTH_LIMIT:
                task.status = QuickMoveReindexTaskStatus.DEPTH_LIMIT_EXCEEDED
                task.reindex_finished_time = datetime.datetime.now()
                task.save()
                return

            SearchIndexer().start_reindex_for_quick_move(task.uid)
        except Exception:
            self.error_log.exception('Reindex start for uid `%s`', task.uid)
            task.status = QuickMoveReindexTaskStatus.REINDEX_FAILED
            task.reindex_finished_time = datetime.datetime.now()
            task.save()
            raise
