# -*- coding: utf-8 -*-
"""

MPFS
CORE

Сервис Социальной Прокси

http://wiki.yandex-team.ru/social/proxy

"""
import cjson
import sys
import urllib
import urlparse

import mpfs.engine.process
import mpfs.common.errors as errors

from mpfs.engine.http.client import open_url
from mpfs.core.services.common_service import Service

ALBUM_PARAM = 'album'
PROFILE_PARAM = 'profile'
DEBUG_PARAM = 'debug_photos_pagination'
NEXT_TOKEN_PARAM = 'next_token'

error_log = mpfs.engine.process.get_error_log()
service_log = mpfs.engine.process.get_service_log('socialproxy')


class SocialProxy(Service):

    VK_PROVIDER = 'vkontakte'
    FACEBOOK_PROVIDER = 'facebook'
    ODNOKLASSNIKI_PROVIDER = 'odnoklassniki'
    MAILRU_PROVIDER = 'mailru'
    GOOGLE_PROVIDER = 'google'
    INSTAGRAM_PROVIDER = 'instagram'
    TWITTER_PROVIDER = 'twitter'

    name = 'socialproxy'
    api_error = errors.SocialProxyNoResponse
    log = service_log    

    command_map = {
        'get_profiles': '/api/user/%(uid)s/profile',
        'user_rights': '/api/token/newest?profile_id=%(profile_id)s&application_id=%(application_id)s&uid=%(uid)s',
        'get_albums': '/proxy2/profile/%(profile)s/photo_albums?locale=%(locale)s',
        'create_album': '/proxy2/profile/%(profile)s/photo_albums/create',
        'album_photos': '/proxy2/profile/%(profile)s/photos?aid=%(album)s',
        'user_photos': '/proxy2/profile/%(profile)s/user_photos',
        'get_friends': '/proxy2/profile/%(profile)s/friends',
        'check_message': '/proxy2/profile/%(profile)s/messages/post',
        'wall_post': '/proxy2/profile/%(profile)s/wall/post',
    }
    
    errors_map = {
        'access-denied': errors.SocialProxyAccessDenied,
    }

    # см. http://wiki.yandex-team.ru/social/providers
    application_ids = {
        FACEBOOK_PROVIDER: 20,
        VK_PROVIDER: 10,
        ODNOKLASSNIKI_PROVIDER: 60,
        TWITTER_PROVIDER: 30,
        MAILRU_PROVIDER: 40,
        GOOGLE_PROVIDER: 52,
        INSTAGRAM_PROVIDER: 90,
    }

    # см. https://wiki.yandex-team.ru/social/proxy/v2#poluchitspisokfotografijjizalbomapolzovatelja
    scopes_for_photos_import = {
        VK_PROVIDER: ['photos'],
        FACEBOOK_PROVIDER: ['user_photos'],
        ODNOKLASSNIKI_PROVIDER: [],
        MAILRU_PROVIDER: [],
        GOOGLE_PROVIDER: ['https://picasaweb.google.com/data/'],
        INSTAGRAM_PROVIDER: [],
    }

    # см. https://wiki.yandex-team.ru/social/proxy/v2#zagruzkafotokvsocialku
    scopes_for_photos_export = {
        VK_PROVIDER: ['photos'],
        FACEBOOK_PROVIDER: ['user_photos', 'publish_actions'],
        ODNOKLASSNIKI_PROVIDER: ['PHOTO_CONTENT'],
        MAILRU_PROVIDER: ['photos'],
    }

    # https://jira.yandex-team.ru/browse/CHEMODAN-8045
    scopes_for_invites = {
        FACEBOOK_PROVIDER: ['user_birthday', 'email', 'xmpp_login'],
    }

    scopes_for_wall_post = {
        FACEBOOK_PROVIDER: ['publish_actions'],
        VK_PROVIDER: ['wall'],
        ODNOKLASSNIKI_PROVIDER: ['PUBLISH_TO_STREAM'],
        TWITTER_PROVIDER: ['write'],
        MAILRU_PROVIDER: None,  # unsupported for this provider
        GOOGLE_PROVIDER: None,  # unsupported for this provider
        INSTAGRAM_PROVIDER: None,  # unsupported for this provider
    }

    scenario_to_scopes_mapping = {
        'invite': scopes_for_invites,
        'wall_post': scopes_for_wall_post,
        'photos_import': scopes_for_photos_import,
        'photos_export': scopes_for_photos_export,
    }

    RATE_LIMIT_EXCEEDED_ERROR = 'rate_limit_exceeded'
    INVALID_PARAMETERS_ERROR = 'invalid_parameters'
    INVALID_TOKEN_ERRORS = ['invalid_token', 'permission_error', 'no_tokens_found']
    ALBUM_NOT_EXISTS_ERROR = 'album_not_exists'

    def get_wall_post_valid_providers(self):
        return [provider_id for provider_id, scopes in self.scopes_for_wall_post.iteritems() if scopes is not None]

    def process(self, data, post=False):
        action_template = self.command_map[sys._getframe(1).f_code.co_name]
        action = action_template % data
        for key in data.keys():
            if key in action_template:
                del(data[key])
        if len(data) == 0 and post:
            data['post_param'] = 1
        return self.process_action(action, data)

    def process_action(self, action, data={}, timeout=None):
        try:
            response = self.open_url(self.base_url + action, data, timeout=timeout)
            result = cjson.decode(response)
        except Exception, e:
            error_log.warn(e)
            raise self.api_error()
        if 'error' in result:
            error = result.get('error')
            message = error.get('description')
            err_num = error.get('name')
            err_class = self.errors_map.get(err_num, errors.PassportBadResult)
            raise err_class(message)
        return result

    def get_profiles(self, uid):
        '''
        Принимаем от прокси: 
        {
          "profiles": [
            ...
            {
              "username": "deadlydrunker", 
              "uid": 89031628, 
              "profile_id": 108390, 
              "userid": "206078155", 
              "allow_auth": false, 
              "provider": "twitter", 
              "addresses": [
                "http://twitter.com/deadlydrunker/"
              ]
            }
            ...
          ]
        }
        
        Отдаем:
        {
            ...
            'twitter': 108390,
            ...
        }
        '''
        result = {}
        for item in self.process({'uid': uid}).get('profiles'):
            result[item['provider']] = item['profile_id']
        return result
    
    def ask_friends(self, uid):
        """
            Имитация асинхронной работы.
        :param uid:
        :return:
        """
        profiles = self.get_profiles(uid)
        return profiles

    def get_friends(self, profile):
        '''
        Получаем результат задачи получения списка друзей
        
        Принимаем:
        {
            "task": {
              "start": 1343052798.8432441, 
              "state": "success", 
              "runtime": 0.88603900000000002, 
              "end": 1343052799.7292831, 
              "profile_id": 108512, 
              "id": "49820e5b-5f91-4975-b5d4-51c74ed66135"
            }, 
            "result": {
              "friends": [
                {
                  "person": {
                    "locale": "ru_RU", 
                    "gender": "m", 
                    "name": {
                      "last": "Veselyh", 
                      "first": "Zanzibar"
                    }, 
                    "birthdate": "1980-09-01", 
                    "avatar": {
                      "50x50": "http://profile.ak.fbcdn.net/hprofile-ak-snc4/370586_100004068433137_1781647964_q.jpg", 
                      "200x600": "http://profile.ak.fbcdn.net/hprofile-ak-snc4/370586_100004068433137_1781647964_n.jpg", 
                      "100x300": "http://profile.ak.fbcdn.net/hprofile-ak-snc4/370586_100004068433137_1781647964_s.jpg", 
                      "50x150": "http://profile.ak.fbcdn.net/hprofile-ak-snc4/370586_100004068433137_1781647964_t.jpg"
                    }
                  }, 
                  "updated": 1342701577, 
                  "userid": 100004068433137
                }
              ]
            }
        }
        
        Отдаем:
        {
            "state": "success", 
            "result": [
                {
                  "person": {
                    "locale": "ru_RU", 
                    "gender": "m", 
                    "name": {
                      "last": "Veselyh", 
                      "first": "Zanzibar"
                    }, 
                    "birthdate": "1980-09-01", 
                    "avatar": {
                      "50x50": "http://profile.ak.fbcdn.net/hprofile-ak-snc4/370586_100004068433137_1781647964_q.jpg", 
                      "200x600": "http://profile.ak.fbcdn.net/hprofile-ak-snc4/370586_100004068433137_1781647964_n.jpg", 
                      "100x300": "http://profile.ak.fbcdn.net/hprofile-ak-snc4/370586_100004068433137_1781647964_s.jpg", 
                      "50x150": "http://profile.ak.fbcdn.net/hprofile-ak-snc4/370586_100004068433137_1781647964_t.jpg"
                    }
                  }, 
                  "updated": 1342701577, 
                  "userid": 100004068433137
                }
            ]
        }
        '''
        response = self.process({'profile': profile})
        state = response.get('task').get('state')
        
        if state == 'success':
            result = response.get('result') or {}
            data   = result.get('friends', [])
            return {
                'state'  : state,
                'result' : self._format_friends(data)
            }
        elif state == 'failure':
            return self._format_failure(response.get('task'))
        else:
            return {
                'state'  : state,
                'result' : [],
            }
            
    def _format_failure(self, task):
        reason = task.get('reason')
        description = reason.get('description', 'internal').lower()
        code = reason.get('code', '')
        
        if ('access' in description and 'token' in description) or \
           ('permission' in description and 'denied' in description) or \
           'No tokens found' in description or\
           'invalid_token' in code:
            reason = 'token'
        else:
            reason = description
            
        return {
            'state'  : 'failure',
            'reason' : reason,
            'result' : []
        }
            
    def _format_friends(self, friends):
        result = []
        
        for item in friends:
            userid = item.get('userid')
            
            if 'person' in item:
                person = item.get('person')
            else:
                person = {}
               
            #=== locale ========= 
                
            if 'locale' in person:
                # facebook
                locale = person.get('locale').split('_').pop(0)
            elif 'location' in person:
                # vk
                locale = person.get('location', {}).get('country', {}).get('code', '').lower()
            else:
                locale = 'ru'
            
            #=== username =======
            
            if 'name' in person:
                name = person.get('name')
                username = name.get('first', '') + u' ' + name.get('last', '')
            else:
                username = ''
            
            #=== avatar =========
            
            if 'avatar' in person:
                avatar = person.get('avatar', {}).get('50x50')
            else:
                avatar = ''
            
            user = {
                'userid': userid,
                'locale': locale,
                'name'  : username,
                'avatar': avatar,
            }
            
            result.append(user)
            
        return result

    def get_profile_for_provider(self, uid, provider):
        '''
        Возвращаем profile_id пользователя uid для соц. сети provider
        '''
        try:
            return self.get_profiles(uid)[provider]
        except KeyError:
            raise errors.SocialProxyNoProfiles(uid)

    def get_albums(self, uid, provider, locale):
        '''
        Получаем список альбомов пользователя
        
        Принимаем:
        {
          "result": [
            {
            {
              "aid": 94595474,
              "created": "1248115920",
              "description": "",
              "photo_count": 22,
              "title": "party'09",
              "visibility": "public"
            },
          ],
          "task": {
            "profile_id": 100568,
            "provider": "vk",
            "runtime": 0.1731719970703125,
            "status": "success"
          }
        }

        Отдаем:
        {
            "state": "success", 
            "result": [
                {
                  "aid": 94595474,
                  "title" : "party'09",
                  "visibility" : "public"
                }
            ]
        }
        '''
        profile = self.get_profile_for_provider(uid, provider)

        if locale == 'ua':  # ua is invalid locale for Ukraine
            locale = 'uk'

        response = self.process({
            'profile': profile,
            'locale': locale
        })
        task = response.get('task')
        state = task.get('state')

        if state == 'success':
            result = response.get('result') or {}
            return {
                'state'  : state,
                'result' : self._format_albums(result)
            }
        elif state == 'failure' and task['reason']['code'] == SocialProxy.RATE_LIMIT_EXCEEDED_ERROR:
            self._log_rate_limit_exceeded(uid, provider)
            raise errors.SocialProxyRateLimitExceeded()
        elif state == 'failure' and task['reason']['code'] in SocialProxy.INVALID_TOKEN_ERRORS:
            raise errors.SocialProxyInvalidToken()
        elif state == 'failure':
            return self._format_failure(task)
        else:
            return {
                'state'  : state,
                'result' : [],
            }

    def _format_albums(self, albums):
        result = []
        
        for item in albums:
            aid = item.get('aid')
            
            if 'title' in item:
                title = item.get('title')
            else:
                title = ''

            if 'visibility' in item:
                visibility = item.get('visibility')
            else:
                visibility = ''

            result.append({'aid' : aid, 'title' : title, 'visibility' : visibility});

        return result

    def wall_post(self, uid, provider, link):
        """
        Постинг на стену

        https://wiki.yandex-team.ru/social/proxy/v2#dobavitpostingnastenupolzovatelja
        """
        profile = self.get_profile_for_provider(uid, provider)
        response = self.process({'profile': profile, 'link': link, 'text': ' '}, post=True)
        task = response.get('task')
        state = task.get('state')
        if state == 'success':
            result = response.get('result') or {}
            return {
                'state'  : state,
                'result' : result
            }
        elif state == 'failure' and task['reason']['code'] == SocialProxy.RATE_LIMIT_EXCEEDED_ERROR:
            self._log_rate_limit_exceeded(uid, provider)
            raise errors.SocialProxyRateLimitExceeded()
        elif state == 'failure':
            return self._format_failure(task)
        return {
            'state'  : state,
            'result' : {},
        }

    def create_album(self, uid, provider, title, privacy):
        '''
        Создаем альбом пользователя
        
        Принимаем:
        {
          "result": {
            "aid": "435294923283275"
           },
          "task": {
            "profile_id": 108726,
            "provider": "facebook",
            "runtime": 0.55472803115844727,
            "state": "success"
        }
        
        Отдаем:
        {
            "state": "success", 
            "result": {"aid": 94595474}
        }
        '''
        profile = self.get_profile_for_provider(uid, provider)
        response = self.process({'profile': profile, 'title': title, 'privacy': privacy})
        task = response.get('task')
        state = task.get('state')

        if state == 'success':
            result = response.get('result') or {}
            return {
                'state'  : state,
                'result' : result
            }
        elif state == 'failure' and task['reason']['code'] == SocialProxy.RATE_LIMIT_EXCEEDED_ERROR:
            self._log_rate_limit_exceeded(uid, provider)
            raise errors.SocialProxyRateLimitExceeded()
        elif state == 'failure':
            return self._format_failure(task)
        else:
            return {
                'state'  : state,
                'result' : [],
            }


    def send_message(self, uid, provider, ids, message):
        """
            Имитация асинхронной работы.
        :param uid:
        :param provider:
        :param ids:
        :param message:
        :return:
        """
        profile = self.get_profile_for_provider(uid, provider)
        return {'profile': profile, 'ids': ids, 'text': message}

    def check_message(self, data):
        '''
        Получаем результат задачи отправки сообщения
        
        Принимаем:
        {
            "task": {
              "start": 1343120842.2003829, 
              "state": "success", 
              "runtime": 0.238263, 
              "end": 1343120842.4386461, 
              "profile_id": 108530, 
              "id": "a0ed6481-379e-49e3-85c8-3c556b82cc21"
            }, 
            "result": {
              "messages": {
                "fail": [], 
                "success": [
                  "1980253"
                ]
              }
            }
        }
        
        Отдаем:
        {
            'state': 'success',
            'messages': {
                "fail": [], 
                "success": [
                    "1980253"
                ]
            }
        }
        '''
        task = self.process(data)
        state = task.get('task').get('state')
        
        if state == 'success':
            return {
                'state'  : state,
                'result' : task.get('result').get('messages')
            }
        elif state == 'failure':
            return self._format_failure(task.get('task'))
        else:
            return {
                'state'  : state,
                'result' : {},
            }
            
    def user_rights(self, uid, scenario):
        '''
        Проверяем наличие у пользователя токенов на указанный сценарий.

        Выдаем абсолютные ссылки на которые можно сходить
        под авторизацией и получить токены.

        Принимаем от прокси наборы значений вида:
        {
          "token": {
            "application": "facebook",
            "secret": "",
            "verified": "2012-07-25 14:20:49",
            "scope": "user_birthday email offline_access xmpp_login",
            "token_id": 169190,
            "profile_id": 108532,
            "confirmed": "2012-07-25 14:20:49",
            "value": "AAADEZBFep6zkBAI2WUO0f0UcmZAI7gEIZB5R2VQmd4GZAZBEiJXxZCj3oyn3hUAVtbBKi3OJnLkTRWVO3XMgP6BPMV8QCjysoAjDIcfj3RBwZDZD",
            "created": "2012-07-25 14:20:49"
          }
        }

        Отдаем в случае когда все права есть:
        {
            'vkontakte': {
                'auth': True,
                'link': '',
            },
            'facebook': {
                'auth: True,
                'link': '',
            }
        }

        Отдаем в случае когда каких-то прав не хватает:
        {
            'vkontakte': {
                'auth': False,
                'link': 'https://social-testrc.yandex.ru/broker/start?consumer=disk&application=vkontakte&scope=friends,messages&retpath=http://disk.yandex.ru',
            },
            'facebook': {
                'auth: True,
                'link': '',
            }
        }
        '''
        result = {}
        need_scopes = self.scenario_to_scopes_mapping[scenario]

        authorized_provider = {
            'auth': True,
            'link': '',
        }

        for provider, profile in self.get_profiles(uid).iteritems():
            if provider not in need_scopes:
                result[provider] = authorized_provider
                continue

            application_id = self.application_ids[provider]

            try:
                token_data = self.process({
                    'uid': uid,
                    'application_id': application_id,
                    'profile_id': profile,
                    })
                existing_scopes = token_data.get('token').get('scope').split()
            except Exception:
                existing_scopes = []

            provider_scopes = need_scopes[provider]
            if provider_scopes is None:  # unsupported for this provider
                continue
            elif not provider_scopes:
                result[provider] = authorized_provider
            else:
                for scope in provider_scopes:
                    if scope not in existing_scopes:
                        link = self.token_url % {
                            'provider': provider,
                            'scopes'  : ','.join(provider_scopes),
                        }
                        result[provider] = {
                            'auth': False,
                            'link': link,
                        }
                        break
                    else:
                        result[provider] = authorized_provider
        return result

    def album_photos(self, uid, provider, album_id=None, debug=False):
        '''
        Получаем список фото пользователя в указанном альбоме.
        Принимаем:
        {
          "result": [
            {
              "caption": "\u0442\u043e\u043b\u044c\u043a\u043e \u043f\u0440\u0438\u043b\u0435\u0442\u0435\u043b\u0438, \u0440\u0430\u0441\u0441\u0430\u0436\u0438\u0432\u0430\u0435\u043c\u0441\u044f",
              "created": 1346504298,
              "images": [
                {
                  "height": 56,
                  "url": "http://cs315627.vk.me/v315627352/cad/z1XuZVj6SaA.jpg",
                  "width": 75
                },
                ...,
                {
                  "height": 960,
                  "url": "http://cs315627.vk.me/v315627352/cb1/xLYpRYrJ43s.jpg",
                  "width": 1280
                }
              ],
              "location": {
                "latitude": 42.431887000000003,
                "longitude": 18.711272999999998
              },
              "pid": 289068348
            },
            ...
          ],
          "task": {
            "profile_id": 100563,
            "provider": "vk",
            "runtime": 0.23988699913024902,
            "state": "success"
          }

          Отдаем:
          [{
            "pid": 289068348,
            "location": {
                "latitude": 42.431887000000003,
                "longitude": 18.711272999999998
            },
            "created": 1346504298,
            "url": "http://cs315627.vk.me/v315627352/cb1/xLYpRYrJ43s.jpg"
          },...]
        '''
        profile_id = self.get_profile_for_provider(uid, provider)
        request_params = {
            PROFILE_PARAM: profile_id,
            ALBUM_PARAM: album_id,
            }
        relative_url = self.command_map['album_photos'] % request_params
        return self._request_photos_with_pagination(relative_url, uid, provider, debug)

    def user_photos(self, uid, provider):
        '''
        Принимаем список фото, на которых отмеченных пользователь (формат см. album_photos)
        '''
        profile_id = self.get_profile_for_provider(uid, provider)
        request_params = {
            PROFILE_PARAM: profile_id
        }
        relative_url = self.command_map['user_photos'] % request_params
        return self._request_photos_with_pagination(relative_url, uid, provider, False)

    def _request_photos_with_pagination(self, relative_url, uid, provider, debug):
        result = []
        next_token = None
        while True:
            params = {}
            if debug:
                params[DEBUG_PARAM] = 1
            if next_token:
                params[NEXT_TOKEN_PARAM] = next_token
            result_url = self._format_url(relative_url, params)
            response = self.process_action(action=result_url, timeout=self.photos_timeout)
            photos = self._process_photos_result(uid, provider, response)
            result.extend(photos)

            next_token = response.get(NEXT_TOKEN_PARAM)
            if not next_token:
                break
        return result

    def _format_url(self, relative_url, params):
        params_str = urllib.urlencode(params)
        if urlparse.urlparse(relative_url)[4]:
            return relative_url + '&' + params_str
        else:
            return relative_url + '?' + params_str

    def _process_photos_result(self, uid, provider, result):
        task = result.get('task')
        state = task.get('state')
        if state == 'success':
            return self._format_photos(result['result'])
        elif state == 'failure' and task['reason']['code'] == SocialProxy.ALBUM_NOT_EXISTS_ERROR:
            return []
        elif state == 'failure' and task['reason']['code'] == SocialProxy.RATE_LIMIT_EXCEEDED_ERROR:
            self._log_rate_limit_exceeded(uid, provider)
            raise errors.SocialProxyRateLimitExceeded()
        elif state == 'failure' and task['reason']['code'] == SocialProxy.INVALID_PARAMETERS_ERROR:
            raise errors.SocialProxyInvalidParameters()
        elif state == 'failure' and task['reason']['code'] in SocialProxy.INVALID_TOKEN_ERRORS:
            raise errors.SocialProxyInvalidToken()
        else:
            error_log.warn('Bad response from %s: %s' % (provider, str(result)))
            raise errors.SocialProxyBadResult()

    def _log_rate_limit_exceeded(self, uid, provider):
        error_log.warn('Rate limit exceeded for %s in %s' % (uid, provider))

    def _format_photos(self, photos):
        result = []

        for photo in photos:
            data = photo.get('videos') # instagram support video
            is_video = True
            if not data:
                data = photo.get('images')
                is_video = False
            if not data:  # skip photo if no size was given
                continue

            formatted = {
                'pid': photo['pid'],
                'is_video': is_video,
            }

            created = photo.get('created')
            if created:
                formatted['created'] = created

            location = photo.get('location')
            if location:
                formatted['location'] = location
            # Соц. прокся отдает ссылки в отсортированном по размеру виде,
            # последнее фото - самое большое
            max_size = data[-1]
            formatted['url'] = max_size['url']

            # Для карусельных постов соц.прокся отдает порядковый номер фото в карусели.
            order_no = photo.get('order_no')
            if order_no is not None:
                formatted['order_no'] = order_no

            result.append(formatted)

        return result
